/*
 * WallFire -- a comprehensive firewall administration tool.
 * 
 * Copyright (C) 2001 Herv Eychenne <rv@wallfire.org>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 */

using namespace std;

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string>
#include <sstream>

#include <stdlib.h>
#if defined(__FreeBSD__)
#include <sys/param.h> /* for htons() */
#else
#include <netinet/in.h> /* for htons() */
#endif
#include <netdb.h> /* for getservbyport */

#include "wfport_range.h"
#include "wfindent.h"
#include "defs.h"


wf_port_range::wf_port_range() :
  _min_port(0),
  _max_port(0)
{}

wf_port_range::wf_port_range(uint16_t min_port, uint16_t max_port) :
  _min_port(min_port),
  _max_port(max_port)
{}

wf_port_range::wf_port_range(const wf_port& min_port,
			     const wf_port& max_port) :
  _min_port(min_port),
  _max_port(max_port)
{}


bool
wf_port_range::set(uint16_t min_port, uint16_t max_port) {
  return _min_port.set(min_port) && _max_port.set(max_port);
}

bool
wf_port_range::set(const wf_port& min_port, const wf_port& max_port) {
  return _min_port.set(min_port.get()) && _max_port.set(max_port.get());
}

uint16_t
wf_port_range::min_port_get() const {
  return _min_port.get();
}

uint16_t
wf_port_range::max_port_get() const {
  return _max_port.get();
}

string
wf_port_range::tostr() const {
  ostringstream os;
  os << min_port_get() << ':' << max_port_get();
  return os.str();
}

bool
wf_port_range::belong(uint16_t port) const {
  return (port >= _min_port.get() && port <= _max_port.get());
}

bool
wf_port_range::belong(const wf_port& port) const {
  return (port.get() >= _min_port.get() && port.get() <= _max_port.get());
}

ostream&
wf_port_range::print(ostream& os) const {
  return os << tostr();
}

ostream&
wf_port_range::output_xml(ostream& os, unsigned int indent_level) const {
  return os << wf_indent(indent_level) << "<port_range>" <<
    _min_port << '-' << _max_port <<"</port_range>" << endl;
}


ostream&
operator<<(ostream& os, const wf_port_range& port_range) {
  return port_range.print(os);
}
