/*=========================================================================

  Program:   Visualization Toolkit
  Module:    $RCSfile: vtkTesting.h,v $

  Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
// .NAME vtkTesting - a unified VTK regression testing framework
// .SECTION Description
//
//  This is a VTK regression testing framework. Looks like this:
// 
//  vtkTesting* t = vtkTesting::New();
//
//  Two options for setting arguments
//  
//  Option 1:
//  for ( cc = 1; cc < argc; cc ++ )
//    {
//    t->AddArgument(argv[cc]);
//    }
//
//  Option 2:
//  t->AddArgument("-D");
//  t->AddArgument(my_data_dir);
//  t->AddArgument("-V");
//  t->AddArgument(my_valid_image);
//
//  ...
//  
//  Two options of doing testing:
//  
//  Option 1:
//  t->SetRenderWindow(renWin);
//  int res = t->RegressionTest(threshold);
//
//  Option 2:
//  int res = t->RegressionTest(test_image, threshold);
//  
//  ...
//
//  if ( res == vtkTesting::PASSED )
//    {
//    Test passed
//    }
//  else
//    {
//    Test failed
//    }
//

#ifndef __vtkTesting_h
#define __vtkTesting_h

#include "vtkObject.h"
#include <vtkstd/vector> // used for argv
#include <vtkstd/string> // used for argv

class vtkRenderWindow;
class vtkImageData;

class VTK_RENDERING_EXPORT vtkTesting : public vtkObject
{
public:
  static vtkTesting *New();
  vtkTypeRevisionMacro(vtkTesting,vtkObject);
  void PrintSelf(ostream& os, vtkIndent indent);

//BTX
  enum ReturnValue {
    FAILED = 0,
    PASSED = 1,
    NOT_RUN = 2,
    DO_INTERACTOR = 3
  };

  static int Test(int argc, char *argv[], vtkRenderWindow *rw, double thresh);
//ETX
  
  // Description:
  // Use front buffer for tests. By default use back buffer.
  vtkSetClampMacro(FrontBuffer, int, 0, 1);
  vtkBooleanMacro(FrontBuffer, int);
  vtkGetMacro(FrontBuffer, int);

  // Description:
  // Perform the test and return result. At the same time the output will be
  // written cout
  virtual int RegressionTest(double thresh);
  virtual int RegressionTest(double thresh,ostream &os);

  // Description:
  // Compare the image with the valid image.
  virtual int RegressionTest(vtkImageData* image, double thresh);
  virtual int RegressionTest(vtkImageData* image, double thresh, ostream& os);

  // Description:
  // Set and get the render window that will be used for regression testing.
  virtual void SetRenderWindow(vtkRenderWindow* rw);
  vtkGetObjectMacro(RenderWindow, vtkRenderWindow);

  // Description:
  // Set/Get the name of the valid image file
  vtkSetStringMacro(ValidImageFileName);
  const char *GetValidImageFileName();

  // Description:
  // Get the image difference.
  vtkGetMacro(ImageDifference, double);

  // Description:
  // Pass the command line arguments into this class to be processed. Many of
  // the Get methods such as GetValidImage and GetBaselineRoot rely on the
  // arguments to be passed in prior to retrieving these values. Just call
  // AddArgument for each argument that was passed into the command line
  void AddArgument(const char *argv);
  
  // Description
  // This method delete all arguments in vtkTesting, this way you can reuse 
  // it in a loop where you would have multiple testing.
  void CleanArguments();
  
  // Description:
  // Get some paramters from the command line arguments, env, or defaults
  const char *GetDataRoot();
  vtkSetStringMacro(DataRoot);

  // Description:
  // Get some paramters from the command line arguments, env, or defaults
  const char *GetTempDirectory();
  vtkSetStringMacro(TempDirectory);

  // Description:
  // Is a valid image specified on the command line areguments?
  int IsValidImageSpecified();

  // Description:
  // Is the interactive mode specified?
  int IsInteractiveModeSpecified();

  // Description:
  // Number of pixels added as borders to avoid problems with
  // window decorations added by some window managers.
  vtkSetMacro(BorderOffset, int);
  vtkGetMacro(BorderOffset, int);

protected:
  vtkTesting();
  ~vtkTesting();

  static char* IncrementFileName(const char* fname, int count);
  static int LookForFile(const char* newFileName);

  int FrontBuffer;
  vtkRenderWindow* RenderWindow;
  char* ValidImageFileName;
  double ImageDifference;
  char *TempDirectory;
  int BorderOffset;
  
//BTX
  vtkstd::vector<vtkstd::string> Args;
//ETX
  char *DataRoot;
  double StartWallTime;
  double StartCPUTime;
  
private:
  vtkTesting(const vtkTesting&);  // Not implemented.
  void operator=(const vtkTesting&);  // Not implemented.
};

#endif


