/*
** This file is part of the ViTE project.
**
** This software is governed by the CeCILL-A license under French law
** and abiding by the rules of distribution of free software. You can
** use, modify and/or redistribute the software under the terms of the
** CeCILL-A license as circulated by CEA, CNRS and INRIA at the following
** URL: "http://www.cecill.info".
** 
** As a counterpart to the access to the source code and rights to copy,
** modify and redistribute granted by the license, users are provided
** only with a limited warranty and the software's author, the holder of
** the economic rights, and the successive licensors have only limited
** liability.
** 
** In this respect, the user's attention is drawn to the risks associated
** with loading, using, modifying and/or developing or reproducing the
** software by the user in light of its specific status of free software,
** that may mean that it is complicated to manipulate, and that also
** therefore means that it is reserved for developers and experienced
** professionals having in-depth computer knowledge. Users are therefore
** encouraged to load and test the software's suitability as regards
** their requirements in conditions enabling the security of their
** systems and/or data to be ensured and, more generally, to use and
** operate it in the same conditions as regards security.
** 
** The fact that you are presently reading this means that you have had
** knowledge of the CeCILL-A license and that you accept its terms.
**
**
** ViTE developpers are (for version 0.* to 1.0):
**
**        - COULOMB Kevin
**        - FAVERGE Mathieu
**        - JAZEIX Johnny
**        - LAGRASSE Olivier
**        - MARCOUEILLE Jule
**        - NOISETTE Pascal
**        - REDONDY Arthur
**        - VUCHENER Clément 
**
*/
/*!
 *\file render_svg.hpp
 */
#ifndef RENDER_SVG
#define RENDER_SVG


#include <iostream>
#include <sstream>
#include <fstream>


#include "render.hpp"
#include "../message/Message.hpp"
#include "resource.hpp"

/*!
 * \brief This class provides a SVG display ,it inherited from the Render interface.
 */
class Svg : public Render{


private:
  
    /*!
     * \brief buffer containing the main objets to be displayed
     */
    std::ostringstream _buffer;

    /*!
     * \brief buffer containning the smallest objet which must be display over main objet to be visible
     */
    std::ostringstream _thin_element_buffer;

    /*!
     * \brief buffer containning each counters values
     */
    std::ostringstream _chronogramme;

    /*!
     * \brief buffer containning the scale
     */
    std::ostringstream _time_scale;

    /*!
     * \brief State of the chronogramme drawer automaton
     * the automaton can be _INIT, _WAIT_FOR_POINT and _WAIT_NEW_CHRONO
     */
    int _chronogramme_state;

    /*!
     * \brief output svg file
     */
    std::ofstream _svg_file;
    /*!
     * \brief width container sum which is the abscisse of the beginnig of the events display
     */
    Element_pos _container_width_max;

    /*!
     * \brief minimum container height which is the height of any event
     */
    Element_pos _container_height_min;


    /*!
     * \brief Statistics and Informations about input trace computed in live
     */
    Element_pos _total_width;
   

    /*!
     * \brief Statistics and Informations about input trace computed in live
     */
    Times _total_time;

    /*!
     * \brief Statistics and Informations about input trace computed in live
     */
    bool _debug_svg_already_closed_once;




    /*!
     * \brief flush the buffer containing svg element on the file
     */
    inline void print();
    /*!
     * \brief add a rectangle to the buffer of drawing
     */   
    inline void rectangle(const char* name,Element_pos w, Element_pos h,Element_pos x1,Element_pos y1, unsigned int r, unsigned int g, unsigned int b);

    /*!
     * \brief add a line to the buffer of drawing
     */   
    inline void line(const char* name, Element_pos x1, Element_pos y1, Element_pos x2, Element_pos y2);


    /*!
     * \brief add a triangle shape to the buffer of drawing
     */ 
    inline void triangle(const char* name,Element_pos x1,Element_pos y1,Element_pos x2,Element_pos y2, Element_pos x3,Element_pos y3);

public:


    /*!
     * \brief Distance between two container
     */
    static const int _MARGIN;

    /*!
     * \brief Spike size
     */
    static const int _ARROWSIZE;


    /*!
     * \brief Distance between two states
     */
    static const double _MARGINSTATE;
    
    /*!
     * \brief offset between the top and the picture due to the scale
     */
    static const int _MARGINTOP;

    /*!
     * \brief a State of chronogramme drawer automaton 
     */
    static const int _WAIT_NEW_CHRONO;

    /*!
     * \brief a State of chronogramme drawer automaton
     */
    static const int _INIT;

    /*!
     * \brief a State of chronogramme drawer automaton
     */
    static const int _WAIT_FOR_POINT;


    /*!
     * \brief Trace can be exported on a selected period of time, beginning with _start_interval_time, default 0 and command line changed with -t _start_interval_time _end_interval_time or the static svg::set_interval(Times t1,Times t2)
     */
    static Times _start_interval_time;

  
    /*!
     * \brief Trace can be exported on a selected period of time, endding with _end_interval_time, default 0 (no upper selection) and command line changed with -t _start_interval_time _end_interval_time  or the static svg::set_interval(Times t1,Times t2)
     */   
    static Times _end_interval_time;


    /*!
     * \brief Svg display is widden by _wide_factor (in pixel/second), default 100, and command line changed with -w _wide_factor
     */   
    static  Element_pos _wide_factor;

    /*!
     *\brief Svg display is grown by _height_factor (in pixel/second), default 20, and command line changed with -w _wide_factor
     It corresponds to the height of a containers in pixel
    */

    static  Element_pos _height_factor;

    /*!
     * \brief accuracy required : an element less wide than  _epsilon will not be printed
     */
    static double _epsilon;


    /*!
     * \brief the scale can be more accurate by increasing the _scale_frequency
     */
    static double _scale_frequency;

    /*STATIC : allows commands line parameter*/

    /*!
     * \brief Trace can be exported on a selected period of time, starting with  _start_interval_time and endding with _end_interval_time, default 0 (no upper selection) and command line changed with -t _start_interval_time _end_interval_time  or the static svg::set_interval(Times t1,Times t2)
     */
    static void set_interval(const Times t1, const Times t2);
    /*!
     * \brief Set the precision of SVG display.
     * \param epsilon Precision
     */
    static void set_accuracy(const double epsilon);
    /*!
     * \brief Svg display is widden by _wide_factor (in pixel/second), default 100, and command line changed with -w _wide_factor
     */
    static void set_wide_factor (const Element_pos w);
    /*!
     * \brief Svg display is grown by _height
     */
    static void set_height_factor (const Element_pos h);

    /*!
     * \brief Return the height factor.
     * \return The height factor.
     */
    static Element_pos get_height_factor();

    /*!
     * \brief Return the widet factor.
     * \return The wide factor.
     */
    static Element_pos get_wide_factor();

    /*!
     * \brief the scale can be more accurate by increasing the _scale_frequency
     */
    static void set_scale_frequency(const Element_pos w);




    /*Statistics : not yet implemented*/
    
    /*!
     * \brief The width of the trace
     */
    void set_total_width(const Element_pos);
    /*!
     * \brief The time span of the trace
     */
    void set_total_time(const Times);




    /*drawing functions*/

    /*!
     * \brief SVG header buiding
     */
    void init(const char *path);

    /*!
     * \brief SVG bottom file buiding
     */
    void end();


    /*!
     * \brief unused
     */
    void start_draw();

    /*!
     * \brief unused
     */
    void start_draw_containers();

    /*!
     * \brief Draw a container according to the parameters
     * \param x the x position of the container
     * \param y the y position of the container
     * \param w the width of the container
     * \param h the height of the container
     */
    void draw_container(const Element_pos x, const Element_pos y, const Element_pos w, const Element_pos h);

    /*!
     * \brief Draw associated container name
     */
    void draw_container_text(const Element_pos x, const Element_pos y, const std::string value);
    /*!
     * \brief unused
     */
    void end_draw_containers();

    /*!
     * \brief unused
     */
    void start_draw_states();

    /*!
     * \brief Draw a state of the trace.
     * \param r the red color rate of the state.
     * \param g the green color rate of the state.
     * \param b the blue color rate of the state.
     * \param start the beginning time of the state.
     * \param end the ending time of the state.
     * \param level refer to the container which state belongs to.
     * \param height Unused parameter yet.
     */
    void draw_state(const Element_pos start , const Element_pos end, const Element_pos level, const Element_pos height, const Element_col r, const Element_col g, const Element_col b);

    /*!
     * \brief unused
     */
    void start_draw_arrows();

    /*!
     * \brief Draw an arrow
     *
     */
    void draw_arrow(const Element_pos start_time, const Element_pos end_time, const Element_pos start_height, const Element_pos end_height);

    /*!
     * \brief unused
     */
    void end_draw_arrows();

    /*!
     * \brief unused
     */
    void end_draw_states();

    /*!
     * \brief warning : not yet implemented
     */
    void draw_event(const Element_pos time, const Element_pos height, const Element_pos container_height);


    /*!
     * \brief initialise drawer automaton
     */
    void start_draw_counter();
    

    /*!
     * \brief add a point or switch chronogramme drawing
     */
    void draw_counter(const Element_pos x, const Element_pos y);
    
    /*!
     * \brief close drawer automaton
     */
    void end_draw_counter();


    /*!
     * \brief display the scale 
     */
    void display_time_scale();


    /*!
     * \brief unused
     */
    void end_draw();
  
    
    virtual ~Svg();


};



/***********************************
 *
 *
 *
 * Drawing function for the trace.
 *
 *
 *
 **********************************/


inline void Svg::start_draw(){
 
}

inline void Svg::draw_container_text(const Element_pos x, const Element_pos y, const std::string value){
  
    Element_pos xprime,yprime;

    /*positions are translated to _MARGINTOP            : the upper region reserved for scale*/
    /*heights are grown by _height_factor              : conversion from time in ms to pixel*/
    yprime =  _height_factor*y + _MARGINTOP;
    xprime =  _height_factor*x;

    _thin_element_buffer << "<text x='" << xprime <<"' y='" << yprime << "'> ";
    _thin_element_buffer << value;
    _thin_element_buffer << "</text>";

} 

inline void Svg::start_draw_containers(){

}

inline void Svg::end_draw_containers(){
}

inline void Svg::start_draw_states(){
}

inline void Svg::end_draw_states(){
}


/********************
 * Counter
 *******************/

inline void Svg::start_draw_counter(){
    _chronogramme_state = _INIT;
 
}

inline void Svg::draw_counter(const Element_pos x, const Element_pos y){
 
    /*computed position according to adjusted parameters*/
    /*widths are widen by _wide_factor                 : conversion from time in ms to pixel*/
    /*heights are grown by _height_factor              : conversion from time in ms to pixel*/
    /*positions are translated to _container_width_max : the left region occupied by containers*/
    /*positions are translated to _MARGINTOP            : the upper region reserved for scale*/
    /*whenever a time selection is set                 : positions are translated to -_start_interval_time (in ms, ie  -_start_interval_time*_wide_factor pixel)*/

    Element_pos xprime = x*_wide_factor +  _container_width_max;
    Element_pos yprime = y* _height_factor + _MARGINTOP;
    Element_pos switchvalue = _container_width_max;

 
  
    if (_chronogramme_state == _INIT) {
        if (xprime == switchvalue) {
           
            _chronogramme << "\n<path d='M"<< xprime +  _start_interval_time*_wide_factor << " " << yprime;
            _chronogramme_state=_WAIT_FOR_POINT;
        }
        else
            std::cerr<<"draw_counter not initialised";
	  
    }
    else if (_chronogramme_state == _WAIT_FOR_POINT) {
        if (xprime == switchvalue) {
           
            _chronogramme << "'/>";
            _svg_file.write(_chronogramme.str().c_str(), _chronogramme.str().size());
            _chronogramme.str("");
            _chronogramme_state=_WAIT_NEW_CHRONO;
        }
        else {
            if (_end_interval_time && x > _end_interval_time)
                return;
	  
            if (x<_start_interval_time)	
                return;
	  

	  _total_time = std::max( _total_time,x);
	  
            _chronogramme << " L" << xprime << " " << yprime;
        }
      
    }
  
    else  if (_chronogramme_state==_WAIT_NEW_CHRONO) {
        if (xprime==switchvalue) {
            _chronogramme << "\n<path d='M"<< xprime +  _start_interval_time*_wide_factor<< " " << yprime;
            _chronogramme_state=_WAIT_FOR_POINT;
        }
        else
            std::cerr<<"draw_counter not initialised";
	  

    }
  

}

inline void Svg::end_draw_counter() {
    if (_chronogramme_state!=_WAIT_NEW_CHRONO && _chronogramme_state!=_INIT)
        std::cerr<<"draw_counter not un-initialised";
}


inline void Svg::end_draw(){
    
}

void Svg::rectangle(const char* name, Element_pos w, Element_pos h,Element_pos x1,Element_pos y1, unsigned int r, unsigned int g, unsigned int b) {
    _buffer << "\n<rect class='"<<name<<"' title='"<<name<<"' width='" << w
            <<"' height='"<< h
            <<"' x='"     << x1
            <<"' y='"     << y1
            <<"' fill='rgb("<<r<<","<<g<<","<<b
            <<")'/>";
    print();
}





void Svg::triangle(const char* name,Element_pos x1,Element_pos y1,Element_pos x2,Element_pos y2, Element_pos x3,Element_pos y3) {
    _thin_element_buffer << "\n<polyline title='"<<name<<"' class='triangle' points='"
                         <<        x1 << "," << y1
                         << " " << x2 << "," << y2
                         << " " << x3 << "," << y3
                         <<"' />";
    
}

void Svg::print() {
    _svg_file.write(_buffer.str().c_str(),  _buffer.str().size());
    _buffer.str("");
}




inline void Svg::line(const char* name, Element_pos x1, Element_pos y1, Element_pos x2, Element_pos y2) {
    _thin_element_buffer << "\n<line title='"<<name<<"' x1='" << x1
                         <<"' y1='"     << y1
                         <<"' x2='"     << x2
                         <<"' y2='"     << y2
                         <<"' />";
    print();
}

inline void Svg::start_draw_arrows(){
}

inline void Svg::draw_arrow(const Element_pos start_time, const Element_pos end_time, const Element_pos start_height, const Element_pos end_height) {

    /*selection*/
    /*may be outofrange*/
    if ((_end_interval_time && start_time>_end_interval_time) || (end_time < _start_interval_time))
        return;

    /*arrow line equation y = a*x+b  */
    Element_pos a =  (1.0*(end_height - start_height))/(end_time - start_time);
    Element_pos b = start_height - a*(start_time);


    /*parameter to be adjusted to the time selection*/
    Element_pos start_timeprime;
    Element_pos end_timeprime;
    Element_pos start_heightprime;
    Element_pos end_heightprime;

    bool trunc = false;

    /*begining : left adjustement*/
    if (start_time<_start_interval_time) {
        start_timeprime =  _start_interval_time ;
        start_heightprime = a*start_timeprime+b;
    }
    else {
        start_timeprime =  start_time ;
        start_heightprime =  start_height;
    }


    /*endding as specified parameter*/
    end_timeprime =  end_time ;
    end_heightprime =  end_height;

    /*except if an _end_interval_time is set */
    if (_end_interval_time) {
        if (end_time>_end_interval_time) {      
            /*adjustement on the right and no spike !*/
            trunc = true;
            end_timeprime = _end_interval_time;
            end_heightprime =  a*end_timeprime+b;
        }

    }



  /*update of the time*/
  _total_time = std::max( _total_time,end_timeprime);


    /*computed position according to adjusted parameters*/
    /*width are widen by _wide_factor                  : conversion from time in ms to pixel*/
    /*heights are grown by _height_factor              : conversion from time in ms to pixel*/
    /*positions are translated to _container_width_max : the left region occupied by containers*/
    /*positions are translated to _MARGINTOP            : the upper region reserved for scale*/
    /*whenever a time selection is set                 : states are translated to -_start_interval_time (in ms, ie  -_start_interval_time*_wide_factor pixel)*/

  
    Element_pos x1=start_timeprime*_wide_factor +  _container_width_max -_start_interval_time*_wide_factor ;
    Element_pos y1=start_heightprime* _height_factor + _MARGINTOP;
    Element_pos x2=end_timeprime*_wide_factor+  _container_width_max -_start_interval_time*_wide_factor;
    Element_pos y2=end_heightprime* _height_factor + _MARGINTOP;


    /*an arrow :*/
    Svg::line("arrow",x1 ,y1 ,x2,y2);

    if (!trunc)
        Svg::triangle("arrow",x2 ,y2+_ARROWSIZE,x2 ,y2-_ARROWSIZE,x2+_ARROWSIZE,y2);
  
  
}


inline void Svg::end_draw_arrows(){
}

inline void Svg::draw_container(const Element_pos x, const Element_pos y, const Element_pos w, const Element_pos h) {

    Element_pos xprime,yprime,wprime,hprime;
  
    if (h<=0 || w<=0 || x<0 || y<0){
        std::cerr<<"draw_container : specified argument isn't positive double"<<std::endl;
        return;
    }

    /*parameters are grown                             : conversion from time in ms to pixel*/
    /*positions are translated to _MARGINTOP            : the upper region reserved for scale*/
    wprime =  _height_factor*w;
    hprime =  _height_factor*h;
    yprime =  _height_factor*y + _MARGINTOP;
    xprime =  _height_factor*x;

 

    Svg::rectangle("container",wprime, hprime, xprime, yprime, 0xbb, 0xbb, 0xbb);
  

  
  _container_width_max = std::max(wprime+xprime+ _MARGIN,_container_width_max);
  _container_height_min = std::min(hprime,_container_height_min);
}



inline void Svg::draw_state(const Element_pos start , const Element_pos end, const Element_pos level, const Element_pos height, const Element_col r, const Element_col g, const Element_col b) {
  
    Element_pos w;
    Element_pos h;
    Element_pos x;
    Element_pos y;

    Element_pos startprime;
    Element_pos endprime;
 
    if (start<=0 || end <=0 || height<0 || level<0){
        std::cerr<<"draw_state : specified argument isn't positive double"<<std::endl;
        return;
    }

  
    /*selection*/
    /*an outofrange state will not be displayed*/
    if ((_end_interval_time && start>_end_interval_time) || end<_start_interval_time)
        return;
  
    /*a state may be shortenend on its left*/
    if (start < _start_interval_time)
        startprime = _start_interval_time;
    else
        startprime = start ;

    /*or on its end*/
    endprime = end;
    if (_end_interval_time && end>_end_interval_time) {
        endprime = _end_interval_time;
    }
    
  

  
  _total_time = std::max( _total_time,endprime);


    /*adjusted parameter to positions*/
    /*states are widen by _wide_factor              : conversion from time in ms to pixel*/
    /*states are grown by _height_factor            : conversion from time in ms to pixel*/
    /*states are translated to _container_width_max : the left region occupied by containers*/
    /*states are translated to _MARGINSTATE          : the margin between two container*/
    /*states are translated to _MARGINTOP            : the upper region reserved for scale*/
    /*states height is not the specified parameter  : container_height_min replaces height which enable the state to have an uniform size */
    /*whenever a time selection is set              : states are translated to -_start_interval_time (in ms, ie  -_start_interval_time*_wide_factor pixel)*/
  
    w = endprime*_wide_factor-startprime*_wide_factor;
    h = _container_height_min;
    x = (startprime )*_wide_factor + _container_width_max + _MARGINSTATE -_start_interval_time*_wide_factor;
    y = level* _height_factor + _MARGINTOP;


    if(w < _epsilon){
        return;
    }
  
 
    Svg::rectangle("state", w, h, x, y, (unsigned int)r*255, (unsigned int)g*255, (unsigned int)b*255);
}



/*!
 * \fn void Svg::draw_event(const Element_pos time, const Element_pos height, const Element_pos container_height)
 * \brief Draw the events in the svg
 * \param time The time the element occur
 * \param height The height of the event
 * \param container_height The height of the container that contains it
 */
inline  void Svg::draw_event(const Element_pos time, const Element_pos height, const Element_pos container_height) {

    if (time<=0 || height <=0 || container_height <=0){
        std::cerr<<"draw_event : specified argument isn't positive double"<<std::endl;
        return;
    }
  
   
  
    /*an outofrange event will not be displayed*/
    if ((_end_interval_time && time > _end_interval_time) || time < _start_interval_time)
        return;
  
    _total_time = std::max(_total_time, time);


    _total_time = std::max( _total_time,time);

  Svg::rectangle("event",_MARGINSTATE,_container_height_min, time +  _container_width_max -_start_interval_time*_wide_factor, height + _MARGINTOP, 0x17, 0x60, 0xe7);
  
}



#endif // RENDER_SVG
