/*
** This file is part of the ViTE project.
**
** This software is governed by the CeCILL-A license under French law
** and abiding by the rules of distribution of free software. You can
** use, modify and/or redistribute the software under the terms of the
** CeCILL-A license as circulated by CEA, CNRS and INRIA at the following
** URL: "http://www.cecill.info".
** 
** As a counterpart to the access to the source code and rights to copy,
** modify and redistribute granted by the license, users are provided
** only with a limited warranty and the software's author, the holder of
** the economic rights, and the successive licensors have only limited
** liability.
** 
** In this respect, the user's attention is drawn to the risks associated
** with loading, using, modifying and/or developing or reproducing the
** software by the user in light of its specific status of free software,
** that may mean that it is complicated to manipulate, and that also
** therefore means that it is reserved for developers and experienced
** professionals having in-depth computer knowledge. Users are therefore
** encouraged to load and test the software's suitability as regards
** their requirements in conditions enabling the security of their
** systems and/or data to be ensured and, more generally, to use and
** operate it in the same conditions as regards security.
** 
** The fact that you are presently reading this means that you have had
** knowledge of the CeCILL-A license and that you accept its terms.
**
**
** ViTE developpers are (for version 0.* to 1.0):
**
**        - COULOMB Kevin
**        - FAVERGE Mathieu
**        - JAZEIX Johnny
**        - LAGRASSE Olivier
**        - MARCOUEILLE Jule
**        - NOISETTE Pascal
**        - REDONDY Arthur
**        - VUCHENER Clément 
**
*/
/*!
 *\file render.hpp
 */

#ifndef RENDER_HPP
#define RENDER_HPP


class Render;


#include "resource.hpp"


/*!
 * \brief This class provides an interface for render classes like OpenGL or SVG.
 */
class Render
{ 

public:
     
    /*!
     * \brief Proceeds with the initialization of draw functions.
     */
    virtual void start_draw() = 0;

    /*!
     * \brief Proceeds with the initialization of container draw functions.
     */
    virtual void start_draw_containers() = 0;

    /*!
     * \brief Draw a container according to the parameters
     * \param x the x position of the container
     * \param y the y position of the container
     * \param w the width of the container
     * \param h the height of the container
     */
    virtual void draw_container(const Element_pos x, const Element_pos y, const Element_pos w, const Element_pos h) = 0;

    /*!
     * \brief Draw the text of a container.
     * \param x the x position of the text.
     * \param y the y position of the text.
     * \param value the string value of the text.
     */
    virtual void draw_container_text(const Element_pos x, const Element_pos y, const std::string value) = 0;

    /*!
     * \brief Called when all container draws are finished.
     */
    virtual void end_draw_containers() = 0;
     
    /*!
     * \brief Proceeds with the initialization of state draw functions.
     */
    virtual void start_draw_states() = 0;

    /*!
     * \brief Draw a state of the trace.
     * \param start the beginning time of the state.
     * \param end the ending time of the state.
     * \param base vertical position of the state.
     * \param height the state height.
     * \param r the red color rate of the state.
     * \param g the green color rate of the state.
     * \param b the blue color rate of the state.
     */
    virtual void draw_state(const Element_pos start , const Element_pos end, const Element_pos base, const Element_pos height, const Element_col r, const Element_col g, const Element_col b) = 0;

    /*!
     * \brief Called when all state draws are finished.
     */
    virtual void end_draw_states() = 0;

    /*!
     * \brief Proceeds with the initialization of arrow draw functions.
     */
    virtual void start_draw_arrows() = 0;

    /*!
     * \brief Draw an arrow.
     * \param start_time the beginning time of the arrow.
     * \param end_time the ending time of the arrow.
     * \param start_height vertical position of the begining time of the arrow.
     * \param end_height vertical position of the ending time of the arrow.
     */
    virtual void draw_arrow(const Element_pos start_time, const Element_pos end_time, const Element_pos start_height, const Element_pos end_height) = 0;

    /*!
     * \brief Called when all arrow draws are finished.
     */
    virtual void end_draw_arrows() = 0;

    /*!
     * \brief Draw an event.
     * \param time time when the event occurs.
     * \param height vertical position of the event.
     * \param container_height information to draw event. It corresponds to the container height when they are drawn horizontally.
     */
    virtual void draw_event(const Element_pos time, const Element_pos height, const Element_pos container_height) = 0;

    /*!
     * \brief Proceeds with the initialization of counter draw functions.
     */
    virtual void start_draw_counter() = 0;
     
    /*!
     * \brief Draw a point of the counter.
     * \param x x position of the point.
     * \param y y position of the point.
     *
     * Each time counter is increased, this function is called with the coordinates of the new point.
     */
    virtual void draw_counter(const Element_pos x, const Element_pos y) = 0;
     
    /*!
     * \brief Called when all counter draws are finished.
     */
    virtual void end_draw_counter() = 0;
     
    /*!
     * \brief Called when all draws are finished.
     */
    virtual void end_draw() = 0;

   
    /*!
     * \brief Set Statistics and Informations about input trace
     */
    virtual void set_total_width(Element_pos w) = 0;

    /*!
     * \brief Set Statistics and Informations about input trace
     */
    virtual void set_total_time(Times t) = 0;
    /*!
     * \brief display the scale 
     */
    virtual void display_time_scale() = 0;
};



#endif
