/*
** This file is part of the ViTE project.
**
** This software is governed by the CeCILL-A license under French law
** and abiding by the rules of distribution of free software. You can
** use, modify and/or redistribute the software under the terms of the
** CeCILL-A license as circulated by CEA, CNRS and INRIA at the following
** URL: "http://www.cecill.info".
** 
** As a counterpart to the access to the source code and rights to copy,
** modify and redistribute granted by the license, users are provided
** only with a limited warranty and the software's author, the holder of
** the economic rights, and the successive licensors have only limited
** liability.
** 
** In this respect, the user's attention is drawn to the risks associated
** with loading, using, modifying and/or developing or reproducing the
** software by the user in light of its specific status of free software,
** that may mean that it is complicated to manipulate, and that also
** therefore means that it is reserved for developers and experienced
** professionals having in-depth computer knowledge. Users are therefore
** encouraged to load and test the software's suitability as regards
** their requirements in conditions enabling the security of their
** systems and/or data to be ensured and, more generally, to use and
** operate it in the same conditions as regards security.
** 
** The fact that you are presently reading this means that you have had
** knowledge of the CeCILL-A license and that you accept its terms.
**
**
** ViTE developpers are (for version 0.* to 1.0):
**
**        - COULOMB Kevin
**        - FAVERGE Mathieu
**        - JAZEIX Johnny
**        - LAGRASSE Olivier
**        - MARCOUEILLE Jule
**        - NOISETTE Pascal
**        - REDONDY Arthur
**        - VUCHENER Clément 
**
*/
/*!
 *\file TokenSource.hpp
 */

#ifndef TOKENSOURCE_HPP
#define TOKENSOURCE_HPP

#include <string>
#include <iostream>

#ifndef VITE_DEBUG
  #include "../message/Errors.hpp"
#else
// For debugging
  #include "../../tests/stubs/Errors.hpp"
#endif

#if defined (WIN32) || !defined(_POSIX_MAPPED_FILES) // They do not have mmap
/*!
 * \def BUFFER_SIZE
 * \brief For windows only, default buffer size
 */
    #define BUFFER_SIZE 2048
    #include <fstream>
#else
    #include <sys/types.h>
    #include <sys/stat.h>
    #include <fcntl.h>
    #include <sys/mman.h>
    #include <stdio.h> 
    #include <stdlib.h>       
    #include <unistd.h>
#endif

/*! 
 * \class TokenSource 
 * \brief supply tokens from a file trace. Tokens are word unit used in PajeFile trace formats.
 */
class TokenSource{
  
private:
    
#if defined (WIN32) || !defined(_POSIX_MAPPED_FILES)
    char _buffer[BUFFER_SIZE];
#else
    char* _buffer;
#endif
  
    std::string _token;
    static int _cursor;

    bool _is_eof;
    bool _is_mmap_ok;
    bool _is_file_opened;
  
#if defined (WIN32) || !defined(_POSIX_MAPPED_FILES)
    int _buffer_size;
    static std::ifstream _file;
  
    static int _filesize;
#else
    int _fd;
  
    static off_t _filesize;
#endif
  
    
#if defined (WIN32) || !defined(_POSIX_MAPPED_FILES)
    bool fill_buffer();
#endif

    bool go_to_next_token();
    bool ensure_capacity();
  
    void build_definition_starter();
    void build_line_breaks();
    void build_composite_token();
    void build_simple_token();

public:
    
    /*!
     *  \fn    TokenSource()
     *  \brief This constructor needs a file to be opened
     */
    TokenSource();
    
    /*!
     *  \fn    TokenSource(const char* filename);
     *  \brief Constructor opening a file
     *  \param filename the name of the file we want to open.
     */
    TokenSource(const char* filename);
  
    /*!
     *  \fn    ~TokenSource
     *  \brief Destuctor which release the file
     */
    ~TokenSource();


    /*!
     *  \fn    open(const char *filename)
     *  \brief open a file
     *  \param filename the name of the file we want to open.
     */
    void open(const char *filename);

    /*!
     *  \fn     read_token
     *  \brief  supplies a token
     *  \return a \\0-ended character string (char*), NULL if the file ended, NULL if no file is opened
     */
    std::string* read_token();


    /*!
     *  \fn get_size_loaded()
     *  \brief get the size already parsed.
     *  \return the scale of the size already parsed. (between 0 and 1)
     */
    static float get_size_loaded();

    /*!
     *  \fn reinit_cursor()
     *  \brief Reinitialise the cursor of the file
     */
    static void reinit_cursor();

 
};

#endif // TOKENSOURCE_HPP
