/*
**  This file is part of Vidalia, and is subject to the license terms in the
**  LICENSE file, found in the top level directory of this distribution. If you
**  did not receive the LICENSE file with this file, you may obtain it from the
**  Vidalia source package distributed by the Vidalia Project at
**  http://www.vidalia-project.net/. No part of Vidalia, including this file,
**  may be copied, modified, propagated, or distributed except according to the
**  terms described in the LICENSE file.
*/

/*
** \file CrashReportUploader.h
** \version $Id$
** \brief Uploads a minidump file and any extra information to a crash
** reporting server.
*/

#ifndef _CRASHREPORTUPLOADER_H
#define _CRASHREPORTUPLOADER_H

#include <QObject>
#include <QHttp>
#include <QMap>

class QUrl;
class QString;
class QByteArray;


class CrashReportUploader : public QObject
{
  Q_OBJECT

public:
  /** Default constructor.
   */
  CrashReportUploader(QObject *parent = 0);

  /** Starts uploading <b>minidump</b> to <b>serverUrl</b> and returns
   * immediately. <b>minidumpId</b> is the minidump GUID generated by
   * the exception handler and used for the minidump's filename. The
   * optional <b>parameters</b> can be used to pass additional form fields
   * to the crash reporting server.
   */
  void uploadMinidump(const QUrl &serverUrl,
                      const QString &minidumpId,
                      const QByteArray &minidump,
                      const QMap<QString,QString> &parameters);

public slots:
  /** Cancels a pending minidump upload.
   */
  void cancel();

signals:
  /** Emitted when the underlying QHttp object posts data to the server.
   * <b>done</b> indicates how many bytes out of <b>total</b>
   * have been sent so far.
   */
  void uploadProgress(int done, int total);

  /** Emitted when the status of the POST request changes. <b>status</b>
   * describes the new current state of the request.
   */
  void statusChanged(const QString &status);

  /** Emitted when the previous minidump upload completes successfully.
   */
  void uploadFinished();

  /** Emitted when the previous crash report upload fails. The QString
   * <b>error</b> is a human-readable string describing the error
   * encountered.
   */
  void uploadFailed(const QString &error);

private slots:
  /** Called when the state of the underlying QHttp object changes. A
   * statusChanged() signal is emitted with the appropriate text
   * describing the new state of the POST request.
   */
  void httpStateChanged(int state); 

  /** Called when the underlying QHttp object used to upload the minidump
   * completes. <b>error</b> is set to false if the upload was
   * successful, or true if the upload failed. If <b>id</b> does not
   * match the request ID previously returned by QHttp::get(), then the
   * signal is ignored since it is the result of a close() or abort()
   * request.
   */
  void httpRequestFinished(int id, bool error);

private:
  /** Generates a "random" 8-byte multipart boundary marker encoded into
   * 16 hex characters.
   */
  QString generateBoundaryMarker() const;

  /** Unique numeric identifier of the current minidump upload POST request.
   */
  int _requestId;

  /** Object used to POST a minidump to the crash server and read the
   * response.
   */
  QHttp *_http;
};

#endif

