/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "visu_rendering.h"
#include "visu_data.h"

#include "visu_object.h"
#include "visu_tools.h"
#include "visu_configFile.h"
#include "coreTools/toolConfigFile.h"

#include <glib.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/**
 * SECTION:visu_rendering
 * @short_description: Methods to create and add new rendering
 * methods.
 *
 * <para> The way visu renders its data is done by modules. They are
 * called rendering methods and they describes how data are drawn on
 * the screen. Many can be defined but only one is used at a time to
 * render the data.</para>
 *
 * <para>One or more file type are associated with a rendering
 * method. And a rendering method must specify the way to load the
 * data it needs. Taking the example of a spin system representation,
 * there are two kinds of file. The first kind describes the position
 * the spin and the second contains their orientations.</para>
 *
 * <para>To create a new rendering method, use
 * renderingMethod_new(). The name is mandatory and must be
 * unique. The description is not compulsory. The number of file kinds
 * is also required. Use renderingMethodSet_fileType() to associated a
 * #GList of #FileFormat. In our example of spin system, the first
 * kind of file is about positions, and the associated file formats
 * are *.ascii, *.d3 and *.xyz.</para>
 *
 * <para>The #VisuRendering_struct has to two pointers on methods
 * that are important. The first, createOpenGLElementFunc() is called
 * when V_Sim needs to create an OpenGL list corresponding to the
 * #VisuElement given as argument. This list then can be used to
 * render each node and thus accelerating the rendering
 * operations. The second method is createOpenGLNodeFunc() and is
 * called by V_Sim for each node of the system when the main OpenGL
 * list (the one for the nodes) is created. Thus, in the contrary of
 * the first method, thios one should not create an OpenGL list but
 * directly call OpenGL routines to draw the node. This method is also
 * responsible to put the node at the right position. Use
 * visuDataGet_nodePosition() to retrieve the position and translate
 * the node accordingly.</para>
 */

/**
 * VisuRendering_struct:
 * @name: a string describing the method (not NULL) ;
 * @printName: a string describing the method (not NULL) that can be safely translated ;
 * @description: a string to give a longer description to the method (can be NULL) ;
 * @icon: a path to a small 16x16 image to represent the method,
 *   if null, no image is used ;
 * @nbFilesType: the number of files required to render one set of data,
 *   for example 2 for one file with positions and one file with orientations
 *   to represent a vector field ;
 * @fileType: an array of GList* that contains #FileFormat ;
 * @fileTypeLabel: an array of string that describes each kind of file ;
 * @loadMethod: a pointer to a method used to read input data ;
 * @createElement: a pointer to a method used to create OpenGL lists for #VisuElement ;
 * @createNode: a pointer to a method used to draw a given node ;
 * @getExtensForElement: not used.
 *
 * This structure is used to describe a rendering method. Besides
 * names, representing icon... this structure stores pointers to
 * method technically used to draw elements with this method.
 */
struct _VisuRendering
{
  GObject parent;

  /* Some variable to describe this rendering method.
     The attribute name is mandatory since it is
     used to identify the method. */
  gchar* name;
  gchar* printName;
  gchar* description;
  /* A path to a small 16x16 image to represent the method.
     If null, no image is used. */
  gchar* icon;

  /* Dealing with file type. */
  /* Number of files required to render one set of data. */
  guint nbFilesType;
  /* Types of files described by this rendering method. */
  GList **fileType;
  /* Label associated to this each kind of file. */
  gchar **fileTypeLabel;

  /* Pointer to the file functions. */
  visuRenderingLoadFunc loadMethod;
  
  /* Pointers to useful openGL methods. */
  visuRenderingCreate_elementFunc createElement;
  visuRenderingCreate_nodeFunc    createNode;
  visuRenderingGet_nodeExtendFunc getExtend;

  gboolean dispose_has_run;
};

enum {
  FILE_TYPE_CHANGED,
  ELEMENT_SIZE_CHANGED,
  LAST_SIGNAL
};

struct _VisuRenderingClass
{
  GObjectClass parent;

  /* A GHashTable to store all the available rendering methods
     in the system. The keys are the name of each method. */
  GHashTable *tableOfMethods;
  /* The list of the available rendering method. */
  GList      *listOfMethods;

  /* A pointer to the in use rendering method. */
  VisuRendering *current;

  /* Id of possible signals. */
  guint signals[LAST_SIGNAL];

  GQuark quark;
};

static VisuRenderingClass *my_class = (VisuRenderingClass*)0;

#define FLAG_FAV_RENDER    "rendering_favoriteMethod"
#define DESC_FAV_RENDER    "Favorite method used to render files ; chain"
#define DEFAULT_RENDER_FAV ""
static gboolean readFavVisuRendering(gchar **lines, int nbLines, int position,
				       VisuData *dataObj, GError **error);
static void exportFavVisuRendering(GString *data, VisuData *dataObj);

/* Local callbacks. */
static void onResourcesLoaded(VisuObject *visu, VisuData *dataObj, gpointer data);

static void visu_rendering_dispose(GObject* obj);
static void visu_rendering_finalize(GObject* obj);



G_DEFINE_TYPE(VisuRendering, visu_rendering, G_TYPE_OBJECT)

static void visu_rendering_class_init(VisuRenderingClass *klass)
{
  VisuConfigFileEntry *resourceEntry;
  GType paramFloat[1] = {G_TYPE_FLOAT};

  DBG_fprintf(stderr, "Visu Rendering: creating the class of the object.\n");

  /* Connect freeing methods. */
  G_OBJECT_CLASS(klass)->dispose = visu_rendering_dispose;
  G_OBJECT_CLASS(klass)->finalize = visu_rendering_finalize;

  /* Initialise signals. */
  klass->signals[FILE_TYPE_CHANGED] = 
    g_signal_newv ("fileTypeChanged",
		   G_TYPE_FROM_CLASS (klass),
		   G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS,
		   NULL /* class closure */,
		   NULL /* accumulator */,
		   NULL /* accu_data */,
		   g_cclosure_marshal_VOID__VOID,
		   G_TYPE_NONE /* return_type */,
		   0     /* n_params */,
		   NULL  /* param_types */);

  klass->signals[ELEMENT_SIZE_CHANGED] = 
    g_signal_newv ("elementSizeChanged",
		   G_TYPE_FROM_CLASS (klass),
		   G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS,
		   NULL /* class closure */,
		   NULL /* accumulator */,
		   NULL /* accu_data */,
		   g_cclosure_marshal_VOID__FLOAT,
		   G_TYPE_NONE /* return_type */,
		   1    /* n_params */,
		   paramFloat  /* param_types */);

  /* Initialise internal variables. */
  resourceEntry = visuConfigFileAdd_entry(VISU_CONFIGFILE_PARAMETER,
					  FLAG_FAV_RENDER, DESC_FAV_RENDER,
					  1, readFavVisuRendering);
  visuConfigFileAdd_exportFunction(VISU_CONFIGFILE_PARAMETER,
				   exportFavVisuRendering);

  klass->current        = (VisuRendering*)0;
  klass->tableOfMethods = g_hash_table_new(g_str_hash, g_str_equal);
  klass->listOfMethods  = (GList*)0;

  klass->quark          =  g_quark_from_static_string("visu_rendering");

  g_signal_connect(VISU_INSTANCE, "resourcesLoaded",
 		   G_CALLBACK(onResourcesLoaded), (gpointer)0);

  my_class = klass;
}

static void visu_rendering_init(VisuRendering *obj)
{
  DBG_fprintf(stderr, "Visu rendering: initializing a new object (%p).\n",
	      (gpointer)obj);
  
  obj->name          = (gchar*)0;
  obj->printName     = (gchar*)0;
  obj->description   = (gchar*)0;
  obj->icon          = (gchar*)0;
  obj->nbFilesType   = 0;
  obj->fileTypeLabel = (gchar**)0;
  obj->fileType      = (GList**)0;
  obj->loadMethod    = (visuRenderingLoadFunc)0;
  obj->createElement = (visuRenderingCreate_elementFunc)0;
  obj->createNode    = (visuRenderingCreate_nodeFunc)0;
  obj->getExtend     = (visuRenderingGet_nodeExtendFunc)0;
  obj->dispose_has_run = FALSE;
}

/* This method can be called several times.
   It should unref all of its reference to
   GObjects. */
static void visu_rendering_dispose(GObject* obj)
{
  DBG_fprintf(stderr, "Visu Rendering: dispose object %p.\n", (gpointer)obj);

  if (VISU_RENDERING(obj)->dispose_has_run)
    return;

  VISU_RENDERING(obj)->dispose_has_run = TRUE;

  /* Chain up to the parent class */
  G_OBJECT_CLASS(visu_rendering_parent_class)->dispose(obj);
}
/* This method is called once only. */
static void visu_rendering_finalize(GObject* obj)
{
  VisuRenderingClass *klass;
  VisuRendering *meth;
  GList *tmpLst;
  guint i;

  g_return_if_fail(obj);

  DBG_fprintf(stderr, "Visu Rendering: finalize object %p.\n", (gpointer)obj);

  meth = VISU_RENDERING(obj);
  klass = VISU_RENDERING_GET_CLASS(meth);

  /* Remove object from class lists list. */
  klass->listOfMethods = g_list_remove(klass->listOfMethods, (gpointer)obj);
  g_hash_table_remove(klass->tableOfMethods, (gpointer)(meth->name));

  if (meth->name)
    g_free(meth->name);
  if (meth->printName)
    g_free(meth->printName);
  if (meth->description)
    g_free(meth->description);
  if (meth->fileType)
    {
      for (i = 0; i < meth->nbFilesType; i++)
	{
	  tmpLst = meth->fileType[i];
	  while(tmpLst)
	    {
	      fileFormatFree((FileFormat*)tmpLst->data);
	      tmpLst = g_list_next(tmpLst);
	    }
	}
      g_free(meth->fileType);
    }
}

VisuRendering* visuRendering_new(const char* name, const char* printName,
				 const char* description, guint nbFileType,
				 visuRenderingLoadFunc           loadMethod,
				 visuRenderingCreate_elementFunc createElement,
				 visuRenderingCreate_nodeFunc    createNode,
				 visuRenderingGet_nodeExtendFunc getExtend)
{
  VisuRendering *method;
  VisuRenderingClass *klass;
  guint i;

  g_return_val_if_fail(name && name[0] && description &&
		       loadMethod && createElement && createNode && getExtend,
		       (VisuRendering*)0);

  DBG_fprintf(stderr, "Visu Rendering: new object...\n");
  method = VISU_RENDERING(g_object_new(VISU_RENDERING_TYPE, NULL));
  g_return_val_if_fail(method, (VisuRendering*)0);

  DBG_fprintf(stderr, "Visu Rendering: setting values.\n");
  method->name          = g_strdup(name);
  method->printName     = g_strdup(printName);
  method->description   = g_strdup(description);
  method->nbFilesType   = nbFileType;
  method->loadMethod    = loadMethod;
  method->createElement = createElement;
  method->createNode    = createNode;
  method->getExtend     = getExtend;

  method->fileType      = g_malloc(sizeof(GList*) * nbFileType);
  method->fileTypeLabel = g_malloc(sizeof(gchar*) * nbFileType);
  for (i = 0; i < nbFileType; i++)
    {
      method->fileType[i]      = (GList*)0;
      method->fileTypeLabel[i] = (gchar*)0;
    }

  DBG_fprintf(stderr, "Visu Rendering: registering a new rendering method ... ");
  klass = VISU_RENDERING_GET_CLASS(method);
  g_hash_table_insert(klass->tableOfMethods,
		      (gpointer)method->name, (gpointer)method);
  klass->listOfMethods = g_list_append(klass->listOfMethods, method);
  
  DBG_fprintf(stderr, "'%s'\n", method->name);

  return method;
}


GList* visuRenderingClassGet_Allobjects()
{
  if (!my_class)
    visu_rendering_get_type();
  
  return my_class->listOfMethods;
}

/* Allocate the structure and give values to all the attributes. */
guint visuRenderingGet_nbFileType(VisuRendering *method)
{
  g_return_val_if_fail(IS_VISU_RENDERING_TYPE(method), -1);

  return method->nbFilesType;
}

void visuRenderingSet_fileType(VisuRendering *method, GList *fileTypeList,
			       guint fileType, const gchar* name)
{
  g_return_if_fail(IS_VISU_RENDERING_TYPE(method) && name && name[0]);
  g_return_if_fail(fileType < method->nbFilesType);

  method->fileType[fileType] = fileTypeList;
  if (method->fileTypeLabel[fileType])
    g_free(method->fileTypeLabel[fileType]);
  method->fileTypeLabel[fileType] = g_strdup(name);
}
void visuRenderingAdd_fileFormat(VisuRendering *method, FileFormat *fmt,
				 guint fileType)
{
  g_return_if_fail(IS_VISU_RENDERING_TYPE(method) && fmt);
  g_return_if_fail(fileType < method->nbFilesType);

  method->fileType[fileType] = g_list_prepend(method->fileType[fileType], fmt);
  g_signal_emit(G_OBJECT(method),
		VISU_RENDERING_GET_CLASS(method)->signals[FILE_TYPE_CHANGED],
		0, NULL);
}
GList* visuRenderingGet_fileType(VisuRendering *method, guint fileType)
{
  g_return_val_if_fail(IS_VISU_RENDERING_TYPE(method), (GList*)0);
  g_return_val_if_fail(fileType < method->nbFilesType, (GList*)0);

  return method->fileType[fileType];
}
const gchar* visuRenderingGet_fileTypeName(VisuRendering *method, guint fileType)
{
  g_return_val_if_fail(IS_VISU_RENDERING_TYPE(method), (gchar*)0);
  g_return_val_if_fail(fileType < method->nbFilesType, (gchar*)0);

  return method->fileTypeLabel[fileType];
}
float visuRenderingGet_sizeOfElement(VisuRendering *method, VisuElement *ele)
{
  g_return_val_if_fail(IS_VISU_RENDERING_TYPE(method) && ele, 0.);

  return method->getExtend(ele);
}
const gchar* visuRenderingGet_name(VisuRendering *method, gboolean UTF8)
{
  g_return_val_if_fail(IS_VISU_RENDERING_TYPE(method), (const gchar*)0);

  if (UTF8)
    return method->printName;
  else
    return method->name;
}
const gchar* visuRenderingGet_description(VisuRendering *method)
{
  g_return_val_if_fail(IS_VISU_RENDERING_TYPE(method), (const gchar*)0);

  return method->description;
}
const gchar* visuRenderingGet_iconPath(VisuRendering *method)
{
  g_return_val_if_fail(IS_VISU_RENDERING_TYPE(method), (const gchar*)0);

  return method->icon;
}

void visuRenderingSet_icon(VisuRendering *method, const char *path)
{
  g_return_if_fail(IS_VISU_RENDERING_TYPE(method));

  if (method->icon)
    {
      g_free(method->icon);
      method->icon = (gchar*)0;
    }
  if (!path)
    return;

  method->icon = g_strdup(path);
}

void visuRenderingEmit_elementSize(VisuRendering *method, float value _U_)
{
  DBG_fprintf(stderr, "Rendering Atomic: emit 'elementSizeChanged'.\n");
  g_signal_emit(G_OBJECT(method),
		VISU_RENDERING_GET_CLASS(method)->signals[ELEMENT_SIZE_CHANGED],
		0, NULL);
  DBG_fprintf(stderr, "Rendering Atomic: emission done.\n");
}


/* Choose the method used to render the data. */
void visuRenderingClassSet_current(VisuRendering* method)
{
  if (!my_class)
    visu_rendering_get_type();
  
  if (method == my_class->current)
    return;

  if (DEBUG)
    {
      if (method)
	fprintf(stderr, "Visu Rendering: set the rendering method to '%s' (%p).\n",
		method->name, (gpointer)method);
      else
	fprintf(stderr, "Visu Rendering: set the rendering method to 'None' (%p).\n",
		(gpointer)method);
    }
  
  my_class->current = method;
  DBG_fprintf(stderr, "Visu Rendering: emit method changed (%p).\n",
	      (gpointer)VISU_INSTANCE);
  g_signal_emit(VISU_INSTANCE, VISU_SIGNALS[RENDERINGCHANGED_SIGNAL],
		0, (gpointer)method, NULL);
}
gboolean visuRenderingClassSet_currentByName(const gchar* name)
{
  VisuRendering* method;

  if (!my_class)
    visu_rendering_get_type();
  
  method = g_hash_table_lookup(my_class->tableOfMethods, (gpointer)name);
  visuRenderingClassSet_current(method);
  return (method != (VisuRendering*)0);
}
/* Get the current method used to render the data. */
VisuRendering* visuRenderingClassGet_current()
{
  if (!my_class)
    visu_rendering_get_type();
  
  return my_class->current;
}
gint visuRenderingFormatCompare_priority(gconstpointer a, gconstpointer b)
{
  if (((RenderingFormatLoad*)a)->priority < ((RenderingFormatLoad*)b)->priority)
    return (gint)-1;
  else if (((RenderingFormatLoad*)a)->priority > ((RenderingFormatLoad*)b)->priority)
    return (gint)+1;
  else
    return (gint)0;
}

visuRenderingCreate_elementFunc visuRenderingClassGet_currentElementFunc()
{
  VisuRendering *current;

  current = visuRenderingClassGet_current();
  g_return_val_if_fail(current, (visuRenderingCreate_elementFunc)0);

  return my_class->current->createElement;
}
visuRenderingCreate_nodeFunc visuRenderingClassGet_currentNodeFunc()
{
  VisuRendering *current;

  current = visuRenderingClassGet_current();
  g_return_val_if_fail(current, (visuRenderingCreate_nodeFunc)0);

  return my_class->current->createNode;
}
visuRenderingLoadFunc visuRenderingClassGet_currentLoadFunc()
{
  VisuRendering *current;

  current = visuRenderingClassGet_current();
  g_return_val_if_fail(current, (visuRenderingLoadFunc)0);

  return my_class->current->loadMethod;
}
VisuRendering* visuRenderingClassGet_methodByName(const gchar *name)
{
  if (!my_class)
    visu_rendering_get_type();
  
  return (VisuRendering*)g_hash_table_lookup(my_class->tableOfMethods,
					     (gpointer)name);
}
gboolean visuRenderingClassIs_currentByName(const gchar *name)
{
  VisuRendering *meth;

  if (!my_class)
    visu_rendering_get_type();
  
  meth = (VisuRendering*)g_hash_table_lookup(my_class->tableOfMethods,
					     (gpointer)name);
  return (meth == my_class->current);
}



/****************/
/* Private area */
/****************/

GQuark visuRenderingClassGet_quark()
{
  if (!my_class)
    visu_rendering_get_type();
  
  return my_class->quark;
}

static void onResourcesLoaded(VisuObject *visu _U_, VisuData *dataObj,
			      gpointer data _U_)
{
  VisuDataIter iter;

  DBG_fprintf(stderr, "Visu Rendering: caught the 'resourcesLoaded' signal.\n");

  if (!my_class)
    visu_rendering_get_type();
  
  if (!dataObj || !my_class->current)
    return;

  visuDataIter_new(dataObj, &iter);
  for(visuDataIter_start(dataObj, &iter); iter.element;
      visuDataIter_nextElement(dataObj, &iter))
    {
      visuElement_createMaterial(iter.element);
      my_class->current->createElement(dataObj, iter.element);
      if (visuElementGet_updateNodesOnMaterialChange())
	visuData_createNodes(dataObj, iter.element);
    }
}

/**************/
/* Parameters */
/**************/

static gboolean readFavVisuRendering(gchar **lines, int nbLines, int position,
				       VisuData *dataObj _U_, GError **error)
{
  int err;

  g_return_val_if_fail(nbLines == 1, FALSE);

  lines[0] = g_strstrip(lines[0]);

  if (!lines[0][0])
    visuRenderingClassSet_current((VisuRendering*)0);
  else
    {
      err = visuRenderingClassSet_currentByName(lines[0]);
      if (!err)
	{
	  *error = g_error_new(CONFIG_FILE_ERROR, CONFIG_FILE_ERROR_VALUE,
			       _("Parse error at line %d:"
				 " the specified method (%s) is unknown.\n"),
			       position, lines[0]);
	  return FALSE;
	}
    }

  return TRUE;
}
static void exportFavVisuRendering(GString *data, VisuData *dataObj _U_)
{
  if (!my_class)
    visu_rendering_get_type();
  
  g_string_append_printf(data, "# %s\n", DESC_FAV_RENDER);
  if (my_class->current)
    g_string_append_printf(data, "%s: %s\n\n", FLAG_FAV_RENDER,
			   my_class->current->name);
  else
    g_string_append_printf(data, "%s:\n\n", FLAG_FAV_RENDER);
}
