/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#include "panelConfig.h"

#include <visu_gtk.h>
#include <visu_tools.h>
#include <visu_object.h>
#include <visu_basic.h>
#include <visu_data.h>
#include <visu_configFile.h>
#include <extraGtkFunctions/gtk_numericalEntryWidget.h>
#include <extraGtkFunctions/gtk_toolPanelWidget.h>
#include <renderingBackend/visu_windowInterface.h>
#include <coreTools/toolColor.h>
#include <coreTools/toolConfigFile.h>

#include <sys/stat.h>


#define FLAG_PARAMETER_TABVIEW_CONFIG   "config_subPanelTabView"
#define FLAG_PARAMETER_SKIN_CONFIG      "config_skin"
#define FLAG_PARAMETER_REFRESH_CONFIG   "config_refreshIsOn"
#define FLAG_PARAMETER_PERIOD_CONFIG    "config_refreshPeriod"
#define DESC_PARAMETER_TABVIEW_CONFIG   "See or not the labels on tabs ; boolean 0 or 1"
#define DESC_PARAMETER_SKIN_CONFIG      "Path to a gtkrc file ; chain"
#define DESC_PARAMETER_REFRESH_CONFIG   "When on V_Sim reloads the file at periodic time ; boolean 0 or 1"
#define DESC_PARAMETER_PERIOD_CONFIG    "The period of reloading in ms ; integer (10 < v < 10000)"
#define PARAMETER_CONFIG_TABVIEW_DEFAULT   0
#define PARAMETER_CONFIG_SKIN_DEFAULT      "None"
#define PARAMETER_CONFIG_REFRESH_DEFAULT   0
#define PARAMETER_CONFIG_PERIOD_DEFAULT    250
static int config_subPanelTabView;
static gchar *config_skin;
static int config_refreshIsOn;
static float config_refreshPeriod;
static GtkWidget *comboUnit;


/* Private functions. */
GtkWidget *createInteriorConfig();
static void setSkin(char* label);
void setAutomaticRefresh(int bool);
void setRefreshPeriod(float val);
static gboolean automaticReload(gpointer data);

/* This function details how to read the preferedNumberOfViews
   parameter. */
static gboolean readConfigTabView(gchar **lines, int nbLines, int position,
				  VisuData *dataObj, GError **error);
static gboolean readConfigSkin(gchar **lines, int nbLines, int position,
			       VisuData *dataObj, GError **error);
static gboolean readConfigRefresh(gchar **lines, int nbLines, int position,
				  VisuData *dataObj, GError **error);
static gboolean readConfigPeriod(gchar **lines, int nbLines, int position,
				 VisuData *dataObj, GError **error);
/* These functions write all the element list to export there associated resources. */
static void exportParametersPanelConfig(GString *data, VisuData* dataObj);
static void onUnitChanged(GtkComboBox *combo, gpointer data);

static void initPanelConfigGtkPart();

static char *defaultSkinPath, *userSkinPath;

gboolean isPanelConfigInitialized;

/* Specific widgets used in this panel. */
static GtkWidget *visuConfigData;
static GtkWidget *checkShowTab;
static GtkWidget *checkRefreshAuto;
static GtkWidget *checkStorePositions;

/* Callbacks */
static void checkShowTabToggled(GtkToggleButton *button, gpointer data);
void checkRememberToggled(GtkToggleButton *button, gpointer data);
void entryRcFilesChanged(GtkEntry *entry, gpointer data);
void directorySelectedForResources(GtkButton *button, gpointer user_data);
void openRcFileSelector(GtkButton *button, gpointer data);
void checkRefreshToggled(GtkToggleButton *button, gpointer data);
static void onRefreshPeriodChanged(GtkSpinButton* button, gpointer data);
void onDataReadySignal(GObject *obj, VisuData *dataObj, gpointer data);
gboolean treePathClicked(GtkWidget *widget, GdkEventButton *event, gpointer user_data);
void directoryRemoveFromResources(GtkButton *button, gpointer user_data);
static void onConfigEnter(ToolPanel *toolPanel, gpointer data);

ToolPanel* panelConfigInit()
{
  char *cl = _("Configure the interface");
  char *tl = _("Configuration");
  VisuConfigFileEntry *resourceEntry;

  resourceEntry = visuConfigFileAdd_entry(VISU_CONFIGFILE_PARAMETER,
					  FLAG_PARAMETER_TABVIEW_CONFIG,
					  DESC_PARAMETER_TABVIEW_CONFIG,
					  1, readConfigTabView);
  resourceEntry = visuConfigFileAdd_entry(VISU_CONFIGFILE_PARAMETER,
					  FLAG_PARAMETER_SKIN_CONFIG,
					  DESC_PARAMETER_SKIN_CONFIG,
					  1, readConfigSkin);
  resourceEntry = visuConfigFileAdd_entry(VISU_CONFIGFILE_PARAMETER,
					  FLAG_PARAMETER_REFRESH_CONFIG,
					  DESC_PARAMETER_REFRESH_CONFIG,
					  1, readConfigRefresh);
  resourceEntry = visuConfigFileAdd_entry(VISU_CONFIGFILE_PARAMETER,
					  FLAG_PARAMETER_PERIOD_CONFIG,
					  DESC_PARAMETER_PERIOD_CONFIG,
					  1, readConfigPeriod);
  visuConfigFileAdd_exportFunction(VISU_CONFIGFILE_PARAMETER,
				   exportParametersPanelConfig);

  defaultSkinPath = g_build_filename(visuBasicGet_dataDir(), "v_sim.rc", NULL);
  userSkinPath    = g_build_filename(visuBasicGet_localDir(), "v_sim.rc", NULL);
  config_subPanelTabView = PARAMETER_CONFIG_TABVIEW_DEFAULT;
  config_refreshPeriod = -1;
  config_refreshIsOn = -1;
  panelConfigSet_refreshPeriod((RenderingWindow*)0, PARAMETER_CONFIG_PERIOD_DEFAULT);
  panelConfigSet_automaticRefresh((RenderingWindow*)0, PARAMETER_CONFIG_REFRESH_DEFAULT);

  /* Now that everything has a value, we create the panel. */
  visuConfigData = toolPanelNew_withIconFromStock("Panel_configuration",
						  cl, tl, GTK_STOCK_PREFERENCES);
  if (!visuConfigData)
    return (ToolPanel*)0;

  isPanelConfigInitialized = FALSE;

  /* Set global callbacks. */
  g_signal_connect(VISU_INSTANCE, "dataReadyForRendering",
		   G_CALLBACK(onDataReadySignal), (gpointer)0);
  g_signal_connect(G_OBJECT(visuConfigData), "page-entered",
		   G_CALLBACK(onConfigEnter), (gpointer)0);

  return TOOL_PANEL(visuConfigData);
}

static void initPanelConfigGtkPart()
{
  GtkWidget *wd;

  if (isPanelConfigInitialized)
    return;

  DBG_fprintf(stderr, "Panel Config : creating the config panel on demand.\n");
  isPanelConfigInitialized = TRUE;

  wd = createInteriorConfig();
  gtk_container_add(GTK_CONTAINER(visuConfigData), wd);

  /* Force the callbacks to initialise the values. */
  checkShowTabToggled((GtkToggleButton*)0, (gpointer)0);
}


GtkWidget *createInteriorConfig()
{
  GtkWidget *vbox, *vbox2;
  GtkWidget *hbox;
  GtkWidget *label;
  GtkWidget *table;
  GtkWidget *align;
  GtkObject *adj;
  GtkWidget *scrollView;
  GtkWidget *spinRefresh;
  const gchar **units;
  int i;
 
#if GTK_MINOR_VERSION < 12 
  GtkTooltips *tooltips;

  tooltips = gtk_tooltips_new ();
#endif

  scrollView = gtk_scrolled_window_new((GtkAdjustment*)0,
				       (GtkAdjustment*)0);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrollView),
				  GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
  gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (scrollView), GTK_SHADOW_NONE);

  vbox = gtk_vbox_new(FALSE, 0);
  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scrollView), vbox);

  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 5);
  label = gtk_label_new(_("Always show labels in tabs: "));
  gtk_widget_set_name(label, "label_head");
  gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 2);
  checkShowTab = gtk_check_button_new();
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkShowTab),
			       (gboolean)config_subPanelTabView);
  gtk_box_pack_start(GTK_BOX(hbox), checkShowTab, FALSE, FALSE, 2);

  table = gtk_table_new(2, 1, FALSE);
  gtk_box_pack_start(GTK_BOX(vbox), table, FALSE, FALSE, 5);
  align = gtk_alignment_new(0., 0.5, 0., 0.);
  gtk_table_attach(GTK_TABLE(table), align, 0, 1, 0, 1, GTK_FILL | GTK_EXPAND, GTK_SHRINK, 2, 0);
  hbox = gtk_hbox_new(FALSE, 0);
  gtk_container_add(GTK_CONTAINER(align), hbox);
  label = gtk_label_new(_("Automatic refresh: "));
  gtk_widget_set_name(label, "label_head");
  gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
  checkRefreshAuto = gtk_check_button_new();
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkRefreshAuto),
			       (gboolean)config_refreshIsOn);
  gtk_box_pack_start(GTK_BOX(hbox), checkRefreshAuto, FALSE, FALSE, 0);
  align = gtk_alignment_new(1., 0.5, 0., 0.);
  gtk_table_attach(GTK_TABLE(table), align, 0, 1, 1, 2, GTK_FILL | GTK_EXPAND, GTK_SHRINK, 2, 0);
  hbox = gtk_hbox_new(FALSE, 0);
  gtk_container_add(GTK_CONTAINER(align), hbox);
  label = gtk_label_new(_("Refresh period: "));
  gtk_widget_set_name(label, "label_head");
  gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
  adj = gtk_adjustment_new(100., 10., 10000., 10., 100., 0.);
  spinRefresh = gtk_spin_button_new(GTK_ADJUSTMENT(adj), 0.05, 0);
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(spinRefresh),
			    config_refreshPeriod);
  gtk_box_pack_start(GTK_BOX(hbox), spinRefresh, FALSE, FALSE, 0);
  label = gtk_label_new(_(" ms"));
  gtk_widget_set_name(label, "label_head");
  gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);

  vbox2 = gtk_vbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox), vbox2, FALSE, FALSE, 5);
  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox2), hbox, FALSE, FALSE, 0);
  label = gtk_label_new(_("Remember windows positions:"));
  gtk_widget_set_name(label, "label_head");
  gtk_misc_set_alignment(GTK_MISC(label), 0., 0.5);
  gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 2);
  checkStorePositions = gtk_check_button_new();
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkStorePositions),
			       (gboolean)TRUE);
  gtk_box_pack_start(GTK_BOX(hbox), checkStorePositions, FALSE, FALSE, 2);
  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox2), hbox, FALSE, FALSE, 0);
  label = gtk_label_new(_("(because I've a bad WM)"));
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
  gtk_box_pack_start(GTK_BOX(hbox), label, TRUE, TRUE, 1);

  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
  label = gtk_label_new(_("Set the prefered unit:"));
  gtk_widget_set_name(label, "label_head");
  gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 5);

  comboUnit = gtk_combo_box_new_text();
  units = visuToolsGet_unitNames();
  for (i = 0; units[i]; i++)
    gtk_combo_box_append_text(GTK_COMBO_BOX(comboUnit), units[i]);
  gtk_combo_box_set_active(GTK_COMBO_BOX(comboUnit), visuBasicGet_preferedUnit());
  g_signal_connect(G_OBJECT(comboUnit), "changed",
		   G_CALLBACK(onUnitChanged), (gpointer)0);
  gtk_box_pack_start(GTK_BOX(hbox), comboUnit, FALSE, FALSE, 0);

  gtk_widget_show_all(scrollView);

  /* Set the callbacks. */
  g_signal_connect(G_OBJECT(checkShowTab), "toggled",
		   G_CALLBACK(checkShowTabToggled), (gpointer)0);
  g_signal_connect(G_OBJECT(checkStorePositions), "toggled",
		   G_CALLBACK(checkRememberToggled), (gpointer)0);
  g_signal_connect(G_OBJECT(checkRefreshAuto), "toggled",
		   G_CALLBACK(checkRefreshToggled), (gpointer)0);
  g_signal_connect(G_OBJECT(spinRefresh), "value-changed",
		   G_CALLBACK(onRefreshPeriodChanged), (gpointer)0);
  
  
  return scrollView;
}

/*************/
/* Callbacks */
/*************/
static void checkShowTabToggled(GtkToggleButton *button _U_, gpointer data _U_)
{
  panelConfigSet_tabView(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(checkShowTab)));
}
void checkRememberToggled(GtkToggleButton *button, gpointer data _U_)
{
  gtkMainClassSet_rememberPosition(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(button)));
}
static void onConfigEnter(ToolPanel *toolPanel _U_, gpointer data _U_)
{
  if (!isPanelConfigInitialized)
    initPanelConfigGtkPart();

  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkStorePositions),
			       gtkMainClassGet_rememberPosition());
}
void checkRefreshToggled(GtkToggleButton *button, gpointer data _U_)
{
  RenderingWindow *window;

  window = RENDERING_WINDOW(visuRenderingWindowGet_current());
  g_return_if_fail(window);

  panelConfigSet_automaticRefresh(window, gtk_toggle_button_get_active(button));
}
static void onRefreshPeriodChanged(GtkSpinButton* button, gpointer data _U_)
{
  RenderingWindow *window;

  window = RENDERING_WINDOW(visuRenderingWindowGet_current());
  g_return_if_fail(window);

  panelConfigSet_refreshPeriod(window, (float)gtk_spin_button_get_value(button));
}
void openRcFileSelector(GtkButton *button _U_, gpointer data _U_)
{
}
static void setSkin(char* label)
{
  GtkSettings *settings;
  char *tmp;

  if (!label || label[0] == '\0' || !strcmp(label, "None"))
    return;

  if (!strcmp(label, "V_Sim"))
    {
      /* We try first a skin in the user config path. */
      if (g_file_test(userSkinPath, G_FILE_TEST_EXISTS | G_FILE_TEST_IS_REGULAR))
	tmp = userSkinPath;
      else
	tmp = defaultSkinPath;
    }
  else
    tmp = label;

  /* test if the file exists or fall back on default installation path */
  DBG_fprintf(stderr,"Panel Config: Reading rc file '%s' ... %d\n",
	      tmp, g_file_test(tmp, G_FILE_TEST_EXISTS | G_FILE_TEST_IS_REGULAR));
  if (!g_file_test(tmp, G_FILE_TEST_EXISTS | G_FILE_TEST_IS_REGULAR))
    return;
  gtk_rc_parse(tmp);

  settings = gtk_settings_get_default ();
  gtk_rc_reparse_all_for_settings(settings, TRUE);
  DBG_fprintf(stderr,"Panel Config: applying RC file OK.\n");
}
void autoReloadAddTimeout(VisuData *data, guint time, gpointer user_data)
{
  guint *timeoutId;

  g_return_if_fail(data);
  
  timeoutId = (guint*)g_object_get_data(G_OBJECT(data), "autoLoad_timeoutId");
  if (!timeoutId)
    {
      timeoutId = g_malloc(sizeof(guint));
      g_object_set_data_full(G_OBJECT(data), "autoLoad_timeoutId",
			     (gpointer)timeoutId, g_free);
      *timeoutId = 0;
    }
  if (!*timeoutId)
    *timeoutId = visuDataAdd_timeout(data, time, automaticReload, user_data);
}
void autoReloadRemoveTimeout(VisuData *data)
{
  guint *timeoutId;

  g_return_if_fail(data);
  
  timeoutId = (guint*)g_object_get_data(G_OBJECT(data), "autoLoad_timeoutId");
  if (timeoutId)
    {
      visuDataRemove_timeout(data, *timeoutId);
      *timeoutId = 0;
    }
}
static gboolean automaticReload(gpointer data)
{
  struct stat statBuf;
  int res, iSet;
  char *file;
  time_t modTime;
  int kind, nbKind;
  gboolean reloadNeeded, ok;
  GenericRenderingWindow window;
  VisuData *dataObj;
  time_t *lastReadTime;
  GError *error;

  DBG_fprintf(stderr, "Panel Config : automatic reload called.\n");
  window = RENDERING_WINDOW(data);
  g_return_val_if_fail(window, TRUE);

  dataObj = renderingWindowGet_visuData(window);
  if (!dataObj)
    {
      DBG_fprintf(stderr, "Panel Config : automatic reload aborted, no file available.\n");
      return TRUE;
    }
  
  lastReadTime = (time_t*)g_object_get_data(G_OBJECT(dataObj), "autoLoad_accessTime");
  g_return_val_if_fail(lastReadTime, TRUE);

  reloadNeeded = FALSE;
  nbKind = visuRenderingGet_nbFileType(visuRenderingClassGet_current());
  for (kind = 0; kind < nbKind; kind++)
    {
      file = visuDataGet_file(dataObj, kind, (FileFormat**)0);
      if (!file)
	return TRUE;

      res = stat(file, &statBuf);
      if (!res)
	{
	  modTime = statBuf.st_ctime;
	  if (modTime > lastReadTime[kind])
	    {
	      reloadNeeded = TRUE;
	      lastReadTime[kind] = modTime;
	    }
	}
    }
  if (reloadNeeded)
    {
      DBG_fprintf(stderr, "Panel Config : automatic reloading in progress...\n");
      iSet = visuDataGet_setId(dataObj);
      visuDataFree_population(dataObj);
      error = (GError*)0;
      ok = visuBasicLoad_dataFromFile(dataObj, (FileFormat*)0, iSet, &error);
      if (!ok)
	{
	  visuGtkRaise_warning(_("Automatic file reloading"), error->message,
			       (GtkWindow*)0);
	  g_error_free(error);
	  visuRenderingWindowSet_visuData(window, (VisuData*)0);
	}
      else
	{
	  visuRenderingWindowSet_visuData(window, dataObj);
	  visuData_createAllNodes(dataObj);
	}
      visuObjectRedraw((gpointer)0);
    }
  return TRUE;
}
void onDataReadySignal(GObject *obj _U_, VisuData *dataObj, gpointer data _U_)
{
  struct stat statBuf;
  int res;
  char *file;
  int kind, nbKind;
  time_t *lastReadTime;

  if (!dataObj)
    return;

  DBG_fprintf(stderr, "Panel Config: caught the 'dataReadyForRendering' signal,"
	      " set acces time for VisuData %p.\n", (gpointer)dataObj);
  nbKind = visuRenderingGet_nbFileType(visuRenderingClassGet_current());
  lastReadTime = g_malloc(sizeof(time_t) * nbKind);
  for (kind = 0; kind < nbKind; kind++)
    {
      file = visuDataGet_file(dataObj, kind, (FileFormat**)0);
      if (!file)
	lastReadTime[kind] = 0;
      else
	{
	  res = stat(file, &statBuf);
	  if (!res)
	    lastReadTime[kind] = statBuf.st_ctime;
	  else
	    lastReadTime[kind] = 0;
	}
    }
  g_object_set_data_full(G_OBJECT(dataObj), "autoLoad_accessTime",
			 lastReadTime, g_free);
  if (config_refreshIsOn)
    autoReloadAddTimeout(dataObj, config_refreshPeriod,
			 (gpointer)visuDataGet_renderingWindow(dataObj));
}
static void onUnitChanged(GtkComboBox *combo, gpointer data _U_)
{
  visuBasicSet_preferedUnit(gtk_combo_box_get_active(combo));
}



/**********************************/
/* Public method to change values */
/**********************************/

void panelConfigSet_tabView(gboolean viewed)
{
  DBG_fprintf(stderr,"Panel Config : toggle tab view -> %d (previously %d)\n",
	      (int)viewed, config_subPanelTabView);
  if (viewed == config_subPanelTabView)
    return;

  toolPanelClassSet_headerVisibility(viewed);
  
  /* set the associated parameter. */
  config_subPanelTabView = viewed;
}
void panelConfigSet_automaticRefresh(RenderingWindow *window, int bool)
{
  DBG_fprintf(stderr, "Panel Config : set refresh to %d (previously %d).\n",
	      bool, config_refreshIsOn);

  if (config_refreshIsOn == bool)
    return;

  config_refreshIsOn = bool;
  if (window && renderingWindowGet_visuData(window))
    {
      if (bool)
	autoReloadAddTimeout(renderingWindowGet_visuData(window),
			     config_refreshPeriod, (gpointer)window);
      else
	autoReloadRemoveTimeout(renderingWindowGet_visuData(window));
    }
}
void panelConfigSet_refreshPeriod(RenderingWindow *window, float val)
{
  VisuData *dataObj;

  DBG_fprintf(stderr, "Panel Config : set the refresh period to %f (previously %f).\n",
	      val, config_refreshPeriod);

  if (config_refreshPeriod == val)
    return;

  config_refreshPeriod = val;
  dataObj = (window)?renderingWindowGet_visuData(window):(VisuData*)0;
  if (dataObj && config_refreshIsOn)
    {
      DBG_fprintf(stderr, "Panel Config : stopping the callback of refresh "
		  "to change the period.\n");
      autoReloadRemoveTimeout(dataObj);
      autoReloadAddTimeout(dataObj, config_refreshPeriod, (gpointer)window);
    }
}


/********************/
/* In/Out functions */
/********************/

/* This function details how to read the preferedNumberOfViews
   parameter. */
static gboolean readConfigTabView(gchar **lines, int nbLines, int position,
				  VisuData *dataObj _U_, GError **error)
{
  gboolean viewed;

  g_return_val_if_fail(nbLines == 1, FALSE);

  if (!configFileRead_boolean(lines[0], position, &viewed, 1, error))
    return FALSE;
  panelConfigSet_tabView(viewed);
  
  return TRUE;
}
static gboolean readConfigSkin(gchar **lines, int nbLines, int position,
			       VisuData *dataObj _U_, GError **error)
{
  g_return_val_if_fail(nbLines == 1, FALSE);

  lines[0] = g_strstrip(lines[0]);

  if (!lines[0][0])
    {
      *error = g_error_new(CONFIG_FILE_ERROR, CONFIG_FILE_ERROR_VALUE,
			   _("Parse error at line %d: 1 string value must appear"
			     " after the %s markup.\n"), position,
			   FLAG_PARAMETER_SKIN_CONFIG);
      config_skin = g_strdup(PARAMETER_CONFIG_SKIN_DEFAULT);
      setSkin(config_skin);
      return FALSE;
    }
  config_skin = g_strdup(lines[0]);
  
  setSkin(config_skin);

  return TRUE;
}
static gboolean readConfigRefresh(gchar **lines, int nbLines, int position,
				  VisuData *dataObj _U_, GError **error)
{
  gboolean bool;
  RenderingWindow *window;

  g_return_val_if_fail(nbLines == 1, FALSE);

  window = RENDERING_WINDOW(visuRenderingWindowGet_current());
  if (!configFileRead_boolean(lines[0], position, &bool, 1, error))
    return FALSE;
  panelConfigSet_automaticRefresh(window, bool);

  return TRUE;
}
static gboolean readConfigPeriod(gchar **lines, int nbLines, int position,
				 VisuData *dataObj _U_, GError **error)
{
  float val;
  RenderingWindow *window;

  g_return_val_if_fail(nbLines == 1, FALSE);

  window = RENDERING_WINDOW(visuRenderingWindowGet_current());
  if (!configFileRead_float(lines[0], position, &val, 1, error))
    return FALSE;
  config_refreshPeriod = val;
  panelConfigSet_refreshPeriod(window, config_refreshPeriod);

  return TRUE;
}

/* These functions write all the element list to export there associated resources. */
static void exportParametersPanelConfig(GString *data, VisuData *dataObj _U_)
{
  g_string_append_printf(data, "# %s\n", DESC_PARAMETER_TABVIEW_CONFIG);
  g_string_append_printf(data, "%s[gtk]: %d\n\n", FLAG_PARAMETER_TABVIEW_CONFIG,
			 config_subPanelTabView);
  g_string_append_printf(data, "# %s\n", DESC_PARAMETER_SKIN_CONFIG);
  g_string_append_printf(data, "%s[gtk]: %s\n\n", FLAG_PARAMETER_SKIN_CONFIG,
			 config_skin);
  g_string_append_printf(data, "# %s\n", DESC_PARAMETER_REFRESH_CONFIG);
  g_string_append_printf(data, "%s[gtk]: %d\n\n", FLAG_PARAMETER_REFRESH_CONFIG,
			 config_refreshIsOn);
  g_string_append_printf(data, "# %s\n", DESC_PARAMETER_PERIOD_CONFIG);
  g_string_append_printf(data, "%s[gtk]: %i\n\n", FLAG_PARAMETER_PERIOD_CONFIG,
			 (int)config_refreshPeriod);
}

