/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Damien CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Damien CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail addresses :
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/

#include "surfaces_points.h"

#include <string.h>

#include <visu_tools.h>

void isosurfacesPointsCheck(SurfacesPoints *points)
{
  int i, j;
  int *nbPolys;

  /* Check the surface index. */
  DBG_fprintf(stderr, " | check surface index for each polygons.\n");
  for (i = 0; i < points->num_polys; i++)
    g_return_if_fail(ABS(points->poly_surf_index[i]) > 0 &&
		     ABS(points->poly_surf_index[i]) <= points->nsurf);

  /* Check vertice index. */
  DBG_fprintf(stderr, " | check vertice index for each polygons.\n");
  for (i = 0; i < points->num_polys; i++)
    for (j = 0; j < points->poly_num_vertices[i]; j++)
      g_return_if_fail(points->poly_vertices[i][j] >= 0 &&
		       points->poly_vertices[i][j] < points->num_points);

  /* Check the number of polygons. */
  nbPolys = g_malloc(sizeof(int) * points->nsurf);
  memset(nbPolys, 0, sizeof(int) * points->nsurf);
  for (i = 0; i < points->num_polys; i++)
    if (points->poly_surf_index[i] > 0)
      nbPolys[points->poly_surf_index[i] - 1] += 1;
  for (i = 0; i < points->nsurf; i++)
    {
      DBG_fprintf(stderr, " | %d counted %7d : stored %7d\n", i, 
		  nbPolys[i], points->num_polys_surf[i]);
      g_return_if_fail(nbPolys[i] == points->num_polys_surf[i]);
    }
  g_free(nbPolys);
}

void isosurfacesPointsTranslate(SurfacesPoints *points, float xyz[3])
{
  gboolean *verticeStatus, visibility, boundary, redraw;
  int i, j;
  int *boundaryPolys, nBounadryPolys;

  g_return_if_fail(points);

  DBG_fprintf(stderr, "IsosurfacesPoints: translate points %p of %gx%gx%g.\n",
	      (gpointer)points, xyz[0], xyz[1], xyz[2]);

  /* As for surface hide: translate and reput in the box
     except if the triangle has some points that have been
     reput in the box. */
  verticeStatus = g_malloc(sizeof(gboolean) * points->num_points);
  /* Apply the translation and compute a translation
     flag for each vertice. */
  for (i = 0; i < points->num_points; i++)
    {
      /* Translations are given in cartesian coordinates. */
/*       points->poly_points_data[j] += xyz[j]; */
      
/*       verticeStatus[i] = planesGet_visibility(planes, surf->basePoints.poly_points_data[i]); */
    }
  /* We store the id of boundary polygons. */
  boundaryPolys = g_malloc(sizeof(int) * points->num_polys);
  nBounadryPolys = 0;
  /* Hide polygons. */
  for (i = 0; i < points->num_polys; i++)
    {
      visibility = TRUE;
      boundary = FALSE;
/*       if (surf->resources[ABS(points->poly_surf_index[i]) - 1]->sensitiveToMaskingPlanes) */
	{
	  for (j = 0; j < points->poly_num_vertices[i]; j++)
	    {
	      visibility = visibility && verticeStatus[points->poly_vertices[i][j]];
	      boundary = boundary || verticeStatus[points->poly_vertices[i][j]];
	    }
	  boundary = !visibility && boundary;
	}
      if (!visibility && points->poly_surf_index[i] > 0)
	{
	  /* Hide this polygon. */
	  points->num_polys_surf[points->poly_surf_index[i] - 1] -= 1;
	  points->poly_surf_index[i] = -points->poly_surf_index[i];
	  redraw = TRUE;
	}
      else if (visibility && points->poly_surf_index[i] < 0)
	{
	  /* Show this polygon. */
	  points->poly_surf_index[i] = -points->poly_surf_index[i];
	  points->num_polys_surf[points->poly_surf_index[i] - 1] += 1;
	  redraw = TRUE;
	}
      if (boundary)
	boundaryPolys[nBounadryPolys++] = i;
    }
/*   if (DEBUG) */
/*     for (i = 0; i < surf->nsurf; i++) */
/*       fprintf(stderr, , " | surface %2d -> %7d polygons\n", i, points->num_polys_surf[i]); */
  /* We count the number of boundaries per surface and allocate
     accordingly the volatile. */

}
void isosurfacesPointsRemove(SurfacesPoints *points, int pos)
{
  int nPoly, nPoint;
  int i, j, iPoly, iPoint;
  SurfacesPoints tmpPoints;
  gboolean *usedPoints;
  int *switchArray;

  points->nsurf -= 1;

  if (!points->num_points)
    {
      points->num_polys_surf = g_realloc(points->num_polys_surf,
					 points->nsurf * sizeof(int));
      return;
    }

  DBG_fprintf(stderr, "IsosurfacesPoints: remove surface %d from points %p.\n",
	  pos, (gpointer)points);
  g_return_if_fail(pos >= 0 && pos <= points->nsurf);
  
  /* Special case when there is only one remaining surface. */
  if (points->nsurf == 0)
    {
      isosurfacesPointsFree(points);
      return;
    }

  /* Simple implementation is to create a new SurfacesPoints object arrays, and to
     copy everything, except the polygons belonging to the given pos. */

  /* Count number of poly and points to remove. */
  usedPoints = g_malloc(sizeof(gboolean) * points->num_points);
  memset(usedPoints, 0, sizeof(gboolean) * points->num_points);
  /* We don't use num_polys_surf since it is restricted to visible surfaces. */
  nPoly = 0;
  for (i = 0; i < points->num_polys; i++)
    if (ABS(points->poly_surf_index[i]) - 1 != pos)
      {
	nPoly += 1;
	for (j = 0; j < points->poly_num_vertices[i]; j++)
	  usedPoints[points->poly_vertices[i][j]] = TRUE;
      }
  nPoint = 0;
  for (i = 0; i < points->num_points; i++)
    if (usedPoints[i])
      nPoint += 1;
  DBG_fprintf(stderr, " | remove %d polygons and %d points.\n",
	      points->num_polys - nPoly,
	      points->num_points - nPoint);

  isosurfacesPointsInit(&tmpPoints, points->bufferSize);
  isosurfacesPointsAllocate(&tmpPoints, points->nsurf, nPoly, nPoint);

  /* Copy from surf to tmpSurf. */
  switchArray = g_malloc(sizeof(int) * points->num_points);
  iPoint = 0;
  for (i = 0; i < points->num_points; i++)
    if (usedPoints[i])
      {
	memcpy(tmpPoints.poly_points_data[iPoint],
	       points->poly_points_data[i], sizeof(float) *
	       (SurfacesPoints_userOffset + points->bufferSize));
	switchArray[i] = iPoint;
	iPoint += 1;
	if (iPoint > nPoint)
	  {
	    g_error("Incorrect point checksum.");
	  }
      }
  iPoly = 0;
  for (i = 0; i < points->num_polys; i++)
    {
      if (ABS(points->poly_surf_index[i]) - 1 != pos)
	{
	  if (points->poly_surf_index[i] > pos + 1)
	    tmpPoints.poly_surf_index[iPoly] = points->poly_surf_index[i] - 1;
	  else if (points->poly_surf_index[i] < -pos - 1)
	    tmpPoints.poly_surf_index[iPoly] = points->poly_surf_index[i] + 1;
	  else
	    tmpPoints.poly_surf_index[iPoly] = points->poly_surf_index[i];
	  tmpPoints.poly_num_vertices[iPoly] = points->poly_num_vertices[i];
	  tmpPoints.poly_vertices[iPoly]     =
	    g_malloc(sizeof(int) * tmpPoints.poly_num_vertices[iPoly]);
	  for (j = 0; j < tmpPoints.poly_num_vertices[iPoly]; j++)
	    tmpPoints.poly_vertices[iPoly][j] =
	      switchArray[points->poly_vertices[i][j]];
	  iPoly += 1;
	  if (iPoly > nPoly)
	    {
	      g_error("Incorrect polygon checksum.");
	    }
	}
    }
  g_free(usedPoints);
  g_free(switchArray);
  /* Check sum. */
  if (iPoly != nPoly || iPoint != nPoint)
    {
      g_error("Incorrect checksum (%d %d | %d %d).", iPoly, nPoly, iPoint, nPoint);
    }

  /* Move the number of polygons per surface. */
  for (i = pos; i < points->nsurf; i++)
    points->num_polys_surf[i] = points->num_polys_surf[i + 1];
  points->num_polys_surf = g_realloc(points->num_polys_surf,
				     sizeof(int) * points->nsurf);
  g_free(tmpPoints.num_polys_surf);

  /* We replace the arrays between tmpSurf and surf. */
  DBG_fprintf(stderr, " | switch and free arrays.\n");
  g_free(points->poly_surf_index);
  points->poly_surf_index = tmpPoints.poly_surf_index;

  g_free(points->poly_num_vertices);
  points->poly_num_vertices = tmpPoints.poly_num_vertices;

  for (i = 0; i< points->num_polys; i++)
    g_free(points->poly_vertices[i]);
  g_free(points->poly_vertices);
  points->poly_vertices = tmpPoints.poly_vertices;

  g_free(points->poly_points_data[0]);
  g_free(points->poly_points_data);
  points->poly_points_data = tmpPoints.poly_points_data;

  /* additionnal tuning. */
  points->num_polys  = nPoly;
  points->num_points = nPoint;
}
void isosurfacesPointsFree(SurfacesPoints *points)
{
  int i;

  if(points->num_polys == 0)
    return;

  if (points->num_polys_surf)
    g_free(points->num_polys_surf);
  if (points->poly_surf_index)
    g_free(points->poly_surf_index);
  if (points->poly_num_vertices)
    g_free(points->poly_num_vertices);
  if (points->poly_vertices)
    {
      for (i = 0; i < points->num_polys; i++)
	g_free(points->poly_vertices[i]);
      g_free(points->poly_vertices);
    }
  if (points->poly_points_data)
    {
      g_free(points->poly_points_data[0]);
      g_free(points->poly_points_data);
    }

  points->nsurf      = 0;
  points->num_polys  = 0;
  points->num_points = 0;
  points->num_polys_surf = (int*)0;
  points->poly_surf_index = (int*)0;
  points->poly_num_vertices = (int*)0;
  points->poly_vertices = (int**)0;
  points->poly_points_data = (float **)0;
}
void isosurfacesPointsInit(SurfacesPoints *points, int bufferSize)
{
  g_return_if_fail(bufferSize >= 0 && points);
  DBG_fprintf(stderr, "IsosurfacesPoints: initialise point definitions (%p-%d).\n",
	  (gpointer)points, bufferSize);

  points->nsurf      = 0;
  points->num_polys  = 0;
  points->num_points = 0;
  points->bufferSize = bufferSize;

  points->num_polys_surf = (int*)0;
  points->poly_surf_index = (int*)0;
  points->poly_num_vertices = (int*)0;
  points->poly_vertices = (int**)0;
  points->poly_points_data = (float **)0;
}
void isosurfacesPointsAllocate(SurfacesPoints *points, int nsurf,
			       int npolys, int npoints)
{
  int i;

  g_return_if_fail(nsurf > 0 && npolys >= 0 && npoints >= 0);
  DBG_fprintf(stderr, "IsosurfacesPoints: allocate point definitions (%d-%d).\n",
	  npolys, npoints);

  points->nsurf      = nsurf;
  points->num_polys  = npolys;
  points->num_points = npoints;

  points->num_polys_surf = g_malloc(nsurf * sizeof(int));
  memset(points->num_polys_surf, 0, sizeof(int) * nsurf);

  if (npolys == 0 || npoints == 0)
    {
      points->poly_surf_index = (int*)0;
      points->poly_num_vertices = (int*)0;
      points->poly_vertices = (int**)0;
      points->poly_points_data = (float **)0;
      return;
    }

  /* for each of the num_polys polygons will contain the surf value
     to which it belongs */
  points->poly_surf_index = g_malloc(npolys * sizeof(int));
  /* for each of the num_polys polygons will contain the number
     of vertices in it */
  points->poly_num_vertices = g_malloc(npolys * sizeof(int));
  /* for each of the num_polys polygons will contain the indices
     of vertices in it */   
  points->poly_vertices = g_malloc(npolys * sizeof(int*));
  memset(points->poly_vertices, 0, npolys * sizeof(int*));

  /* for each of the num_points,
     poly_points_data[][0] = x
     poly_points_data[][1] = y
     poly_points_data[][2] = z
  */ 
  points->poly_points_data = g_malloc(npoints * sizeof(float *));
  points->poly_points_data[0] = g_malloc((SurfacesPoints_userOffset +
					  points->bufferSize) *
					 npoints * sizeof(float));
  for(i = 0; i < npoints; i++)
    points->poly_points_data[i] = points->poly_points_data[0] +
      i * (SurfacesPoints_userOffset + points->bufferSize);

  DBG_fprintf(stderr, " | OK.\n");
}
