/* $Id: ui.c,v 1.13 2002/10/30 19:27:09 bjk Exp $ */
/*
    Copyright (C) 2001-2002  Ben Kibbey <bjk@arbornet.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>
#include <ctype.h>
#include <grp.h>
#include <sys/mman.h>

#include "common.h"
#include "ui.h"

#ifdef HAVE_LIBGEN_H
#include <libgen.h>
#endif

static int escapes(const char *str)
{
    int c = 0;

    if (str[0] != '\\')
	return str[0];

    switch (*++str) {
	case 't':
	    c = '\t';
	    break;
	case 'n':
	    c = '\n';
	    break;
	case '\\':
	    c = '\\';
	    break;
	case 'v':
	    c = '\v';
	    break;
	case 'b':
	    c = '\b';
	    break;
	case 'f':
	    c = '\f';
	    break;
	case 'r':
	    c = '\r';
	    break;
	case '\'':
	    c = '\'';
	    break;
	default:
	    c = 0;
	    break;
    }

    return c;
}

static void usage(int ret)
{
    extern unsigned delimchar;

    printf("Usage: %s [-vhVF] [-j tf] [-D c] options "
	    "[user | -f filename] [...]\n", __progname);
    printf("  Password/Group file information [-P (-%s)]:\n",
	    MOPT_ORDER_PASSWD);
    printf("\t-l  login name\t\t");
    printf("\t-p  encrypted password\n");
    printf("\t-u  user id (uid)\t");
    printf("\t-g  group id (gid)\n");
    printf("\t-c  password change time");
    printf("\t-k  password expire time\n");
    printf("\t-d  home directory\t");
    printf("\t-m  home directory mode\n");
    printf("\t-s  login shell\n");
    printf("\t-iX gecos information "
	    "(X=[n]ame,[o]ffice,[w]phone,[h]phone),[a]ll)\n\n");

    printf("  Mail information [-M (-%s)]:\n", MOPT_ORDER_MAIL);
    printf("\t-o  forward address\t\t");
    printf("-a  mail aliases\n");
    printf("\t-r  folder access time\t\t");
    printf("-w  folder modification time\n");
    printf("\t-z  folder size\n\n");

    printf("  Login information [-L (-%s)]:\n", MOPT_ORDER_LOGIN);
    printf("\t-y  tty\t\t\t\t");
    printf("-e  message status\n");
    printf("\t-t  login time stamp\t\t");
    printf("-b  duration in minutes\n");
    printf("\t-x  hostname\t\t\t");
#if defined(HAVE_PROCFS) || defined(HAVE_LIBKVM)
    printf("-q  seconds idle\n");
    printf("\t-C  parent process id (ppid)\n");
#else
    printf("-q  seconds idle\n");
#endif
    printf("\t-nX lastlog information"
	    " (X=tt[y],[h]ostname,[t]ime,[a]ll)\n\n");

    printf("  -A\tOutput all options (-PML).\n");
    printf("  -D c\tSeparate output with specified character "
	    "('%c').\n", delimchar);
    printf("  -j tf\tstrftime(3) time format ('%s').\n",
	    DEF_TIMEFORMAT);
    printf("  -f\tUsers are owners of the following files.\n");
    printf("  -F\tFollow symbolic links.\n");
    printf("  -v\tVerbose output; displays gids with group names, "
	    "and filenames.\n");
    printf("  -h\tThis help text.\n");
    printf("  -V\tVersion information.\n\n");
    printf("Output key: %s=unknown/error, %s=none, %s=yes/on, "
	    "%s=no/off\n", UNKNOWN, NONE, ON, OFF);

    exit(ret);
}

static int get_lastlog_opts(const char *args)
{
    int i = 0;

    for (i = 0; i < strlen(args); i++) {
	switch (args[i]) {
	    case 'y':
	    case 'h':
	    case 't':
	    case 'a':
		break;
	    default:
		return 1;
	}
    }

    i = 0;

    if (strchr(args, 'a') != NULL) {
	args = "yht";
	get_lastlog_opts(args);
    }

    while (*args) {
	switch (*args++) {
	    case 'y':
	    case 'h':
	    case 't':
		lastlog_opts[i++] = *args;
		break;
	    default:
		return 1;
	}
    }

    return 0;
}

static int get_gecos_opts(const char *args)
{
    int i = 0;

    for (i = 0; i < strlen(args); i++) {
	switch (args[i]) {
	    case 'n':
	    case 'o':
	    case 'w':
	    case 'h':
	    case 'a':
		break;
	    default:
		return 1;
	}
    }

    i = 0;

    if (strchr(args, 'a') != NULL) {
	args = "nowh";
	get_gecos_opts(args);
    }

    while (*args) {
	switch (*args++) {
	    case 'n':
	    case 'o':
	    case 'w':
	    case 'h':
		gecos_opts[i++] = *args;
		break;
	    default:
		return 1;
	}
    }

    return 0;
}

static int command_line_opts(int nargc, char *nargv[], const char *opts)
{
    int opt, nonoutput = 0, i;
    static int options;

    while ((opt = Getopt(nargc, nargv, opts)) != -1) {
	switch (opt) {
	    case 'P':
		if (majoropt[MOPT_PASSWD])
		    continue;

		majoropt[MOPT_PASSWD] = 1;
		return MOPT_PASSWD;
		break;
	    case 'M':
		if (majoropt[MOPT_MAIL])
		    continue;

		majoropt[MOPT_MAIL] = 1;
		return MOPT_MAIL;
		break;
	    case 'L':
		if (majoropt[MOPT_LOGIN])
		    continue;

		majoropt[MOPT_LOGIN] = 1;
		return MOPT_LOGIN;
		break;
	    case 'A':
		if (majoropt[MOPT_ALL])
		    continue;

		majoropt[MOPT_PASSWD] = 1;
		majoropt[MOPT_MAIL] = 1;
		majoropt[MOPT_LOGIN] = 1;
		majoropt[MOPT_ALL] = 1;
		return MOPT_ALL;
		break;
	    case 'D':
		if (optarg[0] != '\\' && strlen(optarg) > 1)
		    usage(1);

		if ((delimchar = escapes(optarg)) == 0)
		    usage(1);

		nonoutput = 1;
		break;
	    case 'j':
		strncpy(tf, optarg, sizeof(tf));
		nonoutput = 1;
		break;
	    case 'h':
		usage(0);
		break;
	    case 'V':
		printf("%s\n%s\n", PACKAGE_STRING, COPYRIGHT);
		exit(EXIT_SUCCESS);
		break;
	    case 'F':
		followsymlinks = 1;
		nonoutput = 1;
		break;
	    case 'f':
		usefile = 1;
		nonoutput = 1;
		break;
	    case 'v':
		verbose = 1;
		nonoutput = 1;
		break;
	    case 'i':
		if (majoropt[MOPT_PASSWD])
		    gecosopts = "a";
		else
		    gecosopts = optarg;
	    case 'l':
	    case 's':
	    case 'u':
	    case 'd':
	    case 'm':
	    case 'p':
	    case 'c':
	    case 'k':
	    case 'g':
		passwd_info = 1;
		break;
	    case 'z':
	    case 'o':
	    case 'r':
	    case 'a':
	    case 'w':
		mail_info = 1;
		break;
	    case 'y':
	    case 'e':
	    case 'b':
	    case 'x':
	    case 'n':
		if (majoropt[MOPT_LOGIN])
		    lastopts = "a";
		else
		    lastopts = optarg;
	    case 'q':
	    case 'C':
	    case 't':
		login_info = 1;
		break;
	    case '?':
	    default:
		usage(EXIT_FAILURE);
		break;
	}

	if (!nonoutput) {
	    for (i = 0; i < ARRAYCNT(optspec); i++) {
		if (opt == optspec[i])
		    goto end;
	    }

	    optspec[options++] = opt;
	}

      end:
	nonoutput = 0;
    }

    return opt;
}

static int junction(INFO * info, const char *arg)
{
    struct stat st;

#ifdef HAVE_SETPASSENT
    setpassent(1);
#else
    setpwent();
#endif

#ifdef HAVE_GETSPNAM
    if (amroot)
	setspent();
#endif

    if (usefile) {
	if ((STAT(arg, &st)) == -1) {
	    warn("%s", arg);
	    return EXIT_FAILURE;
	}

	if ((info->passwd = getpwuid(st.st_uid)) == NULL) {
	    warnx("%s: no such uid %u", arg, st.st_uid);
	    return EXIT_FAILURE;
	}

#ifdef HAVE_GETSPNAM
	if (amroot) {
	    if ((info->spwd = getspnam(info->passwd->pw_name)) == NULL) {
		warnx("%s", "getspnam(): unknown error");
		return EXIT_FAILURE;
	    }
	}
#endif
    }
    else {
	if ((info->passwd = getpwnam(arg)) == NULL) {
	    if (errno == ENOMEM)
		warn("%s", "getpwnam()");
	    else
		warnx("%s: no such user", arg);

	    return EXIT_FAILURE;
	}

#ifdef HAVE_GETSPNAM
	if (amroot) {
	    if ((info->spwd = getspnam(arg)) == NULL) {
		warnx("%s", "getspnam(): unknown error");
		return EXIT_FAILURE;
	    }
	}
#endif
    }

    /*
     * fill the info structure 
     */
    if (passwd_info)
	passwdinfo(info);

    if (mail_info)
	mailinfo(info, arg);

    if (login_info)
	logininfo(info, arg);

    outputinfo(info, arg);
    return EXIT_SUCCESS;
}

int main(int argc, char *argv[])
{
    extern int optreset;
    int i;
    int ret = EXIT_SUCCESS;
    INFO *info;

#ifdef HAVE_LIBGEN_H
    if ((__progname = basename(argv[0])) == NULL) {
	warn("%s", argv[0]);
	__progname = "ui";
    }
#else
    __progname = argv[0];
#endif

    delimchar = DEFDELIM;
    strncpy(tf, DEF_TIMEFORMAT, sizeof(tf));

    opterr = optind = 1;

    while ((i = command_line_opts(argc, argv, CMDLINE_OPTS)) != -1) {
	char *moptargv[2], *mopts = NULL, tmp[MAXOPTIONS];
	int oldoptind;

	tmp[0] = 0;

	switch (i) {
	    case MOPT_PASSWD:
		mopts = MOPT_ORDER_PASSWD;
		strcpy(tmp, "-" MOPT_ORDER_PASSWD);
		moptargv[1] = tmp;
		break;
	    case MOPT_MAIL:
		mopts = MOPT_ORDER_MAIL;
		strcpy(tmp, "-" MOPT_ORDER_MAIL);
		moptargv[1] = tmp;
		break;
	    case MOPT_LOGIN:
		mopts = MOPT_ORDER_LOGIN;
		strcpy(tmp, "-" MOPT_ORDER_LOGIN);
		moptargv[1] = tmp;
		break;
	    case MOPT_ALL:
		strcpy(tmp,
		       MOPT_ORDER_PASSWD MOPT_ORDER_MAIL MOPT_ORDER_LOGIN);
		mopts = tmp;
		strcpy(tmp, "-" MOPT_ORDER_PASSWD MOPT_ORDER_MAIL
		       MOPT_ORDER_LOGIN);
		moptargv[1] = tmp;
		break;
	    default:
		break;
	}

	oldoptind = optind;
	optind = 1;
	optreset = 1;
	moptargv[0] = __progname;

	if ((i = command_line_opts(ARRAYCNT(moptargv), moptargv, mopts)) != -1)
	    usage(1);

	optind = oldoptind;
	optreset = 1;
    }

    if (!mail_info && !passwd_info && !login_info)
	usage(EXIT_FAILURE);

    if (argc == optind)
	use_stdin = 1;

#ifdef HAVE_GETSPNAM
    if (getuid() == 0)
	amroot = 1;
#endif

    info = Calloc(1, sizeof(INFO));

    if (gecosopts) {
	info->gecos = Malloc(sizeof(GECOS));

	if (get_gecos_opts(gecosopts) != 0)
	    usage(EXIT_FAILURE);
    }

    if (lastopts) {
	info->lastinfo = Malloc(sizeof(LASTINFO));

	if (get_lastlog_opts(lastopts) != 0)
	    usage(EXIT_FAILURE);
    }

    for (;; optind++) {
	char *arg = argv[optind];
	char buf[FILENAME_MAX];

	if (!use_stdin && optind == argc)
	    break;

	if (arg && arg[0] == '-' && arg[1] == 0)
	    use_stdin = 1;

	group_info = 0;

	if (use_stdin) {
	    while ((arg = fgets(buf, sizeof(buf), stdin)) != NULL) {
		arg[strlen(arg) - 1] = 0;

		if (arg[0] == 0)
		    continue;

		ret |= junction(info, arg);
	    }

	    break;
	}

	ret |= junction(info, arg);
    }

    if (info->login)
	free(info->login);

    if (info->mail)
	free(info->mail);

    if (info->gecos)
	free(info->gecos);

    if (info->lastinfo)
	free(info->lastinfo);

    free(info);

#ifdef HAVE_PROCFS
    if (procdir)
	closedir(procdir);
#endif

    if (utmpfd)
	close(utmpfd);

    if (lastlogfd)
	close(lastlogfd);

    if (aliasfd > 0) {
	close(aliasfd);
	munmap(aliasbuf, strlen(aliasbuf));
    }

    if (passwd_info)
	endpwent();

#ifdef HAVE_KVM_H
    if (kd)
	kvm_close(kd);
#endif

#ifdef HAVE_GETSPNAM
    if (amroot) {
	if (passwd_info)
	    endspent();
    }
#endif

    if (group_info)
	endgrent();

    exit(ret);
}
