/*
 * twlog:  A basic ham loging program using Motif
 * Copyright (C) 1997->2007 Ted Williams - WA0EIR 
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have recieved a copy of the GNU General Public
 * License along with this program; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge. MA 02139,
 * USA. See the COPYING file in this directory.
 *
 * Version: 2.5 -  Dec 2007
 */

/*
 * DIALOGS FOR TWLOG AND THEIR CALLBACKS
 */

#include "twlog.h"

                                /* from the top down, list QSOs */
#define FORWARD   0             /* recent to oldest QSO or */
#define REVERSE   !FORWARD      /* oldest to recent QSO */
#define DIRECTION FORWARD

extern int shmid;
#if MAKE_ICON && HAVE_X11_XPM_H && HAVE_LIBXPM     
extern Pixmap pixmap;
#endif

struct tag2
{
   Widget searchTF;
   Widget editTxt;
   Widget searchPB;
   Boolean new;
};

int intlen[] = { 0, 15, 10, 15, 15, 10, 10, 10, 10, 10, 80 };

char *charlen[] = {
   "0", "15", "10", "15", "15",
   "10", "10", "10", "10", "10", "1"
};

/*
 * popupEdit Function
 */
void popupEdit (char *filename)
{
   Widget w;

   w = build_editor ();
   XtManageChild (w);
}


/*
 * build_editor
 * This dialog is the log editor.  It is created the first time it is
 * called, and just popped up and down after that by managing its form.
 * It is application-modal to prevent the main interface from writing
 * to the logfile while the editor has it open.  It also keeps controls
 * the save button's sensitivity and loads the matrix if needed
 */
Widget build_editor ()
{
   static Widget editDiag, searchTF, sep1, okPB, savePB;
   static Widget searchPB, form2, searchLabel;
   static struct tag2 textwids;

   if (editDiag == NULL)
   {
      editDiag =
         XtVaCreateWidget ("editDiag", xmDialogShellWidgetClass, logSH,
         XmNtitle, "TWLOG EDITOR",
         #if MAKE_ICON && HAVE_X11_XPM_H && HAVE_LIBXPM     
         XmNiconPixmap, pixmap,
         #endif
         XmNmwmDecorations,
            MWM_DECOR_RESIZEH | MWM_DECOR_TITLE | MWM_DECOR_BORDER,
         NULL);

      form2 = XtVaCreateWidget ("form2", xmFormWidgetClass, editDiag,
         XmNdialogStyle, XmDIALOG_PRIMARY_APPLICATION_MODAL,
         XmNautoUnmanage, False,
         NULL);

      okPB = XtVaCreateManagedWidget ("OK", xmPushButtonWidgetClass, form2,
         XmNwidth, PBWIDTH,
         XmNbottomAttachment, XmATTACH_FORM,
         XmNbottomOffset, MARGIN,
         XmNleftAttachment, XmATTACH_FORM,
         XmNleftOffset, MARGIN,
         NULL);

      savePB =
         XtVaCreateManagedWidget ("Save", xmPushButtonWidgetClass, form2,
            XmNwidth, PBWIDTH,
            XmNbottomAttachment, XmATTACH_FORM,
            XmNbottomOffset, MARGIN,
            XmNleftAttachment, XmATTACH_POSITION,
            XmNleftPosition, 50,
            XmNleftOffset, -PBWIDTH / 2,
            NULL);

      searchPB = XtVaCreateManagedWidget ("Search", xmPushButtonWidgetClass,
         form2,
         XmNsensitive, False,
         XmNwidth, PBWIDTH,
         XmNbottomAttachment, XmATTACH_FORM,
         XmNrightAttachment, XmATTACH_FORM,
         XmNbottomOffset, MARGIN,
         XmNrightOffset, MARGIN,
         NULL);

      sep1 = XtVaCreateManagedWidget ("sep1", xmSeparatorWidgetClass, form2,
         XmNtopAttachment, XmATTACH_NONE,
         XmNbottomAttachment, XmATTACH_FORM,
         XmNbottomOffset, 70,
         XmNleftAttachment, XmATTACH_FORM,
         XmNrightAttachment, XmATTACH_FORM,
         NULL);

      searchTF = XtVaCreateManagedWidget ("searchTF", xmTextFieldWidgetClass,
         form2,
         XmNsensitive, True,
         XmNtopAttachment, XmATTACH_NONE,
         XmNbottomAttachment, XmATTACH_WIDGET,
         XmNbottomWidget, sep1,
         XmNbottomOffset, MARGIN,
         XmNleftAttachment, XmATTACH_FORM,
         XmNrightAttachment, XmATTACH_FORM,
         XmNleftOffset, MARGIN,
         XmNrightOffset, MARGIN,
         NULL);

      searchLabel =
         XtVaCreateManagedWidget ("Search Text:", xmLabelWidgetClass, form2,
         XmNbottomAttachment, XmATTACH_WIDGET,
         XmNbottomWidget, searchTF,
         XmNleftAttachment, XmATTACH_FORM,
         XmNleftOffset, MARGIN,
         NULL);

      mwText = XtVaCreateManagedWidget ("mwText", xbaeMatrixWidgetClass, form2,
         XmNtopAttachment, XmATTACH_FORM,
         XmNtopOffset, MARGIN,
         XmNbottomAttachment, XmATTACH_WIDGET,
         XmNbottomWidget, searchLabel,
         XmNleftAttachment, XmATTACH_FORM,
         XmNleftOffset, MARGIN,
         XmNrightAttachment, XmATTACH_FORM,
         XmNrightOffset, MARGIN,
         XmNcalcCursorPosition, True,
         NULL);

      /*
       * Set up the struct that is passed as client data
       */
      textwids.searchTF = searchTF;
      textwids.editTxt = mwText;
      textwids.searchPB = searchPB;
      textwids.new = True;

      XtAddCallback (okPB, XmNactivateCallback, okCB, (XtPointer) form2);

      XtAddCallback (savePB, XmNactivateCallback, saveCB, (XtPointer) mwText);

      XtAddCallback (searchPB, XmNactivateCallback, searchCB,
         (XtPointer) & textwids);
      XtAddCallback (searchTF, XmNactivateCallback, searchCB,
         (XtPointer) & textwids);
      XtAddCallback (searchTF, XmNmodifyVerifyCallback, searchCB,
         (XtPointer) & textwids);

      XtAddCallback (mwText, XmNmodifyVerifyCallback, cellCB,
         (XtPointer) savePB);
      XtAddCallback (mwText, XmNleaveCellCallback, cellCB,
         (XtPointer) savePB);
      XtAddCallback (mwText, XmNenterCellCallback, cellCB,
         (XtPointer) savePB);
   }

   /*
    *Read the logfile and put it in the widget
    */
   mwLoad (mwText);

   XtVaSetValues (savePB,
      XmNsensitive, False,
      NULL);

   XtManageChild (mwText);
   return (form2);
}


/*
 * okCB Callback Function 
 * Used to unmanage the edit dialog.  The about and help dialog don't use this.
 */
void okCB (Widget w, XtPointer cdata, XtPointer cbs)
{
   XtUnmanageChild ((Widget) cdata);
}


/*
 * cell Callback Function 
 * mod/ver, enter and leave CBs.  Used to make the Save button sensitive
 * and !!! delete a QSO !!!
 */
void cellCB (Widget w, XtPointer cdata, XtPointer cbs)
{
   XbaeMatrixAnyCallbackStruct *pt1 = (XbaeMatrixAnyCallbackStruct *) cbs;
   XbaeMatrixEnterCellCallbackStruct *pt2;
   XbaeMatrixModifyVerifyCallbackStruct *pt3;
   Widget savePB = (Widget) cdata;
   static int mods;

   switch (pt1->reason)
   {
      case XbaeLeaveCellReason:
         if (mods == 1)
         {
            XtVaSetValues (savePB,
               XmNsensitive, True,
               NULL);
         }
         mods = 0;
         break;

      /* so what do we want to use these for? */
      case XbaeModifyVerifyReason:
         mods = 1;
         /* check the row we modified and make sure we don't exceed the len */
         pt3 = (XbaeMatrixModifyVerifyCallbackStruct *) cbs;
         break;

      case XbaeEnterCellReason:
         pt2 = (XbaeMatrixEnterCellCallbackStruct *) cbs;
         if (pt2->column == 0)
         {
//TJW
            // could add a delete row operation here
            // XbaeMatrixSelectRow (w, pt2->row);
            // printf ("delete qso #%s\n", XbaeMatrixGetCell (w,
            //                            pt2->row, pt2->column));
         }
         break;

      default:
         fprintf (stderr, "mkSensCB: Invalid reason\n");
   }
}


/*
 * SaveCB
 */
void saveCB (Widget w, XtPointer cdata, XtPointer cbs)
{
   int row, col, rowcnt;
   Widget mw = (Widget) cdata;
   String cell;
   char fmt[6];
   FILE *fp;

   /* open the file */
   if ((fp = fopen (logpath, "w")) == 0)
   {
      errorDiag (w,"Open logfile for write failed.", !CANCEL_BTN);
      fprintf (stderr, "%s:  Open logfile for write failed\n", logpath);
      return;
   }

   rowcnt = XbaeMatrixNumRows (mw) - 1;

   for (row = rowcnt; row >= 0; row--)
   {
      for (col = 1; col <= 10; col++)
      {
         strcpy (fmt, "%-");
         strcat (fmt, charlen[col]);
         strcat (fmt, "s");
         cell = XbaeMatrixGetCell (mw, row, col);
         fprintf (fp, fmt, cell);
      }
      fprintf (fp, "\n");       /* put \n back on the end of each line */
   }
   fclose (fp);

   /* Saved, so make Save button insensitive again */
   XtVaSetValues (w,
      XmNsensitive, False,
      NULL);
}


/*
 * searchCB Callback Function
 * Activate and modify/verify callbacks for the Search button and textfield.
 * For an activate, get the string, go look for it and if found, display
 * the text by setting the top position so the text will be in the window.
 * If not found, make new = True and clear the search text widget.
 * If it was a modify/verify from the searchTF then the search pattern has
 * changed. Set the new flag to True so we start the search over.
 */
void searchCB (Widget w, XtPointer cdata, XtPointer cbs)
{
   struct tag2 *wids = (struct tag2 *) cdata;
   static int num_rows, start_row, start_col;
   char *find_str, *str;
   Pixel bg;
   XmAnyCallbackStruct *cbsTF = (XmAnyCallbackStruct *) cbs;
   XmTextVerifyCallbackStruct *pt;
   int row, col;

   if (wids->new == True)       /* if true, we are starting a new search */
   {
      start_row = 0;
      start_col = 0;
      num_rows = XbaeMatrixNumRows (mwText);
   }

   switch (cbsTF->reason)
   {
      case XmCR_ACTIVATE:
         XtVaGetValues (wids->searchTF, /* get the search pattern */
            XmNvalue, &find_str,
            NULL);

         if (strlen (find_str) == 0)    /* empty string so no search */
         {
            XtVaSetValues (wids->searchPB,
               XmNsensitive, False,
               NULL);
            wids->new = True;
            return;
         }

         for (row = start_row; row < num_rows; row++)
         {
            for (col = start_col; col <= 10; col++)
            {
               str = XbaeMatrixGetCell (mwText, row, col);
               if (strstr (str, find_str) != NULL)
               {
                  /* display the row as the first row in the window */
                  /* and highlight col */
                  XtVaGetValues (mwText, XmNbottomShadowColor, &bg, NULL);

                  XbaeMatrixSetCellBackground (mwText, row, col, bg);

                  XtVaSetValues (mwText,
                     XmNtopRow, row,
                     NULL);

                  /* setup for next search */
                  if (col == 10)           /* if found in last col */
                  {
                     start_col = 0;        /* start next one in first col */
                     start_row = row + 1;  /* of the next row */
                  }
                  else
                  {
                     start_col = col + 1;  /* else, start at the next col */
                     start_row = row;      /* of the current row */
                  }
                  wids->new = False;
                  return;
               }
            }
            start_col = 0;      /* continue a search at row++,0 */
         }

         /* we get here when nothing is found */
         wids->new = True;
         XtVaSetValues (wids->searchTF,
            XmNvalue, "",
            NULL);

         XtVaSetValues (wids->searchPB,
            XmNsensitive, False,
            NULL);
         XtFree (find_str);
         return;

      case XmCR_MODIFYING_TEXT_VALUE:
         pt = (XmTextVerifyCallbackStruct *) cbs;
         if (pt->startPos == pt->endPos)        /* text was entered */
         {
            wids->new = True;
            XtVaSetValues (wids->searchPB,
               XmNsensitive, True,
               NULL);
            return;
         }

         XtVaGetValues (wids->searchTF, /* get the search pattern */
            XmNvalue, &find_str,
            NULL);

         if (pt->startPos == 0 || pt->endPos == 1) 
         {
            wids->new = False;
            XtVaSetValues (wids->searchPB,
               XmNsensitive, False,
               NULL);
         }
         break;
   }
}


/*
 * aboutDiag Function
 * Just an info dialog
 */
void aboutDiag ()
{
   static Widget diag = NULL;
   Widget pb;
   XmString msg_xs;
   char msg_str[] =
      PACKAGE_STRING
      " \251 1997-2008\nwas written by Ted Williams - WA\330EIR\n\n"
      "Please send bug reports to\n" PACKAGE_BUGREPORT;

   msg_xs = XmStringCreateLtoR (msg_str, XmSTRING_DEFAULT_CHARSET);
   if (diag == NULL)
   {
      diag = XmCreateInformationDialog (logSH, "aboutDiag",
         (ArgList) NULL, 0);

      XtVaSetValues (XtParent (diag),
         XmNtitle, "ABOUT TWLOG",
         #if MAKE_ICON && HAVE_X11_XPM_H && HAVE_LIBXPM     
         XmNiconPixmap, pixmap,
         #endif
         NULL);

      XtVaSetValues (diag,
         XmNmessageAlignment, XmALIGNMENT_CENTER,
         XmNmessageString, msg_xs,
         NULL);
      XmStringFree (msg_xs);

      XtUnmanageChild (XmMessageBoxGetChild (diag, XmDIALOG_CANCEL_BUTTON));
      XtUnmanageChild (XmMessageBoxGetChild (diag, XmDIALOG_HELP_BUTTON));
      XtUnmanageChild (XmMessageBoxGetChild (diag, XmDIALOG_OK_BUTTON));

      pb = XtVaCreateManagedWidget ("OK", xmPushButtonWidgetClass, diag,
         NULL);
   }
   XtManageChild (diag);
}


/*
 * number the rows in a matrix
 */
void numberRows (Widget mw)
{
   int i, j, rows;
   char num[6];

   rows = XbaeMatrixNumRows (mw);       /* how many rows to number? */
   if (DIRECTION == FORWARD)
   {
      j = rows;
      for (i = 1; i <= rows; i++, j--)
      {
         sprintf (num, "%d", j);
         XbaeMatrixSetCell (mw, i - 1, 0, num);
      }
   }
   else                         /* REVERSE */
   {
      //j = 1;
      for (i = 1; i <= rows; i++)
      {
         sprintf (num, "%d", i);
         XbaeMatrixSetCell (mw, i - 1, 0, num);
      }
   }
}


/*
 * mwLoad - reads logfile into matrix
 */
void mwLoad (Widget mw)
{
   int i, r;
   FILE *fp;
   char str[200];
   char *ptr = str;
   String cells[12];
   char fields[12][80];

   /*
    * make sure it is empty first
    */
   r = XbaeMatrixNumRows (mw);
   XbaeMatrixDeleteRows (mw, 0, r);

   /*
    * Load the matrix from the file
    */
   for (i = 0; i <= 11; i++)
      cells[i] = NULL;

   /* open the file */
   if ((fp = fopen (logpath, "r")) == 0)
   {
      errorDiag (XtParent(mw), "Open logfile for read failed.", !CANCEL_BTN);
      fprintf (stderr, "open logfile for read failed\n");
      return;
   }

   /* get lines */
   while (fgets (str, 200, fp) != NULL)
   {
      ptr = str;
      for (i = 1; i <= 10; i++)
      {
         snprintf (fields[i], intlen[i], "%s", ptr);
         cells[i] = fields[i];

         /* Trim each cell */
         for (r = strlen (cells[i]) - 1; r >= 0; r--)
         {
            if (cells[i][r] == ' ' || cells[i][r] == '\n')
            {
               cells[i][r] = '\0';
            }
            else
               break;
         }
         ptr = ptr + intlen[i];
      }

      /* put cells into matrix */
      if (DIRECTION == FORWARD)
      {
         XbaeMatrixAddRows (mw, 0, cells, NULL, NULL, 1);
      }
      else
      {
         XbaeMatrixAddRows (mw, XbaeMatrixNumRows (mw), cells, NULL, NULL, 1);
      }
   }
   fclose (fp);
   numberRows (mw);
}


/*
 * build converter Dialog 
 */
void build_convDiag (void)
{
   int i, j;
   int h = 35;
   char path[200] = "\0";
   static Widget convDiag, topLabel, form4;
   static Widget inLabel, inBrowsePB, typeLabel;
   static Widget outLabel, outBrowsePB;
   static Widget wids[3];
   Arg args[10];              /* room to spare */
   XmString xs[5];

   if (convDiag == NULL)      /* Lazy instantiation */
   {
      i = 0;
      xs[i] = XmStringCreateLocalized ("OK"); i++;          /* 0 */
      xs[i] = XmStringCreateLocalized ("Cancel"); i++;      /* 1 */

      j = 0;
      XtSetArg (args[j], XmNokLabelString, xs[0]); j++;
      XtSetArg (args[j], XmNcancelLabelString, xs[1]); j++;
      XtSetArg (args[j], XmNtitle, "Convert Log Dialog"); j++;
      XtSetArg (args[j], XmNdialogStyle,
                         XmDIALOG_PRIMARY_APPLICATION_MODAL); j++;
      XtSetArg (args[j], XmNautoUnmanage, False); j++;
      #if MAKE_ICON && HAVE_X11_XPM_H && HAVE_LIBXPM     
      XtSetArg (args[j], XmNiconPixmap, pixmap); j++;
      #endif

      convDiag = XmCreateTemplateDialog (logSH, "convDiag", args, j);

      for (j=0; j<i; j++)
      {
	 XmStringFree (xs[j]);
      }

      form4 = XtVaCreateWidget ("convForm", xmFormWidgetClass, convDiag,
         NULL);

      topLabel = XtVaCreateManagedWidget ("Convert Log",
         xmLabelWidgetClass, form4,
         XmNtopAttachment, XmATTACH_FORM,
         XmNleftAttachment, XmATTACH_FORM,
         XmNbottomAttachment, XmATTACH_NONE,
         XmNrightAttachment, XmATTACH_FORM,
         NULL);

      inLabel = XtVaCreateManagedWidget ("Input File:", xmLabelWidgetClass,
         form4,
         XmNheight, h,
         XmNwidth, 100,
         XmNtopAttachment, XmATTACH_WIDGET,
         XmNtopWidget, topLabel,
         XmNleftAttachment, XmATTACH_FORM,
         XmNbottomAttachment, XmATTACH_NONE,
         XmNrightAttachment, XmATTACH_NONE,
         NULL);

      strcpy (path, dirpath);
      strcat (path, "/adif.log");      /* default if To twlog */
      wids[0] = XtVaCreateManagedWidget ("inTextF", xmTextFieldWidgetClass,
         form4,
         XmNcolumns, 30,
         XmNvalue, path,
         XmNcursorPosition, strlen(path),
         XmNtopAttachment, XmATTACH_WIDGET,
         XmNtopWidget, topLabel,
         XmNleftAttachment, XmATTACH_WIDGET,
         XmNleftWidget, inLabel,
         XmNbottomAttachment, XmATTACH_OPPOSITE_WIDGET,
         XmNbottomWidget, inLabel,
         XmNrightAttachment, XmATTACH_NONE,
         NULL);
      XmTextFieldShowPosition (wids[0], strlen(path));

      inBrowsePB = XtVaCreateManagedWidget ("Browse...",
         xmPushButtonWidgetClass, form4,
         XmNtopAttachment, XmATTACH_WIDGET,
         XmNtopWidget, topLabel,
         XmNleftAttachment, XmATTACH_WIDGET,
         XmNleftWidget, wids[0],
         XmNbottomAttachment, XmATTACH_OPPOSITE_WIDGET,
         XmNbottomWidget, inLabel,
         NULL);

      typeLabel = XtVaCreateManagedWidget ("Convert to:", xmLabelWidgetClass,
         form4,
         XmNheight, h,
         XmNtopAttachment, XmATTACH_WIDGET,
         XmNtopWidget, inLabel,
         XmNleftAttachment, XmATTACH_FORM,
         XmNbottomAttachment, XmATTACH_NONE,
         XmNrightAttachment, XmATTACH_OPPOSITE_WIDGET,
         XmNrightWidget, inLabel,
         NULL);

      i = 0;
      xs[i] = XmStringCreateLocalized ("twlog"); i++;
      xs[i] = XmStringCreateLocalized ("adif"); i++;

      wids[2] = XmVaCreateSimpleOptionMenu (form4, "typeOptMenu", NULL,
         '\0', 0, NULL,
         XmVaPUSHBUTTON, xs[0], '\0', "", NULL,
         XmVaPUSHBUTTON, xs[1], '\0', "", NULL,
         NULL);

      for (j=0; j<i; j++)
      {
         XmStringFree (xs[j]);
      }

      XtVaSetValues (wids[2],
         XmNtopAttachment, XmATTACH_WIDGET,
         XmNtopWidget, inLabel,
         XmNleftAttachment, XmATTACH_WIDGET,
         XmNleftWidget, typeLabel,
         XmNbottomAttachment, XmATTACH_OPPOSITE_WIDGET,
         XmNbottomWidget, typeLabel,
         XmNrightAttachment, XmATTACH_NONE,
         XmNborderWidth, 0,
         NULL);

      outLabel = XtVaCreateManagedWidget ("Output File:", xmLabelWidgetClass,
         form4,
         XmNheight, h,
         XmNtopAttachment, XmATTACH_WIDGET,
         XmNtopWidget, typeLabel,
         XmNleftAttachment, XmATTACH_FORM,
         XmNbottomAttachment, XmATTACH_NONE,
         XmNrightAttachment, XmATTACH_OPPOSITE_WIDGET,
         XmNrightWidget, inLabel,
         NULL);

      strcpy (path, dirpath);      /* default to twlog so just dir */

      wids[1] = XtVaCreateManagedWidget ("outTextF", xmTextFieldWidgetClass,
         form4,
         XmNvalue, path,
         XmNcolumns, 20,
         XmNcursorPosition, strlen(path),
         XmNtopAttachment, XmATTACH_WIDGET,
         XmNtopWidget, typeLabel,
         XmNleftAttachment, XmATTACH_WIDGET,
         XmNleftWidget, outLabel,
         XmNbottomAttachment, XmATTACH_OPPOSITE_WIDGET,
         XmNbottomWidget, outLabel,
         XmNrightAttachment, XmATTACH_OPPOSITE_WIDGET,
         XmNrightWidget, wids[0],
         NULL);

      outBrowsePB = XtVaCreateManagedWidget ("Browse...",
         xmPushButtonWidgetClass, form4,
         XmNheight, h,
         XmNtopAttachment, XmATTACH_WIDGET,
         XmNtopWidget, typeLabel,
         XmNleftAttachment, XmATTACH_WIDGET,
         XmNleftWidget, wids[1],
         XmNbottomAttachment, XmATTACH_OPPOSITE_WIDGET,
         XmNbottomWidget, outLabel,
         NULL);

      XtManageChild (wids[2]);
      XtManageChild (form4);

      XtAddCallback (inBrowsePB, XmNactivateCallback, browserCB, wids[0]);
      XtAddCallback (outBrowsePB, XmNactivateCallback, browserCB, wids[1]);
      XtAddCallback (convDiag, XmNokCallback, convCB, wids);
      XtAddCallback (convDiag, XmNcancelCallback, convCB, wids);
   }
   XtManageChild (convDiag);
}

/*
 * browserCB - Creates and pops up the fsb for input or output fsb.
 * Client data is the wid for the coresponding text field with 
 * path string.
 */
void browserCB (Widget w, XtPointer cdata, XtPointer cbs)
{
   static Widget fsb1;
   Widget wid = (Widget)cdata;
   char dirStr[200] = "\0";
   XmString xs;

   strcpy (dirStr, getenv ("HOME"));       /* build the dir string */
   strcat (dirStr, LOGDIR);
   strcat (dirStr, "/*");
   xs = XmStringCreateLocalized (dirStr);  /* build xs for fsb dir */

   /* create fsb for input file */
   fsb1 = XmCreateFileSelectionDialog (w, "fsb1", NULL, 0);

   XtVaSetValues (fsb1,
      XmNdirMask, xs,                    /* set dir value */
      XmNmustMatch, True,
      XmNdialogStyle, XmDIALOG_PRIMARY_APPLICATION_MODAL,
      #if MAKE_ICON && HAVE_X11_XPM_H && HAVE_LIBXPM     
      XmNiconPixmap, pixmap,
      #endif
      NULL);
   XmStringFree (xs);

   XtVaSetValues (XtParent(fsb1),
      XmNtitle, "TWLOG FILE BROWSER",
      #if MAKE_ICON && HAVE_X11_XPM_H && HAVE_LIBXPM     
      XmNiconPixmap, pixmap,
      #endif
      NULL);

   XtUnmanageChild (XmFileSelectionBoxGetChild (fsb1, XmDIALOG_HELP_BUTTON));

   /* add some callbacks */
   XtAddCallback (fsb1, XmNokCallback, fsbCB, (XtPointer) wid);
   XtAddCallback (fsb1, XmNcancelCallback, fsbCB, (XtPointer) NULL);
   XtAddCallback (fsb1, XmNnoMatchCallback, fsbCB, (XtPointer) NULL);
   /* pop it up */
   XtManageChild (fsb1);
}


/*
 * fsbCB Callback Function 
 * handles the FSB callbacks
 */
void fsbCB (Widget w, XtPointer cdata, XtPointer cbs)
{
   char *fname;
   Widget wid = (Widget)cdata;
   XmFileSelectionBoxCallbackStruct *p =
      (XmFileSelectionBoxCallbackStruct*) cbs;

   switch (p->reason)
   {
      case XmCR_CANCEL:
         XtUnmanageChild (w);
         return; 

      case XmCR_OK:
         fname = XmStringUnparse (p->value, NULL, XmCHARSET_TEXT,
                                  XmCHARSET_TEXT, NULL, 0, XmOUTPUT_ALL);

         XtVaSetValues (wid,
            XmNvalue, fname,
            NULL);
         XmTextShowPosition (wid, strlen(fname));
         XtFree (fname);                 /* Stop memory leak */
         break;

      case XmCR_NO_MATCH:
         errorDiag (w, "You need to select a file.", !CANCEL_BTN);
         return;

      default:
         fprintf (stderr, "fsbCB: invalid request\n");
         break;
   }
   XtUnmanageChild ((Widget) w);
}


char  *btnstr, *infile, *outfile;     /* some globals */

/*
 * convCB Callback - for OK and Cancel buttons on Convert Dialog
 */
void convCB (Widget w, XtPointer cdata, XtPointer cbs)
{
   int rtn;
   struct stat buf;
   Widget btnwid;
   XmString xs;
   Widget *wids = (Widget *) cdata;
   XmAnyCallbackStruct *p = (XmAnyCallbackStruct *) cbs;

   if (p->reason == XmCR_CANCEL)  /* if from Cancel button, unmanage dialog */
   {
      XtUnmanageChild (w);
      return;
   }
                                  /* otherwise it the OK button */
   XtVaGetValues (wids[0],        /* so get the paths form the text wids */
      XmNvalue, &infile,
      NULL);

   XtVaGetValues (wids[1],
      XmNvalue, &outfile,
      NULL);

   XtVaGetValues (wids[2],        /* and get the menu button wid. */
      XmNmenuHistory, &btnwid,
      NULL);
   /*
    * check the paths and permissions
    * if they are not valid, popup error dialog and return so the
    * convDiag is still managed
    */
   /***** Input path *****/
   if ((rtn = stat (infile, &buf)) == -1)          /* stat input file */
   {
      errorDiag (w, "The Input file was not found.\n", !CANCEL_BTN);
      fprintf (stderr, "Input file was not found\n");
      return;
   }
   if (S_ISDIR(buf.st_mode) == 1)                  /* can't be a dir */
   {
      errorDiag (w, "The Input file can not be a directory.", !CANCEL_BTN);
      fprintf (stderr, "Input file can't be a directory\n");
      return;
   }
   if ((S_IRUSR & buf.st_mode) == 0) 
   {
      errorDiag (w, "You do not have read permission on the Input file.\n",
                 !CANCEL_BTN);
      fprintf (stderr, "You do not have read premission\n");
      return;
   }

   /***** Output path *****/
   rtn = stat (outfile, &buf);
   if ((rtn == 0) && (S_ISDIR(buf.st_mode) == 1))  /* can't be a dir */
   {
      errorDiag (w, "The Output file can not be a directory.", !CANCEL_BTN);
      fprintf (stderr, "Output file can't be a directory.\n");
      return;
   }
   if (strcmp (infile, outfile) == 0)              /* names can't be == */
   {
      errorDiag (w, "Input and Output files can not be the same.", !CANCEL_BTN);
      fprintf (stderr, "Input and Output fills can not be the same.\n");
      return;
   }
   if ((rtn == 0) && ((S_IWUSR & buf.st_mode) == 0)) /* exists but no write*/
   {
      errorDiag (w, "You do not have write permission on the Output File.",
                 !CANCEL_BTN);
      fprintf (stderr, "You do not have write permission on the Output File\n");
      return;
   }

   /* get label for selected button and convert it to a char */
   XtVaGetValues (btnwid,
      XmNlabelString, &xs,
      NULL);

   btnstr = XmStringUnparse (xs, NULL, XmCHARSET_TEXT, XmCHARSET_TEXT,
                             NULL, 0, XmOUTPUT_ALL);

   /* this must be the last check */
   /* check for exists and writable here */
   if ((rtn == 0) && S_ISREG(buf.st_mode) == 1)  /* if file exists */
   {            //w or do we want the shell? 
      confirmDiag (w);
      fprintf (stderr, "The Output file exists.");
      return;
   }

   /* if we get here, all is good so convert to requested type */
   mk_log();
   XtUnmanageChild (w);
   return;
}


/*
 * confirmDiag
 */
void confirmDiag (Widget w)
{
   Widget confDiag;
   XmString xs_mess, xs_yes, xs_no;

   confDiag = XmCreateErrorDialog (w, "confDiag", NULL, 0);
   xs_mess = XmStringCreateLocalized ("The ouput file exists.\n"
                                      "Do you want to overwrite it?");
   xs_yes = XmStringCreateLocalized("Yes");
   xs_no = XmStringCreateLocalized("No");

   XtVaSetValues (XtParent(confDiag),
      XmNtitle, "CONVERT DIALOG",
      XmNmwmFunctions, MWM_FUNC_ALL | MWM_FUNC_CLOSE,
      NULL);

   XtVaSetValues (confDiag,
      XmNdialogStyle, XmDIALOG_PRIMARY_APPLICATION_MODAL,
      XmNmessageString, xs_mess,
      XmNcancelLabelString, xs_no,
      XmNokLabelString, xs_yes,
      NULL);
   XmStringFree (xs_yes);
   XmStringFree (xs_no);

   XtUnmanageChild (XmMessageBoxGetChild(confDiag, XmDIALOG_HELP_BUTTON));

   XtAddCallback (confDiag, XmNokCallback, confOkCB, (XtPointer) w);
//   XtAddCallback (confDiag, XmNcancelCallback, confCancelCB, w);

   XtManageChild (confDiag);
}


/*
 * confOkCB - OK callback for confDiag - OK to overwrite the existing
 * output file
 */
void confOkCB (Widget w, XtPointer cdata, XtPointer cbs)
{
   Widget convdiag = (Widget) cdata;

   /* it OK so create a logfile */
   mk_log();
   XtUnmanageChild (convdiag);
   return;
} 


/* 
 * mk_log - write the logfile.  The option button string says what
 * format to write.
 */
void mk_log (void)
{
   if (strcmp (btnstr, "twlog") == 0)
   {
      de_adif (infile, outfile);
   }
   else
   {
      to_adif (infile, outfile);
   }
   XtFree (btnstr);
}


/*
 * Error Dialog - creates an error dialog with either an OK or Exit Button.
 * Called from many places and passed the parent widget, the error message
 * to display and a flag that indicates if there should be an OK button 
 * or a Cancel/Exit button.
 */
void errorDiag (Widget w, char *emess, int flag)
{
   Widget errDiag;
   XmString xs_emess, xs_cancel;

   errDiag = XmCreateErrorDialog (w, "errDiag", NULL, 0);
   xs_emess = XmStringCreateLocalized(emess);
   xs_cancel = XmStringCreateLocalized("Exit");

   XtVaSetValues (XtParent(errDiag),
      XmNtitle, "TWLOG ERROR",
      #if MAKE_ICON && HAVE_X11_XPM_H && HAVE_LIBXPM     
      XmNiconPixmap, pixmap,
      #endif
      XmNmwmFunctions, MWM_FUNC_ALL | MWM_FUNC_CLOSE,
      NULL);

   XtVaSetValues (errDiag,
      XmNdialogStyle, XmDIALOG_PRIMARY_APPLICATION_MODAL,
      XmNcancelLabelString, xs_cancel,
      XmNmessageString, xs_emess,
      NULL);
   XmStringFree (xs_emess);

   XtUnmanageChild (XmMessageBoxGetChild(errDiag, XmDIALOG_HELP_BUTTON));
   XtAddCallback (errDiag, XmNokCallback, errOkCB, (XtPointer) -1);

   if (flag == CANCEL_BTN)
   {
      XtUnmanageChild (XmMessageBoxGetChild(errDiag, XmDIALOG_OK_BUTTON));
      XtAddCallback (errDiag, XmNcancelCallback, errCancelCB, w);
   }
   else
   {
      XtUnmanageChild (XmMessageBoxGetChild(errDiag, XmDIALOG_CANCEL_BUTTON));
      XtAddCallback (errDiag, XmNcancelCallback, errOkCB, XtParent(w));
   }

   XtManageChild (errDiag);
}


/*
 * errOkCB - used by the File Exists Dialog - cdata is wid of convert diag
 * which needs to be unmanaged now.
 */
void errOkCB (Widget w, XtPointer cdata, XtPointer cbs)
{
   /* nothing to do */
} 


/*
 * errCancelCB - unmanage the shell to exit
 */
void errCancelCB (Widget w, XtPointer cdata, XtPointer cbs)
{
   shmctl (shmid, IPC_RMID, NULL);    /* close the shared memory */
   XtUnmanageChild (cdata);
}
