/**
 * Copyright (C) 2007-2012 Lawrence Murray
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * @author Lawrence Murray <lawrence@indii.org>
 * $Rev: 380 $
 * $Date: 2012-02-04 20:30:12 +0800 (Sat, 04 Feb 2012) $
 */
#ifndef INDII_MODEL_MODEL_HPP
#define INDII_MODEL_MODEL_HPP

#include "Observable.hpp"
#include "../image/ColourSpace.hpp"

#include "wx/image.h"
#include "wx/dcbuffer.h"

namespace indii {
/**
 * Abstract model.
 */
class Model : public Observable {
public:
  /**
   * Constructor.
   */
  Model();

  /**
   * Get red channel proportion for conversion to greyscale.
   *
   * @return Red channel proportion.
   */
  float getRedMix() const;

  /**
   * Get green channel proportion for conversion to greyscale.
   *
   * @return Green channel proportion.
   */
  float getGreenMix() const;

  /**
   * Get blue channel proportion for conversion to greyscale.
   *
   * @return Blue channel proportion.
   */
  float getBlueMix() const;

  /**
   * Set red channel proportion for conversion to greyscale.
   *
   * @param r Red channel proportion.
   */
  void setRedMix(const float r);

  /**
   * Set red channel proportion for conversion to greyscale.
   *
   * @param g Green channel proportion.
   */
  void setGreenMix(const float g);

  /**
   * Set blue channel proportion for conversion to greyscale.
   *
   * @param b Blue channel proportion.
   */
  void setBlueMix(const float b);

  /**
   * Set greyscale mix.
   *
   * @param r Red channel proportion.
   * @param g Green channel proportion.
   * @param b Blue channel proportion.
   */
  void setChannelMix(const float r, const float g, const float b);

  /**
   * Set default parameters.
   */
  virtual void setDefaults() = 0;

  /**
   * Set parameters for dialog.
   */
  virtual void setForDialog() = 0;

  /**
   * Calculate foreground of subimage.
   *
   * @param rect Rectangular region of interest, relative to scaled image.
   * @param width Scaled width of image.
   * @param height Scaled height of image.
   * @param[out] o Output image.
   */
  virtual void calcFg(const wxRect& rect, const unsigned width,
      const unsigned height, wxImage& o);

  /**
   * Calculate foreground of whole image.
   *
   * @param width Scaled width of image.
   * @param height Scaled height of image.
   * @param[out] o Output image.
   */
  virtual void calcFg(const unsigned width, const unsigned height,
      wxImage& o);

  /**
   * Calculate annotation layer of subimage.
   *
   * @param rect Rectangular region of interest, relative to scaled image.
   * @param width Scaled width of image.
   * @param height Scaled height of image.
   * @param zoom Zoom factor to create this scale.
   * @param[out] dc Device context to annotate.
   */
  virtual void calcAn(const wxRect& rect, const unsigned width,
      const unsigned height, wxDC& dc);

  /**
   * Calculate annotation layer of whole image.
   *
   * @param width Scaled width of image.
   * @param height Scaled height of image.
   * @param[out] dc Device context to annotate.
   */
  virtual void calcAn(const unsigned width, const unsigned height,
      wxDC& dc);

protected:
  /**
   * Colour space model.
   */
  ColourSpace cs;

  /**
   * Greyscale red mixing.
   */
  float greyR;

  /**
   * Greyscale green mixing.
   */
  float greyG;

  /**
   * Greyscale green mixing.
   */
  float greyB;
};
}

inline float indii::Model::getRedMix() const {
  return greyR;
}

inline float indii::Model::getGreenMix() const {
  return greyG;
}

inline float indii::Model::getBlueMix() const {
  return greyB;
}

inline void indii::Model::calcFg(const wxRect& rect,
    const unsigned width, const unsigned height, wxImage& o) {
  //
}

inline void indii::Model::calcFg(const unsigned width,
    const unsigned height, wxImage& o) {
  wxRect rect(0, 0, width, height);
  calcFg(rect, width, height, o);
}

inline void indii::Model::calcAn(const wxRect& rect,
    const unsigned width, const unsigned height, wxDC& dc) {
  //
}

inline void indii::Model::calcAn(const unsigned width,
    const unsigned height, wxDC& dc) {
  wxRect rect(0, 0, width, height);
  calcAn(rect, width, height, dc);
}

#endif
