/* Copyright (C) 2000-2005  Thomas Bopp, Thorsten Hampel, Ludger Merkens
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 * 
 */
//inherit "/kernel/module";
inherit "/kernel/secure_mapping.pike";
inherit "/kernel/persistence.pike";

#include <macros.h>
#include <classes.h>
#include <database.h>
#include <exception.h>
#include <attributes.h>
#include <configure.h>

//#define LDAPPERSISTENCE_DEBUG 1

#ifdef LDAPPERSISTENCE_DEBUG
#define LDAPPERS_LOG(s, args...) werror("ldappersistence: "+s+"\n", args)
#else
#define LDAPPERS_LOG(s, args...)
#endif

#define DEPENDENCIES cache ldap

string get_identifier() { return "persistence:ldap"; }
object get_environment() { return 0; }

static object ldap;
static mapping    config;

static int persistence_id;
static int oid_length = 4;  // nr of bytes for object id

static object user_cache;
static object group_cache;
static int cache_time = 0;

static void init_module()
{
  ldap = get_module("ldap");
  if ( !objectp(ldap) ) {
    MESSAGE("Could not get LDAP module.");
    FATAL("Could not get LDAP module.");
    return;
  }

  if ( !ldap->ldap_activated() ) {
    MESSAGE("LDAP deactivated.");
    return;  // ldap deactivated
  }

  config = ldap->get_config();
  if ( !mappingp(config) ) {
    MESSAGE("LDAP persistence not started - missing configuration !");
    return;
  }

  persistence_id = _Persistence->register( "ldap", this_object() );

  if ( intp(config->cacheTime) )
    cache_time = config->cacheTime;
  else if ( !stringp(config->cacheTime) || (sscanf(config->cacheTime, "%d", cache_time) < 1) )
    cache_time = 0;
  user_cache = get_module("cache")->create_cache( "ldappersistence:users", cache_time );
  group_cache = get_module("cache")->create_cache("ldappersistence:groups", cache_time );
}

int get_persistence_id ()
{
  return persistence_id;
}


mixed lookup_user(string identifier, void|string password)
{
  if ( !objectp(ldap) || !ldap->ldap_activated() )
    return UNDEFINED;

  LDAPPERS_LOG( "lookup_user(%s)", identifier );

  if ( ! zero_type( user_cache->get( identifier ) ) ) {
    LDAPPERS_LOG( "user from cache: %s", identifier );
    return UNDEFINED;  // let Persistence use the database copy of the user
  }

  mixed data = ldap->fetch_user( identifier, password );

  if ( !mappingp(data) ) return UNDEFINED;

  mixed exc = catch {

  mapping result = ([ "ldap:dn" : data["dn"] ]);
  if(data[config->passwordAttr])
    result->UserPassword = data[config->passwordAttr];
  if(data[config->emailAttr])
    result[USER_EMAIL] = data[config->emailAttr];
  if(data[config->fullnameAttr])
    result[USER_FULLNAME] = data[config->fullnameAttr];
  if(data[config->nameAttr])
    result[USER_FIRSTNAME] = data[config->nameAttr];
  if(data[config->descriptionAttr])
    result[OBJ_DESC] = data[config->descriptionAttr];

  if ( stringp(config->groupAttr) && sizeof(config->groupAttr)>0 )
  {
      // primary group:
      if ( stringp(config->groupId) && stringp(data[config->groupId]) ) {
	  string searchstr = "(&(" + config->groupId + "=" + data[config->groupId] + ")"
	      + "(objectClass=" + config->groupClass + "))";
	  mixed primary_group = get_module("ldap")->fetch( "", searchstr );
	  if ( mappingp(primary_group) && stringp(config->groupAttr)
	       && stringp(primary_group[config->groupAttr]) )
	      result["ActiveGroup"] = primary_group[config->groupAttr];
      }
      // groups:
      if ( stringp(config->memberAttr) ) {
	  array groups = ({ "sTeam" });
	  string searchstr = "(&(" + config->memberAttr + "=" + identifier + ")"
	      + "(objectClass=" + config->groupClass + "))";
	  mixed ldap_groups = get_module("ldap")->fetch( "", searchstr );
	  if ( mappingp(ldap_groups) )  // single group, make an array out of it:
	      ldap_groups = ({ ldap_groups });

          if ( arrayp(ldap_groups) ) {
            foreach ( ldap_groups, mixed ldap_group ) {
	      if ( !mappingp(ldap_group) ) {
                werror( "ldappersistence: invalid ldap group: %O\n", ldap_group );
                continue;
	      }
	      if ( stringp(ldap_group[config->groupAttr]) )
                groups += ({ ldap_group[config->groupAttr] });
	      else
                werror( "ldappersistence: invalid ldap group name: %O\n", ldap_group[config->groupAttr] );
            }
	  }
	  result["Groups"] = groups;
      }
  }

  user_cache->put( identifier, 1 );  // remember that the user has been looked up

  return result;

  };
  if ( exc != 0 ) werror( "LDAP: lookup_user(\"%s\") : %O\n", identifier, exc );
}

mixed lookup_group(string identifier)
{
  if ( !objectp(ldap) || !ldap->ldap_activated() )
    return UNDEFINED;

  LDAPPERS_LOG( "lookup_group(%s)", identifier );

  if ( !stringp(config->groupAttr) || sizeof(config->groupAttr)<1 )
    return UNDEFINED;

  if ( ! zero_type( group_cache->get( identifier ) ) ) {
    LDAPPERS_LOG( "group from cache: %s", identifier );
    return UNDEFINED;  // let Persistence use the database copy of the group
  }

  mixed data = get_module("ldap")->fetch_group(identifier);
  if ( !mappingp(data) ) return UNDEFINED;

  mixed exc = catch {

  mapping result = ([
    "ldap:dn" : data["dn"],
    "OBJ_DESC" : data[config->descriptionAttr],
  ]);
  // users:
  if ( stringp(config->memberAttr) ) {
    mixed ldap_users = data[config->memberAttr];
    if ( stringp(ldap_users) )  // single user, make an array out of it:
      ldap_users = ({ ldap_users });
    result["Users"] = ldap_users;
  }

  group_cache->put( identifier, 1 );  // remember that the group has been looked up

  return result;

  };
  if ( exc != 0 ) werror( "LDAP: lookup_user(\"%s\") : %O\n", identifier, exc );
}

void update_user ( object user )
{
  //TODO: implement update_user(obj) ?
  //werror("* server/modules/ldappersistence.pike : update_user() not implemented yet\n" );
}

void update_group ( object group )
{
  //TODO: implement update_group(obj) ?
  //werror("* server/modules/ldappersistence.pike : update_group() not implemented yet\n" );
}

/**
 * Returns a user or group object that should receive notifications about
 * noteworthy situations concerning the ldap persistence (e.g. conflicting
 * user entries).
 *
 * @return user or group object of the maintainer to notify
 */
object get_maintainer () {
  object maintainer;
  string maintainerStr = config["adminAccount"];
  if ( stringp(maintainerStr) ) {
    maintainer = USER( maintainerStr );
    if ( !objectp(maintainer) ) maintainer = GROUP( maintainerStr );
  }
  if ( !objectp(maintainer) ) maintainer = GROUP("admin");
  return maintainer;
}
