/*
 * dvpump.h -- DV Pump and Sundry Common Implementations
 * Copyright (C) 2002 Charles Yates <charles.yates@pandora.be>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifndef _KINO_DV_PUMP_
#define _KINO_DV_PUMP_

#include <iostream>
#include <string>
#include <deque>
using std::string;
using std::deque;

#include <pthread.h>
#include "Diagnostics.h"
#include <Threader.h>

/** Template DataPump class.
*/

template < class Item > class DataPump 
{	
	private:
		int size;
		deque < Item * > input;
		deque < Item * > output;
		pthread_mutex_t mutex;
		bool auto_flush;
		pthread_cond_t condition;
		pthread_mutex_t condition_mutex;
		bool terminated;
		double usable;
		bool paused;

	public:
		DataPump( ) :
			size( 0	), 
			auto_flush( false ),
			terminated( false ),
			usable( 0.5 ),
			paused( false )
		{
			pthread_mutex_init( &mutex, NULL );
			pthread_mutex_init( &condition_mutex, NULL );
			pthread_cond_init( &condition, NULL );
		}

		virtual ~DataPump( )
		{
			for ( int index = input.size(); index > 0; index -- )
			{
				delete input[ 0 ];
				input.pop_front( );
			}
			for ( int index = output.size(); index > 0; index -- )
			{
				delete output[ 0 ];
				output.pop_front( );
			}
		
			pthread_mutex_lock( &condition_mutex );
			pthread_cond_broadcast( &condition );
			pthread_mutex_unlock( &condition_mutex );
		
			pthread_mutex_destroy( &mutex );
			pthread_mutex_destroy( &condition_mutex );
			pthread_cond_destroy( &condition );
		}

		virtual bool IsSeekable( )
		{
			return false;
		}
		
		virtual bool IsBlockable( ) 
		{
			return false;
		}
		
		virtual bool TogglePause( )
		{
			paused = !paused;
			if ( paused && !IsBlockable( ) )
				FlushOutput( );
			return paused;
		}

		virtual bool IsPaused( ) const
		{
			return paused;
		}
		
		void SetUsable( double _usable )
		{
			if ( _usable >= 0 && _usable < 0.9 )
				usable = _usable;
		}
		
		void SetPumpSize( int _size )
		{
			if ( _size > size )
			{
				pthread_mutex_lock( &mutex );
				for ( int index = size; index < _size; index ++ )
					input.push_back( new Item( ) );
				size = _size;
				pthread_mutex_unlock( &mutex );
			}
		}
		
		int GetPumpSize( )
		{
			return size;
		}

		int GetInputUsableSize( )
		{
			return ( int )( size * usable );
		}
		
		// This method should be removed
		bool IsInputCritical( )
		{
			return false;
		}
		
		// Methods used to provide input frames
		void SetAutoFlush( bool _auto_flush )
		{
			auto_flush = _auto_flush;
			if ( auto_flush )
				FlushOutput( );
		}
		
		int GetInputAvailable( bool wait = true )
		{
			int available = input.size( );
			while ( wait && available <= GetInputUsableSize( ) && !terminated )
			{
				pthread_mutex_lock( &condition_mutex );
				if ( !terminated )
					pthread_cond_wait( &condition, &condition_mutex );
				pthread_mutex_unlock( &condition_mutex );
				available = input.size( );
			}
			return available;
		}
		
		Item &GetInputFrame( )
		{
			if ( !GetInputAvailable() )
				throw "No input frames available";
			return *input[ 0 ];
		}
		
		void QueueOutputFrame( )
		{
			if ( !IsPaused( ) || output.size( ) == 0 )
			{
				pthread_mutex_lock( &mutex );
				output.push_back( input[ 0 ] );
				input.pop_front( );
				pthread_mutex_unlock( &mutex );
				if ( auto_flush )
					FlushOutput( );
				pthread_mutex_lock( &condition_mutex );
				pthread_cond_broadcast( &condition );
				pthread_mutex_unlock( &condition_mutex );
			}
			else 
			{
				FlushOutput( );
			}
		}

		// Methods used to provide output frames
		int GetOutputAvailable( bool wait = true )
		{
			int size = output.size( );
			while ( wait && size == 0 && !terminated )
			{
				pthread_mutex_lock( &condition_mutex );
				if ( !terminated )
					pthread_cond_wait( &condition, &condition_mutex );
				pthread_mutex_unlock( &condition_mutex );
				size = output.size( );
			}
			return size;
		}
		
		Item &GetOutputFrame( )
		{
			if ( GetOutputAvailable( ) == 0 )
				throw "No output frames available";
			output[ 0 ]->ExtractHeader( );
			return *output[ 0 ];
		}
		
		void QueueInputFrame( )
		{
			if ( !IsPaused( ) )
			{
				pthread_mutex_lock( &mutex );
				if ( output.begin() != output.end() )
				{
					input.push_back( output[ 0 ] );
					output.pop_front( );
				}
				pthread_mutex_unlock( &mutex );
				pthread_mutex_lock( &condition_mutex );
				pthread_cond_broadcast( &condition );
				pthread_mutex_unlock( &condition_mutex );
			}
		}
		
		void FlushOutput( )
		{
			pthread_mutex_lock( &mutex );
			if ( output.begin() != output.end() )
			{
				Item *current = output[ 0 ];
				output.pop_front( );
				for ( int index = output.size(); index > 0; index -- )
				{
					input.push_back( output[ 0 ] );
					output.pop_front( );
				}
				output.push_back( current );
			}
			pthread_mutex_unlock( &mutex );
			pthread_mutex_lock( &condition_mutex );
			pthread_cond_broadcast( &condition );
			pthread_mutex_unlock( &condition_mutex );
		}

		void ClearPump( )
		{
			pthread_mutex_lock( &condition_mutex );
			terminated = true;
			pthread_cond_broadcast( &condition );
			pthread_mutex_unlock( &condition_mutex );
		}
		
		bool PumpIsNotCleared( )
		{
			return !terminated;
		}
};

#include <frame.h>

/** The DVPump class provides a ring buffer for DV Frames. This is a highly 
	important component of the smilutils tool set as it provides the mechanism 
	through which the threads propogate DV frames.
*/

class DVPump : 
	public DataPump < Frame >,
	virtual public DiagnosticsId
{	
};

#endif

