// =============================================================================
// Scilab ( http://www.scilab.org/ ) - This file is part of Scilab
// Copyright (C) 2008 - INRIA - Allan CORNET
// Copyright (C) 2009 - INRIA - Michael Baudin, Antoine Elias
// Copyright (C) 2010 - DIGITEO - Michael Baudin
//
//  This file is distributed under the same license as the Scilab package.
// =============================================================================

//
// <-- JVM NOT MANDATORY -->
// <-- Bugzilla URL -->
//

//
// assert_close --
//   Returns 1 if the two real matrices computed and expected are close,
//   i.e. if the relative distance between computed and expected is lesser than epsilon.
// Arguments
//   computed, expected : the two matrices to compare
//   epsilon : a small number
//
function flag = assert_close ( computed, expected, epsilon )
  if expected==0.0 then
    shift = norm(computed-expected);
  else
    shift = norm(computed-expected)/norm(expected);
  end
  if shift < epsilon then
    flag = 1;
  else
    flag = 0;
  end
  if flag <> 1 then pause,end
endfunction

//
// assert_equal --
//   Returns 1 if the two real matrices computed and expected are equal.
// Arguments
//   computed, expected : the two matrices to compare
//   epsilon : a small number
//
function flag = assert_equal ( computed , expected )
  if ( type(computed) == 1 & type(expected) == 1 ) then
    csize = size (computed,"*")
    esize = size (expected,"*")
    cnonan = find(~isnan(computed))
    enonan = find(~isnan(computed))
    if ( and(cnonan == enonan) & and(computed(enonan) == expected(cnonan)) ) then
      flag = 1;
    else
      flag = 0;
    end
  else
    if computed==expected then
      flag = 1;
    else
      flag = 0;
    end
  end
  if flag <> 1 then pause,end
endfunction

// Bibliography :
// "IEEE Standard for Floating-Point Arithmetic"
// IEEE Std 754-2008
// 29 August 2008
// (Revision of IEEE Std 754-1985)

// Check that properties of double precision floating point numbers
// are the one which are expected.
// Strict equality is required here.
radix = number_properties("radix");
assert_equal ( radix , 2 );
p = number_properties("digits");
assert_equal ( p , 53 );
eps = number_properties("eps");
assert_equal ( eps , 1.11022302462515650e-016 );
assert_equal ( %eps , 2.22044604925031310e-016 );
huge = number_properties("huge");
assert_equal ( huge , 1.797693134862315708e+308 );
tiny = number_properties("tiny");
assert_equal ( tiny , 2.22507385850720140e-308 );
denorm = number_properties("denorm");
assert_equal ( denorm , %t );
tiniest = number_properties("tiniest");
assert_equal ( tiniest , 4.94065645841246540e-324 );
minexp = number_properties("minexp");
assert_equal ( minexp , -1021 );
maxexp = number_properties("maxexp");
assert_equal ( maxexp , 1024 );

// Check that elementary operators +,-,*,/,sqrt and modulo
// satisfy basic IEEE requirements.

expectedadd = [
-%inf 	-%inf 	-%inf 	-%inf 	-%inf 	%nan 	%nan
-%inf 	-2 	-1 	-1 	0 	%inf 	%nan
-%inf 	-1 	-0 	0 	1 	%inf 	%nan
-%inf 	-1 	0 	0 	1 	%inf 	%nan
-%inf 	0 	1 	1 	2 	%inf 	%nan
%nan 	%inf 	%inf 	%inf 	%inf 	%inf 	%nan
%nan 	%nan 	%nan 	%nan 	%nan 	%nan 	%nan];

expectedminus = [
%nan 	-%inf 	-%inf 	-%inf 	-%inf 	-%inf 	%nan
%inf 	0 	-1 	-1 	-2 	-%inf 	%nan
%inf 	1 	0 	-0 	-1 	-%inf 	%nan
%inf 	1 	0 	0 	-1 	-%inf 	%nan
%inf 	2 	1 	1 	0 	-%inf 	%nan
%inf 	%inf 	%inf 	%inf 	%inf 	%nan 	%nan
%nan 	%nan 	%nan 	%nan 	%nan 	%nan 	%nan];

expectedmult = [
%inf 	%inf 	%nan 	%nan 	-%inf 	-%inf 	%nan
%inf 	1 	0 	-0 	-1 	-%inf 	%nan
%nan 	0 	0 	-0 	-0 	%nan 	%nan
%nan 	-0 	-0 	0 	0 	%nan 	%nan
-%inf 	-1 	-0 	0 	1 	%inf 	%nan
-%inf 	-%inf 	%nan 	%nan 	%inf 	%inf 	%nan
%nan 	%nan 	%nan 	%nan 	%nan 	%nan 	%nan];

expecteddiv = [
%nan 	%inf 	%inf 	-%inf 	-%inf 	%nan 	%nan
0 	1 	%inf 	-%inf 	-1 	-0 	%nan
0 	0 	%nan 	%nan 	-0 	-0 	%nan
-0 	-0 	%nan 	%nan 	0 	0 	%nan
-0 	-1 	-%inf 	%inf 	1 	0 	%nan
%nan 	-%inf 	-%inf 	%inf 	%inf 	%nan 	%nan
%nan 	%nan 	%nan 	%nan 	%nan 	%nan 	%nan];

// See http://bugzilla.scilab.org/show_bug.cgi?id=2409
// We must use imult(%inf) to get the mathematical number i * inf
// since %i * %inf produces imult(%inf), as expected by the 
// intermediate multiplication 0*%inf = %nan.
expectedsqrt = [
imult(%inf)
%i
0
0
1.
%inf
%nan];

expectedmodulo = [
%nan  %nan  -%inf -%inf  %nan  %nan  %nan
%nan     0.   -1.    -1.   0.  %nan  %nan
%nan     0.    0.     0.   0.  %nan  %nan
%nan     0.    0.     0.   0.  %nan  %nan
%nan     0.    1.     1.   0.  %nan  %nan
%nan  %nan   %inf  %inf  %nan  %nan  %nan
%nan  %nan   %nan  %nan  %nan  %nan  %nan];

left=[-%inf,-1,-0,+0,1,%inf,%nan];
right=left;
n=size(left,2);

computedadd = zeros(n,n);
computedminus = zeros(n,n);
computedmult = zeros(n,n);
computeddiv = zeros(n,n);
computedsqrt = zeros(n);
computedsqrt = zeros(n);
computedmodulo = zeros(n,n);
ieee(2)
for i=1:n
  l=left(i);
  computedsqrt (i) = sqrt(l);
  for j=1:n
    r=right(j);
    computedadd (i,j)=l+r;
    computedminus (i,j)=l-r;
    computedmult (i,j)=l*r;
    computeddiv (i,j)=l/r;
    computedmodulo (i,j) = modulo(l , r);
  end
end

assert_equal ( computedadd    , expectedadd );
assert_equal ( computedminus  , expectedminus );
assert_equal ( computedmult   , expectedmult );
assert_equal ( computeddiv    , expecteddiv );
assert_equal ( computedsqrt   , expectedsqrt );
assert_equal ( computedmodulo , expectedmodulo );


