# -*- coding: UTF-8 -*-

# Phatch - Photo Batch Processor
# Copyright (C) 2007-2008  www.stani.be
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see http://www.gnu.org/licenses/

import os, optparse, sys, urllib
from data.info import INFO
from core.config import load_locale

VERSION = "%(name)s %(version)s"%INFO

def fix_path(path):
    if path[:7] == 'file://': return urllib.unquote(path[7:])
    return path

def parse_locale(config_paths):
    if '-l' in sys.argv:
        index       = sys.argv.index('-l')+1
        if index >= len(sys.argv):
            sys.exit('Please specify locale language after "-l".')
        canonical   = sys.argv[index]
    else:
        canonical   = 'default'
    load_locale('phatch',config_paths["PHATCH_LOCALE_PATH"],canonical)
    
def parse_options():
    parser  = optparse.OptionParser(
        usage   = """
  %(name)s [actionlist]
  %(name)s [options] [actionlist] [image folders/files/urls]
  %(name)s --inspect [image files/urls]
  %(name)s --droplet [actionlist/recent] [image files/urls]"""%INFO+"""

%s:
  phatch action_list.phatch
  phatch --verbose --recursive action_list.phatch image_file.png image_folder
  phatch --inspect image_file.jpg
  phatch --droplet recent"""%_('Examples'), 
        version = VERSION,
    )
    parser.add_option("-c", "--console", action="store_true", 
        dest    = "console", 
        default = False,
        help    = _("Run %s as console program without a gui")%INFO['name'])
    parser.add_option("-d", "--droplet", action="store_true", 
        dest    = "droplet", 
        default = False,
        help    = _("Run %s as a gui droplet")%INFO['name'])
    parser.add_option("-f", "--force", action="store_false", 
        dest    = "stop_for_errors", 
        default = True,
        help    = _("Ignore errors"))
    parser.add_option("-i", "--interactive", action="store_true", 
        dest    = "interactive", 
        default = False,
        help    = _("Interactive"))
    parser.add_option("-l", action="store", 
        dest    = "locale",
        default = 'default',
        type    = "string", 
        help    = _("Specify locale language (for example en or en_GB)")) 
    parser.add_option("-m", "--metadata", action="store_true", 
        dest    = "save_metadata", 
        default = False,
        help    = _("Save metadata (requires exif & iptc plugin)"))
    parser.add_option("-n", "--inspect", action="store_true", 
        dest    = "image_inspector", 
        default = False,
        help    = _("Inspect metadata (requires exif & iptc plugin)"))
    parser.add_option("-o", "--overwrite", action="store_true", 
        dest    = "overwrite_existing_images", 
        default = False,
        help    = _("Overwrite existing images"))
    parser.add_option("-r", "--recursive", action="store_true", 
        dest    = "recursive", 
        default = False,
        help    = _("Include all subfolders"))
    parser.add_option("-s", "--strict", action="store_false", 
        dest    = "create_missing_folders", 
        default = True,
        help    = _("Do not create missing folders"))
    parser.add_option("-t", "--trust", action="store_false", 
        dest    = "check_images_first", 
        default = True,
        help    = _("Do not check images first"))
    parser.add_option("-v", "--verbose", action="store_true", 
        dest    = "verbose", 
        default = False,
        help    = _("Verbose"))
    options, paths = parser.parse_args()
    paths   = [fix_path(path) for path in paths if path and path[0]!= '%']
    return options, paths
    
def reexec_with_pythonw(f=None):
    """'pythonw' needs to be called for any wxPython app 
    to run from the command line on Mac Os X."""
    if sys.platform == 'darwin' and\
           not sys.executable.endswith('/Python'):
        sys.stderr.write('re-executing using pythonw')
        if not f: f = __file__
        os.execvp('pythonw',['pythonw',f] + sys.argv[1:])
        
def console(config_paths):
    main(config_paths,app_file=None,gui=True)
    
def _gui(app_file,paths,settings):
    reexec_with_pythonw(app_file) #ensure pythonw for mac
    from pyWx import gui
    if paths:
        actionlist = paths[0]
    else:
        actionlist = ''
    gui.main(settings,actionlist)

def _inspect(app_file,paths):
    reexec_with_pythonw(app_file) #ensure pythonw for mac
    from pyWx import gui
    gui.inspect(paths)

def _droplet(app_file,paths,settings):
    reexec_with_pythonw(app_file) #ensure pythonw for mac
    from pyWx import gui
    gui.drop(actionlist=paths[0],paths=paths[1:],settings=settings)

def ext(path):
    return os.path.splitext(path)[1].lower()

def _console(paths,settings):
    from console import console
    if paths and ext(paths[0]) == INFO['extension']:
        console.main(actionlist=paths[0],paths=paths[1:],settings=settings)
    else:
        console.main(actionlist='',paths=paths,settings=settings)
        
def main(config_paths,app_file):
    """init should be called first!"""
    parse_locale(config_paths)
    options, paths  = parse_options()
    from core.settings import create_settings
    settings        = create_settings(config_paths,options)
    if settings['image_inspector']:
        _inspect(app_file,paths)
        return
    if len(paths) == 1 and ext(paths[0])!= INFO['extension']:
        settings['droplet'] = True
        paths.insert(0,'recent')
    if settings['droplet']:
        _droplet(app_file,paths,settings)
    elif len(paths)>1 or settings['console'] or settings['interactive']:
        _console(paths,settings)
    else:
        _gui(app_file,paths,settings)
        
if __name__=='__main__':
    main()
