#!/bin/sh
# EBS Snapshot script
#
# Copyright 2011 Canonical
#
# This script will take snapshots of the EBS volumes attached to this instance at regular intervals
# Requires: euca2ools and eucarc config file XFS filesystem

KEEPSNAPSHOTS=3

## Required variables
while getopts "t:c:l:h" flag
do
    case "$flag" in
        t)
            # Define the cloud type
            if [ -z "$OPTARG" ]
            then
                CLOUDTYPE="UEC" # Valid options are UEC or EC2
            else
                CLOUDTYPE="$OPTARG"
            fi
            ;;
        c)
            # Define cloudrc location
            if [ -z "$OPTARG" ]
            then
                CLOUDRC="/home/ubuntu/.euca/eucarc"
            else
                CLOUDRC="$OPTARG"
            fi
            ;;
        l)
            # Define local mysql credentials
            if [ -z "$OPTARG" ]
            then
                LOCAL_MYSQL_USER="root"
                LOCAL_MYSQL_PASS="insecure"
            else
                LOCAL_MYSQL_USER=`echo $OPTARG | awk -F":" '{print $1}'`
                LOCAL_MYSQL_PASS=`echo $OPTARG | awk -F":" '{print $2}'`
            fi
            ;;
        h)
            # Show help
            echo "Usage: $0 -t [cloud_type] -c [cloudrc_location] -l [localmysql_credentials]"
            echo "Parameters:"
            echo " t: Cloud type, can be UEC or EC2"
            echo " c: Cloudrc file, is the file where all your credentials can be sourced from"
            echo " l: Local MySQL root credentials (in user:pass format)"
            echo "Example:"
            echo "$0 -t UEC -c /home/ubuntu/.euca/eucarc -l root:insecure"
            exit 2
            ;;
    esac
done

## Binaries
case "$CLOUDTYPE" in
    UEC)
        INSTANCE_DESCRIBE="/usr/bin/euca-describe-instances"
        VOLUME_DESCRIBE="/usr/bin/euca-describe-volumes"
        VOLUME_CREATE="/usr/bin/euca-create-volume"
        SNAPSHOT_DESCRIBE="/usr/bin/euca-describe-snapshots"
        SNAPSHOT_CREATE="/usr/bin/euca-create-snapshot"
        SNAPSHOT_DELETE="/usr/bin/euca-delete-snapshot"
        TOOLSNAME="euca2ools"
        ;;
    EC2)
        INSTANCE_DESCRIBE="/usr/bin/ec2-describe-instances"
        VOLUME_DESCRIBE="/usr/bin/ec2-describe-volumes"
        VOLUME_CREATE="/usr/bin/ec2-create-volume"
        SNAPSHOT_DESCRIBE="/usr/bin/ec2-describe-snapshots"
        SNAPSHOT_CREATE="/usr/bin/ec2-create-snapshot"
        SNAPSHOT_DELETE="/usr/bin/ec2-delete-snapshot"
        TOOLSNAME="ec2-api-tools"
        ;;
    *)
        echo "ERROR: No valid cloud type found, please edit the script configuration"
        exit 1
        ;;
esac

## Checks before execution

echo "`date`- Preflight checks"

# Check for euca2ools presence
if [ ! -f "$INSTANCE_DESCRIBE" ]
then
    echo "ERROR: Missing $TOOLSNAME, please make sure they're installed"
    exit 1
fi

# Check that mysql is running
if [ $(pidof mysqld | wc -l) = 0 ]
then
    echo "ERROR: mysql is not running"
    exit 1
fi

# Check that we have a binlog
if [ "${LOCAL_MYSQL_PASS}" = "" ]
then
	MYSQLBINCHECK=`/usr/bin/mysql -u${LOCAL_MYSQL_USER} -e "show master status;" | wc -l`
else
	MYSQLBINCHECK=`/usr/bin/mysql -u${LOCAL_MYSQL_USER} -p ${LOCAL_MYSQL_PASS} -e "show master status;" | wc -l`
fi
if [ $MYSQLBINCHECK = 0 ]
then
    echo "ERROR: mysql has no binlog"
    exit 1
fi

# Source cloud config
echo "`date` - Importing cloud credentials"
. $CLOUDRC
MYIP=`/usr/bin/facter ipaddress`
echo "`date` - Recovering Instance ID"
MYINSTANCE=`$INSTANCE_DESCRIBE | grep $MYIP | awk '{print $2}'`
if [ -z "$MYINSTANCE" ]
then
    echo "ERROR: Missing credentials, please make sure the credentials file is properly referenced in the config"
    exit 1
fi
echo "`date` - Recovering volume ID"
MYVOLUME=`$VOLUME_DESCRIBE | grep ${MYINSTANCE} | awk '{print $2}'`

# Get available snapshots
echo "`date` - Purging old snapshots"
PURGE_SNAPSHOTS=`$SNAPSHOT_DESCRIBE | grep ${MYVOLUME} | grep -i completed | sort -k5 -r | awk '{print $2}' | tail -n+${KEEPSNAPSHOTS} | tr "\n" " "`
# We need to purge snapshots after the cut through
for PURGED_SNAPSHOT in $PURGE_SNAPSHOTS
do
    $SNAPSHOT_DELETE $PURGED_SNAPSHOT
done

# Get filesystem mountpoint 
echo "`date`- Finding volume mountpoint"
MYMOUNTPOINT=`/bin/mount -l | grep xfs | awk '{print $3}'`
if [ -z "$MYMOUNTPOINT" ]
then
    echo "ERROR: The filesystem to snapshot must be XFS for consistency reasons"
    exit 1
fi

# Sync FS before freezing, helps create the snapshot faster
echo "`date` - Freezing filesystem"
/bin/sync

# Freeze filesystem and create new snapshot
/usr/sbin/xfs_freeze -u $MYMOUNTPOINT 2> /dev/null
FREEZEEXIT=1
while [ $FREEZEEXIT != 0 ]
do
     sleep 1
     /usr/sbin/xfs_freeze -f $MYMOUNTPOINT
     FREEZEEXIT=`echo $?`
done

# Take snapshot
echo "`date` - Taking snapshot"
NEWSNAPSHOT=`$SNAPSHOT_CREATE $MYVOLUME | awk '{print $2}'`
if [ $(echo $NEWSNAPSHOT | grep snap | wc -l) -eq 0 ]
then
    echo "ERROR: Problem taking snapshot, please check your cloud limits"
    /usr/sbin/xfs_freeze -u $MYMOUNTPOINT
    exit 1
fi

# While were at it, recover precious bin information
echo "`date` - Getting master status"
if [ "$LOCAL_MYSQL_PASSWORD" = "" ]
then
	MASTER_PARAMETERS=`/usr/bin/mysql -Bs -u${LOCAL_MYSQL_USER} -e "show master status;" | awk '{print $1"|"$2}'`
else
	MASTER_PARAMETERS=`/usr/bin/mysql -Bs -u${LOCAL_MYSQL_USER} -p${LOCAL_MYSQL_PASSWORD} -e "show master status;" | awk '{print $1"|"$2}'`
fi

# Sleep while snapshot is created
echo "`date` - Sleep while snapshot finishes..."
while [ $(${SNAPSHOT_DESCRIBE} | grep $NEWSNAPSHOT | grep -i pending | wc -l) -eq 1 ]
do
    sleep 2
done

# Unfreeze
echo "`date` - Unfreezing filesystem"
/usr/sbin/xfs_freeze -u $MYMOUNTPOINT

# Write snapshot info to database
echo "`date`- Storing snapshot information in Database"
MYSQL_BIN_FILE=`echo $MASTER_PARAMETERS | awk -F"|" '{print $1}'`
MYSQL_BIN_POSITION=`echo $MASTER_PARAMETERS | awk -F"|" '{print $2}'`

if [ "$LOCAL_MYSQL_PASSWORD" = "" ]
then
	# We need to create the table
	/usr/bin/mysql -u${LOCAL_MYSQL_USER} mysql -e "create table if not exists replication (date datetime,snapshot varchar(20),binlog_file varchar(30),binlog_position int(10));"
	#Insert new data into the table
	/usr/bin/mysql -u${LOCAL_MYSQL_USER} mysql -e "insert into replication values (now(),'$NEWSNAPSHOT','$MYSQL_BIN_FILE','$MYSQL_BIN_POSITION');"
else
	# We need to create the table
	/usr/bin/mysql -u${LOCAL_MYSQL_USER} -p${LOCAL_MYSQL_PASSWORD} mysql -e "create table if not exists replication (date datetime,snapshot varchar(20),binlog_file varchar(30),binlog_position int(10));"
	#Insert new data into the table
	/usr/bin/mysql -u${LOCAL_MYSQL_USER} -p${LOCAL_MYSQL_PASSWORD} mysql -e "insert into replication values (now(),'$NEWSNAPSHOT','$MYSQL_BIN_FILE','$MYSQL_BIN_POSITION');"
fi

# All operations okay
echo "`date` - Finished"
exit 0

