///////////////////////////////////////////////////////////////////////////
//
// Copyright (c) 2012 DreamWorks Animation LLC
//
// All rights reserved. This software is distributed under the
// Mozilla Public License 2.0 ( http://www.mozilla.org/MPL/2.0/ )
//
// Redistributions of source code must retain the above copyright
// and license notice and the following restrictions and disclaimer.
//
// *     Neither the name of DreamWorks Animation nor the names of
// its contributors may be used to endorse or promote products derived
// from this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
// OWNER OR CONTRIBUTORS BE LIABLE FOR ANY INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
// IN NO EVENT SHALL THE COPYRIGHT HOLDERS' AND CONTRIBUTORS' AGGREGATE
// LIABILITY FOR ALL CLAIMS REGARDLESS OF THEIR BASIS EXCEED US$250.00.
//
///////////////////////////////////////////////////////////////////////////

#include <cppunit/extensions/HelperMacros.h>
#include <openvdb/Exceptions.h>
#include <openvdb/openvdb.h>
#include <openvdb/math/Frustum.h>
#include <openvdb/math/BBox.h>
#include <openvdb/Types.h>


namespace {
const float TOLERANCE = 1.0e-6;
}


class TestFrustum : public CppUnit::TestCase
{
public:
    CPPUNIT_TEST_SUITE(TestFrustum);
    CPPUNIT_TEST(testFrustum);
    CPPUNIT_TEST_SUITE_END();

    void testFrustum();
};

CPPUNIT_TEST_SUITE_REGISTRATION(TestFrustum);


void
TestFrustum::testFrustum()
{
    typedef openvdb::Real                Real;
    typedef openvdb::Vec3R               Vec3R;
    typedef openvdb::math::Ray<Real>     RayType;
    typedef openvdb::math::Frustum<Real> FrustumType;
    typedef openvdb::math::BBox<Real>    BBoxType;
    typedef FrustumType::PlaneType       PlaneType;

    // Cartesian frame-of-reference: X:right, Y:up, Z:out of screen!!
    openvdb::Real t0,t1;

    {   //empty
        FrustumType F;
        //std::cerr << F;
        CPPUNIT_ASSERT(!F.isValid());
    }
    {   // Simple cube frustum from 6 planes
        const Vec3R X(1,0,0), Y(0,1,0), Z(0,0,1), min(0,0,0), max(1,1,1);
        FrustumType F(PlaneType(max, Z),//  front (+z)
                      PlaneType(min,-Z),//  back  (-z)
                      PlaneType(min,-X),//  left  (-x)
                      PlaneType(max, X),//  right (+x)
                      PlaneType(max, Y),//  up    (+y)
                      PlaneType(min,-Y));// down  (-y)
        //std::cerr << std::endl << F;
        CPPUNIT_ASSERT(F.isValid());
        CPPUNIT_ASSERT(F.isInside(Vec3R(0.5,0.5,0.5)));
        CPPUNIT_ASSERT(F.isOutside(Vec3R(1.5,0.5,0.5)));
        CPPUNIT_ASSERT(F.isOutside(Vec3R(0.5,1.5,0.5)));
        CPPUNIT_ASSERT(F.isOutside(Vec3R(0.5,0.5,1.5)));

        const Vec3R orig(0.5,0.5,2), dir(0,0,-1);//ray-direction is into screen!
        RayType R(orig,dir);
        CPPUNIT_ASSERT(F.intersects(R));
        CPPUNIT_ASSERT(F.intersects(R,t0,t1)==2);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(1.0, t0, TOLERANCE);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(2.0, t1, TOLERANCE);

        // one hit with origin inside frustum
        R.setMinTime(1.5);
        CPPUNIT_ASSERT(F.intersects(R));
        CPPUNIT_ASSERT(F.intersects(R,t0,t1)==1);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(1.5, t0, TOLERANCE);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(2.0, t1, TOLERANCE);
        //fprintf(stderr,"t0=%f t1=%f\n",t0,t1);

        // completely inside frustum -> no hits
        R.setMaxTime(1.75);
        CPPUNIT_ASSERT(F.intersects(R));//inside counts as hit!
        CPPUNIT_ASSERT(F.intersects(R,t0,t1)==-1);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(1.5, t0, TOLERANCE);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(1.75, t1, TOLERANCE);

        // one hit with origin outside frustum
        R.setMinTime(0.0);
        R.setMaxTime(1.5);
        CPPUNIT_ASSERT(F.intersects(R));
        CPPUNIT_ASSERT(F.intersects(R,t0,t1)==1);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(1.0, t0, TOLERANCE);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(1.5, t1, TOLERANCE);

        // completely outside frustum => no hits
        R.setMinTime(0.0);
        R.setMaxTime(0.5);
        CPPUNIT_ASSERT(!F.intersects(R));
        CPPUNIT_ASSERT(F.intersects(R,t0,t1)==0);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(0.0, t0, TOLERANCE);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(0.5, t1, TOLERANCE);

        // completely outside frustum => no hits
        R.setMinTime(2.5);
        R.setMaxTime(3.5);
        CPPUNIT_ASSERT(!F.intersects(R));
        CPPUNIT_ASSERT(F.intersects(R,t0,t1)==0);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(2.5, t0, TOLERANCE);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(3.5, t1, TOLERANCE);
    }
    {   // Second simple block frustum from 6 planes
        const Vec3R X(1,0,0), Y(0,1,0), Z(0,0,1), min(8,8,8), max(16,16,16);
        FrustumType F(PlaneType(max, Z),//  front (+z)
                      PlaneType(min,-Z),//  back  (-z)
                      PlaneType(min,-X),//  left  (-x)
                      PlaneType(max, X),//  right (+x)
                      PlaneType(max, Y),//  up    (+y)
                      PlaneType(min,-Y));// down  (-y)
        //std::cerr << F << std::endl;
        CPPUNIT_ASSERT(F.isValid());
        CPPUNIT_ASSERT(F.isInside(Vec3R(9,9,9)));
        //for (int i=0; i<6; ++i) {
        //    std::cerr << "Outside plane["<<i<<"]: "
        //        <<(F.getPlane(i).isOutside(Vec3R(9,1,1))?"TRUE":"FALSE")<<std::endl;
        //}

        CPPUNIT_ASSERT(F.isOutside(Vec3R(9,1,1)));

        const Vec3R orig(9,1,1), dir(1,0,0);//ray-direction is out of screen!
        RayType R(orig,dir);
        CPPUNIT_ASSERT(!F.intersects(R));
        //CPPUNIT_ASSERT(F.intersects(R,t0,t1)==2);
        //CPPUNIT_ASSERT_DOUBLES_EQUAL(1.0, t0, TOLERANCE);
        //CPPUNIT_ASSERT_DOUBLES_EQUAL(2.0, t1, TOLERANCE);

        // one hit with origin inside frustum
        //R.setMinTime(1.5);
        //CPPUNIT_ASSERT(F.intersects(R));
        //CPPUNIT_ASSERT(F.intersects(R,t0,t1)==1);
        //CPPUNIT_ASSERT_DOUBLES_EQUAL(1.5, t0, TOLERANCE);
        //CPPUNIT_ASSERT_DOUBLES_EQUAL(2.0, t1, TOLERANCE);
        //fprintf(stderr,"t0=%f t1=%f\n",t0,t1);

        // completely inside frustum -> no hits
        //R.setMaxTime(1.75);
        //CPPUNIT_ASSERT(F.intersects(R));//inside counts as hit!
        //CPPUNIT_ASSERT(F.intersects(R,t0,t1)==-1);
        //CPPUNIT_ASSERT_DOUBLES_EQUAL(1.5, t0, TOLERANCE);
        //CPPUNIT_ASSERT_DOUBLES_EQUAL(1.75, t1, TOLERANCE);

        // one hit with origin outside frustum
        //R.setMinTime(0.0);
        //R.setMaxTime(1.5);
        //CPPUNIT_ASSERT(F.intersects(R));
        //CPPUNIT_ASSERT(F.intersects(R,t0,t1)==1);
        //CPPUNIT_ASSERT_DOUBLES_EQUAL(1.0, t0, TOLERANCE);
        //CPPUNIT_ASSERT_DOUBLES_EQUAL(1.5, t1, TOLERANCE);

        // completely outside frustum => no hits
        //R.setMinTime(0.0);
        //R.setMaxTime(0.5);
        //CPPUNIT_ASSERT(!F.intersects(R));
        //CPPUNIT_ASSERT(F.intersects(R,t0,t1)==0);
        //CPPUNIT_ASSERT_DOUBLES_EQUAL(0.0, t0, TOLERANCE);
        //CPPUNIT_ASSERT_DOUBLES_EQUAL(0.5, t1, TOLERANCE);

        // completely outside frustum => no hits
        //R.setMinTime(2.5);
        //R.setMaxTime(3.5);
        //CPPUNIT_ASSERT(!F.intersects(R));
        //CPPUNIT_ASSERT(F.intersects(R,t0,t1)==0);
        //CPPUNIT_ASSERT_DOUBLES_EQUAL(2.5, t0, TOLERANCE);
        //CPPUNIT_ASSERT_DOUBLES_EQUAL(3.5, t1, TOLERANCE);
    }
    {   //Frustum from camera parameters
        const Vec3R position(0,0,0);//camera position
        const Vec3R direction(0,0,-1);//unit-vec from camera to mid of near plane
        const Vec3R up(0,0.5,0);//from midt of near-plane to upper edge of near plane
        const Vec3R left(-0.5,0,0);//from midt of near-plane to left edge of near plane
        const Real  z_near=1,z_far=2;
        FrustumType F(position,direction,up,left,z_near,z_far);
        RayType R(position,direction);
        //std::cerr << std::endl << F;
        CPPUNIT_ASSERT(F.isValid());
        CPPUNIT_ASSERT(F.intersects(R));
        CPPUNIT_ASSERT(F.intersects(R,t0,t1));
        //fprintf(stderr,"t0=%f t1=%f\n",t0,t1);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(1.0, t0, TOLERANCE);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(2.0, t1, TOLERANCE);
        CPPUNIT_ASSERT(F.isInside(Vec3R(0,0,-1.5)));
        CPPUNIT_ASSERT(F.isOutside(Vec3R(1.5,0.5,0.5)));
        CPPUNIT_ASSERT(F.isOutside(Vec3R(0.5,1.5,0.5)));
        CPPUNIT_ASSERT(F.isOutside(Vec3R(0.5,0.5,1.5)));
    }
    {   //Frustum from grid with LinearTransform
        openvdb::FloatGrid::Ptr grid = openvdb::createGrid<openvdb::FloatGrid>(/*bg=*/5.0);
        CPPUNIT_ASSERT(grid->transformPtr());

        FrustumType F(grid->transform());
        //std::cerr << std::endl << F;

        CPPUNIT_ASSERT(F.isValid());
        CPPUNIT_ASSERT(F.isInside(Vec3R(0.5,0.5,0.5)));
        CPPUNIT_ASSERT(F.isOutside(Vec3R(1.5,0.5,0.5)));
        CPPUNIT_ASSERT(F.isOutside(Vec3R(0.5,1.5,0.5)));
        CPPUNIT_ASSERT(F.isOutside(Vec3R(0.5,0.5,1.5)));

        const Vec3R orig(0.5,0.5,2), dir(0,0,-1);//ray-direction is into screen!
        RayType R(orig,dir);
        CPPUNIT_ASSERT(F.intersects(R));
        CPPUNIT_ASSERT(F.intersects(R,t0,t1)==2);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(1.0, t0, TOLERANCE);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(2.0, t1, TOLERANCE);

        // one hit with origin inside frustum
        R.setMinTime(1.5);
        CPPUNIT_ASSERT(F.intersects(R));
        CPPUNIT_ASSERT(F.intersects(R,t0,t1)==1);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(1.5, t0, TOLERANCE);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(2.0, t1, TOLERANCE);
        //fprintf(stderr,"t0=%f t1=%f\n",t0,t1);


        // completely inside frustum -> no hits
        R.setMaxTime(1.75);
        CPPUNIT_ASSERT(F.intersects(R));//inside counts as hit!
        CPPUNIT_ASSERT(F.intersects(R,t0,t1)==-1);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(1.5, t0, TOLERANCE);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(1.75, t1, TOLERANCE);

        // one hit with origin outside frustum
        R.setMinTime(0.0);
        R.setMaxTime(1.5);
        CPPUNIT_ASSERT(F.intersects(R));
        CPPUNIT_ASSERT(F.intersects(R,t0,t1)==1);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(1.0, t0, TOLERANCE);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(1.5, t1, TOLERANCE);

        // completely outside frustum => no hits
        R.setMinTime(0.0);
        R.setMaxTime(0.5);
        CPPUNIT_ASSERT(!F.intersects(R));
        CPPUNIT_ASSERT(F.intersects(R,t0,t1)==0);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(0.0, t0, TOLERANCE);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(0.5, t1, TOLERANCE);

        // completely outside frustum => no hits
        R.setMinTime(2.5);
        R.setMaxTime(3.5);
        CPPUNIT_ASSERT(!F.intersects(R));
        CPPUNIT_ASSERT(F.intersects(R,t0,t1)==0);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(2.5, t0, TOLERANCE);
        CPPUNIT_ASSERT_DOUBLES_EQUAL(3.5, t1, TOLERANCE);
    }
}

// Copyright (c) 2012 DreamWorks Animation LLC
// All rights reserved. This software is distributed under the
// Mozilla Public License 2.0 ( http://www.mozilla.org/MPL/2.0/ )
