///////////////////////////////////////////////////////////////////////////
//
// Copyright (c) 2012 DreamWorks Animation LLC
//
// All rights reserved. This software is distributed under the
// Mozilla Public License 2.0 ( http://www.mozilla.org/MPL/2.0/ )
//
// Redistributions of source code must retain the above copyright
// and license notice and the following restrictions and disclaimer.
//
// *     Neither the name of DreamWorks Animation nor the names of
// its contributors may be used to endorse or promote products derived
// from this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
// OWNER OR CONTRIBUTORS BE LIABLE FOR ANY INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
// IN NO EVENT SHALL THE COPYRIGHT HOLDERS' AND CONTRIBUTORS' AGGREGATE
// LIABILITY FOR ALL CLAIMS REGARDLESS OF THEIR BASIS EXCEED US$250.00.
//
///////////////////////////////////////////////////////////////////////////
//
/// @author Ken Museth
/// @file Normalize.h

#ifndef OPENVDB_TOOLS_NORMALIZE_HAS_BEEN_INCLUDED
#define OPENVDB_TOOLS_NORMALIZE_HAS_BEEN_INCLUDED

#include <openvdb/Grid.h>
#include <openvdb/math/Math.h>
#include <tbb/parallel_for.h>


namespace openvdb {
OPENVDB_USE_VERSION_NAMESPACE
namespace OPENVDB_VERSION_NAME {
namespace tools {

/// @brief Normalizes a vector VDB volume
template<typename InGridT>
class Normalize
{
public:
    typedef InGridT                             GridType;
    typedef typename GridType::ValueType        Vec3Type;
    typedef typename Vec3Type::value_type       Real;
    typedef typename GridType::ConstAccessor    Accessor;
    typedef typename GridType::TreeType         TreeType;
    typedef typename TreeType::LeafNodeType     LeafType;
    typedef typename tree::IteratorRange<typename TreeType::LeafCIter> RangeType;
    typedef GridType                            OutGridType;

    Normalize(const GridType& grid):
        mInputGrid(&grid),
        mOutputGrid(grid.copy()),
        mAccessor(mInputGrid->getConstAccessor())
    {
    }

    /// Copy constructor called by tbb::parallel_for threads
    Normalize(const Normalize& other):
        mInputGrid(other.mInputGrid),
        mOutputGrid(other.mOutputGrid),
        mAccessor(mInputGrid->getConstAccessor()) // local accessor
    {
    }

    ~Normalize() {}

    Vec3Type operator()(const Coord& xyz) const
    {
        Vec3Type vec = mAccessor.getValue(xyz);
        if (!vec.normalize(/*eps=*/Real(1.0e-7))) vec.setZero();
        return vec;
    }

    typename OutGridType::Ptr process(bool threaded = true)
    {
        Vec3Type bg = mInputGrid->background();
        if (!bg.normalize(/*eps=*/Real(1.0e-7))) bg.setZero();
        mOutputGrid->setBackground(bg);
        RangeType range(mInputGrid->tree().cbeginLeaf());
        if (threaded) {
            tbb::parallel_for(range,*this);
        } else {
            (*this)(range);
        }
        return mOutputGrid;
    }

    void operator()(RangeType& range) const
    {
        Vec3Type vec;
        typename OutGridType::Accessor outAccessor = mOutputGrid->getAccessor();
        for ( ; range; ++range) {
            for (typename LeafType::ValueOnCIter v = range.iterator()->beginValueOn(); v; ++v) {
                const Coord xyz = v.getCoord();
                outAccessor.setValue(xyz, (*this)(xyz));
            }
        }
    }

protected:
    const GridType*             mInputGrid;
    typename OutGridType::Ptr   mOutputGrid;
    Accessor                    mAccessor;
}; // end of Normalize class

} // namespace tools
} // namespace OPENVDB_VERSION_NAME
} // namespace openvdb

#endif // OPENVDB_TOOLS_NORMALIZE_HAS_BEEN_INCLUDED

// Copyright (c) 2012 DreamWorks Animation LLC
// All rights reserved. This software is distributed under the
// Mozilla Public License 2.0 ( http://www.mozilla.org/MPL/2.0/ )
