///////////////////////////////////////////////////////////////////////////
//
// Copyright (c) 2012 DreamWorks Animation LLC
//
// All rights reserved. This software is distributed under the
// Mozilla Public License 2.0 ( http://www.mozilla.org/MPL/2.0/ )
//
// Redistributions of source code must retain the above copyright
// and license notice and the following restrictions and disclaimer.
//
// *     Neither the name of DreamWorks Animation nor the names of
// its contributors may be used to endorse or promote products derived
// from this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
// OWNER OR CONTRIBUTORS BE LIABLE FOR ANY INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
// IN NO EVENT SHALL THE COPYRIGHT HOLDERS' AND CONTRIBUTORS' AGGREGATE
// LIABILITY FOR ALL CLAIMS REGARDLESS OF THEIR BASIS EXCEED US$250.00.
//
///////////////////////////////////////////////////////////////////////////

#ifndef OPENVDB_TOOLS_MESH_TO_VOLUME_HAS_BEEN_INCLUDED
#define OPENVDB_TOOLS_MESH_TO_VOLUME_HAS_BEEN_INCLUDED


#include <openvdb/Types.h>
#include <openvdb/math/FiniteDifference.h>
#include <openvdb/math/Operators.h>
#include <openvdb/math/Proximity.h>
#include <openvdb/tools/LevelSetUtil.h>
#include <openvdb/util/Util.h>

#include <tbb/blocked_range.h>
#include <tbb/parallel_for.h>
#include <tbb/parallel_reduce.h>
#include <tbb/tick_count.h>

#include <list>
#include <deque>
#include <limits>

namespace openvdb {
OPENVDB_USE_VERSION_NAMESPACE
namespace OPENVDB_VERSION_NAME {
namespace tools {


////////////////////////////////////////

/// Conversion flags, used to control the MeshToVolume output.
enum { CFLAG_DISTANCE_FIELD = 0x1, CFLAG_UNSIGNED = 0x2, CFLAG_PRIM_INDEX = 0x4, CFLAG_HERMITE = 0x8 };


// MeshToVolume
template<typename DistGridT>
class MeshToVolume
{
public:
    /// @{
    /// @brief Custom Grid and Tree types
    typedef typename DistGridT::TreeType DistTreeT;
    typedef typename DistTreeT::ValueType DistValueT;
    typedef typename DistTreeT::template ValueConverter<Int32>::Type IndexTreeT;
    typedef Grid<IndexTreeT> IndexGridT;
    typedef typename DistTreeT::template ValueConverter<bool>::Type StencilTreeT;
    typedef Grid<StencilTreeT> StencilGridT;
    typedef typename DistTreeT::template ValueConverter<Hermite>::Type HermiteTreeT;
    typedef Grid<HermiteTreeT> HermiteGridT;
    /// @}

  
    MeshToVolume(openvdb::math::Transform::Ptr&, int conversionFlags = CFLAG_DISTANCE_FIELD, int signSweeps = 1);


    /// @brief Mesh conversion
    ///
    /// @param pointList    list of points, preferably unique and shared by different polygons
    ///                     (The points are expected to be in local grid space.)
    /// @param polygonList  list of triangles and/or quads
    /// @param exBandWidth  The exterior narrow-band width in world space units.
    /// @param inBandWidth  The interior narrow-band width in world space units.
    void convert(const std::vector<Vec3s>& pointList, const std::vector<Vec4I>& polygonList,
        DistValueT exBandWidth, DistValueT inBandWidth = DistValueT(0.0));

    void clear();

    /// Returns a narrow-band (signed) distance field / level set grid.
    typename DistGridT::Ptr distGridPtr() const { return mDistGrid; }

    /// Returns a grid containing the closest-primitive index for each
    /// voxel in the narrow-band.
    typename IndexGridT::Ptr indexGridPtr() const { return mIndexGrid; }

    /// Return a Hermite data grid.
    typename HermiteGridT::Ptr hermiteGridPtr() const { return mHermiteGrid; }

private:
    // disallow copy by assignment
    void operator=(const MeshToVolume<DistGridT>&) {}

    openvdb::math::Transform::Ptr mTransform;
    int mConversionFlags, mSignSweeps;

    typename DistGridT::Ptr     mDistGrid;
    typename IndexGridT::Ptr    mIndexGrid;
    typename HermiteGridT::Ptr  mHermiteGrid;
    typename StencilGridT::Ptr  mIntersectingVoxelsGrid;
};


////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////

// Internal utility objects and implementation details 



namespace internal {


template<typename DistTreeT, typename IndexTreeT>
inline void
combine(DistTreeT& lhsDist, IndexTreeT& lhsIndex, DistTreeT& rhsDist, IndexTreeT& rhsIndex)
{
    typedef typename DistTreeT::ValueType DistValueT;
    typename tree::ValueAccessor<DistTreeT> lhsDistAccessor(lhsDist);
    typename tree::ValueAccessor<IndexTreeT> lhsIndexAccessor(lhsIndex);
    typename tree::ValueAccessor<IndexTreeT> rhsIndexAccessor(rhsIndex);
    typename DistTreeT::LeafCIter iter = rhsDist.cbeginLeaf();
    
    DistValueT rhsValue;
    Coord ijk;

    for ( ; iter; ++iter) {
        typename DistTreeT::LeafNodeType::ValueOnCIter it = iter->cbeginValueOn();

        for ( ; it; ++it) {
            
            ijk = it.getCoord();
            rhsValue = it.getValue();
            DistValueT& lhsValue = const_cast<DistValueT&>(lhsDistAccessor.getValue(ijk));

            if (-rhsValue < std::abs(lhsValue)) {
                lhsValue = rhsValue;
                lhsIndexAccessor.setValue(ijk, rhsIndexAccessor.getValue(ijk));
            }
        }
    }  
}

//////////

/// MeshVoxelizer
/// @brief TBB class object to voxelize a mesh of triangles and/or quads into a collection
/// of VDB grids, namely a square distance grid, closest primitive grid and a intersecting
/// voxels grid (the voxels intersect the mesh).
/// @note Only the VDB leaf nodes that intersect the mesh are allocated, and only a narrow-band
/// of (2-3) voxels (in proximity to the mesh's surface) are rasterized (populated with distance
/// values and primitive indexes).
template<typename DistTreeT>
class MeshVoxelizer
{
public:

    
    /// @{
    /// @brief Custom types
    typedef typename DistTreeT::ValueType DistValueT;
    typedef typename tree::ValueAccessor<DistTreeT> DistAccessorT;
    typedef typename DistTreeT::template ValueConverter<Int32>::Type IndexTreeT;
    typedef typename tree::ValueAccessor<IndexTreeT> IndexAccessorT;
    typedef typename DistTreeT::template ValueConverter<bool>::Type StencilTreeT;
    typedef typename tree::ValueAccessor<StencilTreeT> StencilAccessorT;
    typedef typename DistTreeT::template ValueConverter<Hermite>::Type HermiteTreeT;
    typedef typename tree::ValueAccessor<HermiteTreeT> HermiteAccessorT;
    /// @}

    MeshVoxelizer(const std::vector<Vec3s>& pointList, const std::vector<Vec4I>& polygonList);

    ~MeshVoxelizer() {}

    void runParallel();
    void runSerial();

    MeshVoxelizer(MeshVoxelizer<DistTreeT>& rhs, tbb::split);
    void operator() (const tbb::blocked_range<size_t> &range);
    void join(MeshVoxelizer<DistTreeT>& rhs);

    DistTreeT& sqrDistTree() { return mSqrDistTree; }
    IndexTreeT& primIndexTree() { return mPrimIndexTree; }
    StencilTreeT& intersectionTree() { return mIntersectionTree; }

private:
    // disallow copy by assignment
    void operator=(const MeshVoxelizer<DistTreeT>&) {}

    bool evalVoxel(const Coord& ijk, const Int32 polyIdx);

    std::vector<Vec3s> const * const mPointList;
    std::vector<Vec4I> const * const mPolygonList;

    DistTreeT mSqrDistTree;
    DistAccessorT mSqrDistAccessor;

    IndexTreeT mPrimIndexTree;
    IndexAccessorT mPrimIndexAccessor;

    /// @todo Extract Hermite data.
    //HermiteTreeT mHermiteTree;
    //HermiteAccessorT mHermiteAccessor;

    StencilTreeT mIntersectionTree;
    StencilAccessorT mIntersectionAccessor;

    // Used internally for acceleration
    IndexTreeT mLastPrimTree;
    IndexAccessorT mLastPrimAccessor;
};


template<typename DistTreeT>
void
MeshVoxelizer<DistTreeT>::runParallel()
{
    tbb::parallel_reduce(tbb::blocked_range<size_t>(0, mPolygonList->size()), *this);
}

template<typename DistTreeT>
void
MeshVoxelizer<DistTreeT>::runSerial()
{
    (*this)(tbb::blocked_range<size_t>(0, mPolygonList->size()));
}

template<typename DistTreeT>
MeshVoxelizer<DistTreeT>::MeshVoxelizer(
    const std::vector<Vec3s>& pointList, const std::vector<Vec4I>& polygonList)
    : mPointList(&pointList)
    , mPolygonList(&polygonList)
    , mSqrDistTree(std::numeric_limits<DistValueT>::max())
    , mSqrDistAccessor(mSqrDistTree)
    , mPrimIndexTree(Int32(util::INVALID_IDX))
    , mPrimIndexAccessor(mPrimIndexTree)
    , mIntersectionTree(false)
    , mIntersectionAccessor(mIntersectionTree)
    , mLastPrimTree(Int32(util::INVALID_IDX))
    , mLastPrimAccessor(mLastPrimTree)
{
}

template<typename DistTreeT>
MeshVoxelizer<DistTreeT>::MeshVoxelizer(MeshVoxelizer<DistTreeT>& rhs, tbb::split)
    : mPointList(rhs.mPointList)
    , mPolygonList(rhs.mPolygonList)
    , mSqrDistTree(std::numeric_limits<DistValueT>::max())
    , mSqrDistAccessor(mSqrDistTree)
    , mPrimIndexTree(Int32(util::INVALID_IDX))
    , mPrimIndexAccessor(mPrimIndexTree)
    , mIntersectionTree(false)
    , mIntersectionAccessor(mIntersectionTree)
    , mLastPrimTree(Int32(util::INVALID_IDX))
    , mLastPrimAccessor(mLastPrimTree)
{
}

template<typename DistTreeT>
void
MeshVoxelizer<DistTreeT>::operator()(const tbb::blocked_range<size_t> &range)
{
    StencilTreeT auxTree(false);
    StencilAccessorT auxAcc(auxTree);
    Coord ijk, n_ijk;

    for (size_t n = range.begin(); n < range.end(); ++n) {

        const Int32 primIdx = n;
        Vec3d p0((*mPointList)[(*mPolygonList)[n][0]]);
        ijk = util::nearestCoord(p0);

        evalVoxel(ijk, primIdx);

        mLastPrimAccessor.setValue(ijk, primIdx);
        auxAcc.setActiveState(ijk, true);

        while (!auxTree.empty()) {

            typename StencilTreeT::LeafIter leafIter = auxTree.beginLeaf();
            for (; leafIter; leafIter.next()) {

                typename StencilTreeT::LeafNodeType::ValueOnIter iter = leafIter->beginValueOn();
                for (; iter; iter.next()) {
                    ijk = iter.getCoord();
                    iter.setValueOff();

                    mIntersectionAccessor.setActiveState(ijk, true);

                    for (Int32 i = 0; i < 26; ++i) {
                        n_ijk = ijk + util::COORD_OFFSETS[i];

                        if (primIdx != mLastPrimAccessor.getValue(n_ijk)) {
                            mLastPrimAccessor.setValue(n_ijk, n);
                            if (evalVoxel(n_ijk, n)) auxAcc.setActiveState(n_ijk, true);
                        }
                    }
                }
            }

            auxTree.pruneInactive();
        }
    }
}

template<typename DistTreeT>
bool
MeshVoxelizer<DistTreeT>::evalVoxel(const Coord& ijk, const Int32 polyIdx)
{
    Vec3d voxelCenter(ijk[0], ijk[1], ijk[2]);
    Vec4I verts = (*mPolygonList)[polyIdx];

    // Grab the triangle's points
    Vec3d p0((*mPointList)[verts[0]]);
    Vec3d p1((*mPointList)[verts[1]]);
    Vec3d p2((*mPointList)[verts[2]]);

    double dist = math::triToPtnDistSqr(p0, p1, p2, voxelCenter);

    // Split-up quad into a second triangle and calac distance.
    if (util::INVALID_IDX != verts[3]) {
        p1 = Vec3d((*mPointList)[verts[3]]);

        double secondDist = math::triToPtnDistSqr(p0, p1, p2, voxelCenter);
        if (secondDist < dist) dist = secondDist;
    }

    const DistValueT tmp(dist);
    if (tmp < std::abs(mSqrDistAccessor.getValue(ijk))) {
        mSqrDistAccessor.setValue(ijk, -tmp);
        mPrimIndexAccessor.setValue(ijk, polyIdx);
    }

    return (dist < 0.86602540378443861);
}

template<typename DistTreeT>
void
MeshVoxelizer<DistTreeT>::join(MeshVoxelizer<DistTreeT>& rhs)
{
    typename DistTreeT::LeafCIter iter = rhs.mSqrDistTree.cbeginLeaf();
    DistValueT rhsDist;
    Coord ijk;

    for ( ; iter; ++iter) {
        typename DistTreeT::LeafNodeType::ValueOnCIter it = iter->cbeginValueOn();

        for ( ; it; ++it) {
            
            ijk = it.getCoord();
            rhsDist = it.getValue();
            DistValueT lhsDist = mSqrDistAccessor.getValue(ijk);

            if (-rhsDist < std::abs(lhsDist)) {
                mSqrDistAccessor.setValue(ijk, rhsDist);
                mPrimIndexAccessor.setValue(ijk, rhs.mPrimIndexAccessor.getValue(ijk));
            }
        }
    }

    mIntersectionTree.merge(rhs.mIntersectionTree);
}

//////////

/// ContourTracer
/// @brief TBB Class object that slices up the volume into 2D slices that can be processed in
/// parallel and marks the exterior contour of disjoint voxel sets in each slice.

template<typename DistTreeT>
class ContourTracer
{
public:

    /// @{
    /// @brief Custom types
    typedef typename DistTreeT::ValueType DistValueT;
    typedef typename tree::ValueAccessor<DistTreeT> DistAccessorT;
    typedef typename DistTreeT::template ValueConverter<bool>::Type StencilTreeT;
    typedef typename tree::ValueAccessor<const StencilTreeT> StencilAccessorT;
    /// @}


    ContourTracer(DistTreeT&, const StencilTreeT&);
    ~ContourTracer() {}

    void runParallel();
    void runSerial();

    ContourTracer(const ContourTracer<DistTreeT>& rhs);
    void operator()(const tbb::blocked_range<int> &range) const;

private:
    void operator=(const ContourTracer<DistTreeT>&) {}

    int sparseScan(int slice) const;

    DistTreeT& mDistTree;
    DistAccessorT mDistAccessor;

    const StencilTreeT& mIntersectionTree;
    StencilAccessorT mIntersectionAccessor;

    CoordBBox mBBox;

    /// List of value-depth dependant step sizes.
    std::vector<Index> mStepSize;
};

template<typename DistTreeT>
void
ContourTracer<DistTreeT>::runParallel()
{
    tbb::parallel_for(tbb::blocked_range<int>(mBBox.min()[0], mBBox.max()[0]+1), *this);
}

template<typename DistTreeT>
void
ContourTracer<DistTreeT>::runSerial()
{
    (*this)(tbb::blocked_range<int>(mBBox.min()[0], mBBox.max()[0]+1));
}

template<typename DistTreeT>
ContourTracer<DistTreeT>::ContourTracer(
    DistTreeT& distTree, const StencilTreeT& intersectionTree)
    : mDistTree(distTree)
    , mDistAccessor(mDistTree)
    , mIntersectionTree(intersectionTree)
    , mIntersectionAccessor(mIntersectionTree)
    , mBBox(CoordBBox())
    , mStepSize(0)
{
    // Build the step size table for different tree value depths.
    std::vector<Index> dims;
    mDistTree.getNodeLog2Dims(dims);

    mStepSize.resize(dims.size()+1, 1);
    Index exponent = 0;
    for (int idx = static_cast<int>(dims.size()) - 1; idx > -1; --idx) {
        exponent += dims[idx];
        mStepSize[idx] = 1 << exponent;
    }

    mDistTree.evalLeafBoundingBox(mBBox);

    // Make sure that mBBox coincides with the min and max corners of the internal nodes.
    const int tileDim = mStepSize[0];

    for (size_t i = 0; i < 3; ++i) {

        int n;
        double diff = std::abs(double(mBBox.min()[i])) / double(tileDim);

        if (mBBox.min()[i] <= tileDim) {
            n = int(std::ceil(diff));
            mBBox.min()[i] = - n * tileDim;
        } else {
            n = int(std::floor(diff));
            mBBox.min()[i] = n * tileDim;
        }

        n = int(std::ceil(std::abs(double(mBBox.max()[i] - mBBox.min()[i])) / double(tileDim)));
        mBBox.max()[i] =  mBBox.min()[i] + n * tileDim;
    }
}

template<typename DistTreeT>
ContourTracer<DistTreeT>::ContourTracer(const ContourTracer<DistTreeT> &rhs)
    : mDistTree(rhs.mDistTree)
    , mDistAccessor(mDistTree)
    , mIntersectionTree(rhs.mIntersectionTree)
    , mIntersectionAccessor(mIntersectionTree)
    , mBBox(rhs.mBBox)
    , mStepSize(rhs.mStepSize)
{
}

template<typename DistTreeT>
void
ContourTracer<DistTreeT>::operator()(const tbb::blocked_range<int> &range) const
{
    // Slice up the volume and trace contours.
    int iStep = 1;
    for (int n = range.begin(); n < range.end(); n += iStep) {
        iStep = sparseScan(n);
    }
}

template<typename DistTreeT>
int
ContourTracer<DistTreeT>::sparseScan(int slice) const
{

    bool lastVoxelWasOut = true;
    int last_k;

    Coord ijk(slice, mBBox.min()[1], mBBox.min()[2]);
    Coord step(mStepSize[mDistAccessor.getValueDepth(ijk) + 1]);
    Coord n_ijk;

    for (ijk[1] = mBBox.min()[1]; ijk[1] <= mBBox.max()[1]; ijk[1] += step[1]) { // j

        step[1] = mStepSize[mDistAccessor.getValueDepth(ijk) + 1];
        step[0] = std::min(step[0], step[1]);

        for (ijk[2] = mBBox.min()[2]; ijk[2] <= mBBox.max()[2]; ijk[2] += step[2]) { // k

            step[2] = mStepSize[mDistAccessor.getValueDepth(ijk) + 1];
            step[1] = std::min(step[1], step[2]);
            step[0] = std::min(step[0], step[2]);

            // If the current voxel is set?
            if (mDistAccessor.isValueOn(ijk)) {

                // Is this a boundary voxel?
                if (mIntersectionAccessor.isValueOn(ijk)) {

                    lastVoxelWasOut = false;
                    last_k = ijk[2];

                } else if (lastVoxelWasOut) {

                    DistValueT& val = const_cast<DistValueT&>(mDistAccessor.getValue(ijk));
                    val = -val; // flip sign

                } else {

                    DistValueT val;
                    for (Int32 n = 3; n < 6; n += 2) {
                        n_ijk = ijk + util::COORD_OFFSETS[n];

                        if (mDistAccessor.probeValue(n_ijk, val) && val > 0) {
                            lastVoxelWasOut = true;
                            break;
                        }
                    }

                    if (lastVoxelWasOut) {

                        DistValueT& v = const_cast<DistValueT&>(mDistAccessor.getValue(ijk));
                        v = -v; // flip sign
                        
                        const int tmp_k = ijk[2];

                        // backtrace
                        for (--ijk[2]; ijk[2] >= last_k; --ijk[2]) {
                            if (mIntersectionAccessor.isValueOn(ijk)) break;
                            DistValueT& v = const_cast<DistValueT&>(mDistAccessor.getValue(ijk));
                            if(v < DistValueT(0.0)) v = -v; // flip sign
                        }

                        last_k = tmp_k;
                        ijk[2] = tmp_k;

                    } else {
                        last_k = std::min(ijk[2], last_k);
                    }

                }

            } // end isValueOn check
        } // end k
    } // end j
    return step[0];
}

//////////

/// IntersectingVoxelSign
/// @brief TBB Class object that traversers all the intersecting voxels (defined by the
/// intersectingVoxelsGrid) and potentially flips their sign, by comparing the 'closest point'
/// directions of outside-marked and non-intersecting neighbouring voxel.

template<typename DistTreeT>
class IntersectingVoxelSign
{
public:

    /// @{
    /// @brief Custom types
    typedef typename DistTreeT::ValueType DistValueT;
    typedef typename tree::ValueAccessor<DistTreeT> DistAccessorT;
    typedef typename DistTreeT::template ValueConverter<Int32>::Type IndexTreeT;
    typedef typename tree::ValueAccessor<IndexTreeT> IndexAccessorT;
    typedef typename DistTreeT::template ValueConverter<bool>::Type StencilTreeT;
    typedef typename tree::ValueAccessor<StencilTreeT> StencilAccessorT;
    typedef tree::LeafManager<StencilTreeT> StencilArrayT;
    /// @}
    
    IntersectingVoxelSign(
        const std::vector<Vec3s>& pointList,
        const std::vector<Vec4I>& polygonList,
        DistTreeT& distTree,
        IndexTreeT& indexTree,
        StencilTreeT& intersectionTree,
        StencilArrayT& leafs);

    ~IntersectingVoxelSign() {}

    void runParallel();
    void runSerial();

    IntersectingVoxelSign(const IntersectingVoxelSign<DistTreeT> &rhs);
    void operator()(const tbb::blocked_range<size_t>&) const;

private:
    void operator=(const IntersectingVoxelSign<DistTreeT>&) {}

    void evalVoxel(const Coord& ijk) const;
    Vec3d getClosestPointDir(const Coord& ijk) const;

    std::vector<Vec3s> const * const mPointList;
    std::vector<Vec4I> const * const mPolygonList;

    DistTreeT& mDistTree;
    DistAccessorT mDistAccessor;

    IndexTreeT& mIndexTree;
    IndexAccessorT mIndexAccessor;

    StencilTreeT& mIntersectionTree;
    StencilAccessorT mIntersectionAccessor;
    StencilArrayT& mLeafs;
};

template<typename DistTreeT>
void
IntersectingVoxelSign<DistTreeT>::runParallel()
{
    tbb::parallel_for(mLeafs.getRange(), *this);
}

template<typename DistTreeT>
void
IntersectingVoxelSign<DistTreeT>::runSerial()
{
    (*this)(mLeafs.getRange());
}

template<typename DistTreeT>
IntersectingVoxelSign<DistTreeT>::IntersectingVoxelSign(
    const std::vector<Vec3s>& pointList,
    const std::vector<Vec4I>& polygonList,
    DistTreeT& distTree,
    IndexTreeT& indexTree,
    StencilTreeT& intersectionTree,
    StencilArrayT& leafs)
    : mPointList(&pointList)
    , mPolygonList(&polygonList)
    , mDistTree(distTree)
    , mDistAccessor(mDistTree)
    , mIndexTree(indexTree)
    , mIndexAccessor(mIndexTree)
    , mIntersectionTree(intersectionTree)
    , mIntersectionAccessor(mIntersectionTree)
    , mLeafs(leafs)
{
}

template<typename DistTreeT>
IntersectingVoxelSign<DistTreeT>::IntersectingVoxelSign(
    const IntersectingVoxelSign<DistTreeT> &rhs)
    : mPointList(rhs.mPointList)
    , mPolygonList(rhs.mPolygonList)
    , mDistTree(rhs.mDistTree)
    , mDistAccessor(mDistTree)
    , mIndexTree(rhs.mIndexTree)
    , mIndexAccessor(mIndexTree)
    , mIntersectionTree(rhs.mIntersectionTree)
    , mIntersectionAccessor(mIntersectionTree)
    , mLeafs(rhs.mLeafs)
{
}

template<typename DistTreeT>
void
IntersectingVoxelSign<DistTreeT>::operator()(
    const tbb::blocked_range<size_t>& range) const
{
    typename StencilTreeT::LeafNodeType::ValueOnCIter iter;

    for (size_t n = range.begin(); n < range.end(); ++n) {
        iter = mLeafs.leaf(n).cbeginValueOn();
        for (; iter; ++iter) {
            evalVoxel(iter.getCoord());
        }
    }
}

template<typename DistTreeT>
void
IntersectingVoxelSign<DistTreeT>::evalVoxel(const Coord& ijk) const
{
    const DistValueT val = mDistAccessor.getValue(ijk), zeroVal(0.0);

    if(!(val < zeroVal)) return;
 
    Vec3d dir = getClosestPointDir(ijk), n_dir;
    DistValueT n_val;
    Coord n_ijk;

    // Check voxel-face adjacent neighbours.
    for (Int32 n = 0; n < 26; ++n) {
        n_ijk = ijk + util::COORD_OFFSETS[n];

        if (mIntersectionAccessor.isValueOn(n_ijk)) continue;
        if (!mDistAccessor.probeValue(n_ijk, n_val)) continue;
        if (n_val < zeroVal) continue;

        n_dir = getClosestPointDir(n_ijk);

        if (n_dir.dot(dir) > 0.0 ) {
            const_cast<IntersectingVoxelSign<DistTreeT> *>(this)->
                mDistAccessor.setValue(ijk, -val);
            break;
        }
    }
}

template<typename DistTreeT>
Vec3d
IntersectingVoxelSign<DistTreeT>::getClosestPointDir(const Coord& ijk) const
{
    Vec3d voxelCenter(ijk[0], ijk[1], ijk[2]);
    Vec4I prim = (*mPolygonList)[mIndexAccessor.getValue(ijk)];

    // Grab the first triangle's points
    Vec3d p0((*mPointList)[prim[0]]);
    Vec3d p1((*mPointList)[prim[1]]);
    Vec3d p2((*mPointList)[prim[2]]);

    Vec2d uv;
    double dist = math::sTri3ToPointDistSqr(p0, p1, p2, voxelCenter, uv);

    // Check if quad.
    if (prim[3] != util::INVALID_IDX) {
        Vec3d p3((*mPointList)[prim[3]]);

        Vec2d uv2;
        double dist2 = math::sTri3ToPointDistSqr(p0, p3, p2, voxelCenter, uv2);

        if (dist2 < dist) {
            p1 = p3;
            uv = uv2;
        }
    }

    Vec3d closestPoint = p0 * uv[0] +
                         p1 * uv[1] +
                         p2 * (1.0 - uv(0) - uv(1));

    Vec3d dir = (voxelCenter-closestPoint);
    dir.normalize();
    return dir;
}

//////////

/// IntersectingVoxelCleaner
/// @brief TBB Class object that removes intersecting voxels that where set by rasterizing
/// self-intersecting parts of the mesh.

template<typename DistTreeT>
class IntersectingVoxelCleaner
{
public:

    /// @{
    /// @brief Custom types
    typedef typename DistTreeT::ValueType DistValueT;
    typedef typename tree::ValueAccessor<DistTreeT> DistAccessorT;
    typedef typename DistTreeT::template ValueConverter<Int32>::Type IndexTreeT;
    typedef typename tree::ValueAccessor<IndexTreeT> IndexAccessorT;
    typedef typename DistTreeT::template ValueConverter<bool>::Type StencilTreeT;
    typedef typename tree::ValueAccessor<StencilTreeT> StencilAccessorT;
    typedef tree::LeafManager<StencilTreeT> StencilArrayT;
    /// @}

    IntersectingVoxelCleaner(DistTreeT& distTree, IndexTreeT& indexTree,
        StencilTreeT& intersectionTree, StencilArrayT& leafs);

    ~IntersectingVoxelCleaner() {}

    void runParallel();
    void runSerial();

    IntersectingVoxelCleaner(const IntersectingVoxelCleaner<DistTreeT> &rhs);
    void operator()(const tbb::blocked_range<size_t>&) const;

private:
    void operator=(const IntersectingVoxelCleaner<DistTreeT>&) {}

    DistTreeT& mDistTree;
    DistAccessorT mDistAccessor;

    IndexTreeT& mIndexTree;
    IndexAccessorT mIndexAccessor;

    StencilTreeT& mIntersectionTree;
    StencilAccessorT mIntersectionAccessor;
    StencilArrayT& mLeafs;
};

template<typename DistTreeT>
void
IntersectingVoxelCleaner<DistTreeT>::runParallel()
{
    tbb::parallel_for(mLeafs.getRange(), *this);
    mIntersectionTree.pruneInactive();
}

template<typename DistTreeT>
void
IntersectingVoxelCleaner<DistTreeT>::runSerial()
{
    (*this)(mLeafs.getRange());
    mIntersectionTree.pruneInactive();
}

template<typename DistTreeT>
IntersectingVoxelCleaner<DistTreeT>::IntersectingVoxelCleaner(
    DistTreeT& distTree,
    IndexTreeT& indexTree,
    StencilTreeT& intersectionTree,
    StencilArrayT& leafs)
    : mDistTree(distTree)
    , mDistAccessor(mDistTree)
    , mIndexTree(indexTree)
    , mIndexAccessor(mIndexTree)
    , mIntersectionTree(intersectionTree)
    , mIntersectionAccessor(mIntersectionTree)
    , mLeafs(leafs)
{
}

template<typename DistTreeT>
IntersectingVoxelCleaner<DistTreeT>::IntersectingVoxelCleaner(
    const IntersectingVoxelCleaner<DistTreeT>& rhs)
    : mDistTree(rhs.mDistTree)
    , mDistAccessor(mDistTree)
    , mIndexTree(rhs.mIndexTree)
    , mIndexAccessor(mIndexTree)
    , mIntersectionTree(rhs.mIntersectionTree)
    , mIntersectionAccessor(mIntersectionTree)
    , mLeafs(rhs.mLeafs)
{
}

template<typename DistTreeT>
void
IntersectingVoxelCleaner<DistTreeT>::operator()(
    const tbb::blocked_range<size_t>& range) const
{
    Coord ijk, m_ijk;
    bool turnOff;
    DistValueT value, bg = mDistTree.getBackground();

    typename StencilTreeT::LeafNodeType::ValueOnCIter iter;

    for (size_t n = range.begin(); n < range.end(); ++n) {

        typename StencilTreeT::LeafNodeType& leaf = mLeafs.leaf(n);
        iter = leaf.cbeginValueOn();

        for (; iter; ++iter) {

            ijk = iter.getCoord();

            turnOff = true;
            for (Int32 m = 0; m < 26; ++m) {
                m_ijk = ijk + util::COORD_OFFSETS[m];
                if (mDistAccessor.probeValue(m_ijk, value)) {
                    if (value > 0.0) {
                        turnOff = false;
                        break;
                    }
                }
            }

            if (turnOff) leaf.setValueOff(ijk, bg);
        }
    }
}

//////////

/// ShellVoxelCleaner
/// @brief TBB Class object that removes non-intersecting voxels that where set by rasterizing
/// self-intersecting parts of the mesh.

template<typename DistTreeT>
class ShellVoxelCleaner
{
public:

    /// @{
    /// @brief Custom types
    typedef typename DistTreeT::ValueType DistValueT;
    typedef typename tree::ValueAccessor<DistTreeT> DistAccessorT;
    typedef tree::LeafManager<DistTreeT> DistArrayT;
    typedef typename DistTreeT::template ValueConverter<Int32>::Type IndexTreeT;
    typedef typename tree::ValueAccessor<IndexTreeT> IndexAccessorT;
    typedef typename DistTreeT::template ValueConverter<bool>::Type StencilTreeT;
    typedef typename tree::ValueAccessor<StencilTreeT> StencilAccessorT;
    /// @}

    ShellVoxelCleaner(DistTreeT& distTree, DistArrayT& leafs, IndexTreeT& indexTree,
        StencilTreeT& intersectionTree);

    ~ShellVoxelCleaner() {}

    void runParallel();
    void runSerial();

    ShellVoxelCleaner(const ShellVoxelCleaner<DistTreeT> &rhs);
    void operator()(const tbb::blocked_range<size_t>&) const;

private:
    void operator=(const ShellVoxelCleaner<DistTreeT>&) {}

    DistTreeT& mDistTree;
    DistArrayT& mLeafs;
    DistAccessorT mDistAccessor;
    
    IndexTreeT& mIndexTree;
    IndexAccessorT mIndexAccessor;

    StencilTreeT& mIntersectionTree;
    StencilAccessorT mIntersectionAccessor;
};

template<typename DistTreeT>
void
ShellVoxelCleaner<DistTreeT>::runParallel()
{
    tbb::parallel_for(mLeafs.getRange(), *this);
    mDistTree.pruneInactive();
    mIndexTree.pruneInactive();
}

template<typename DistTreeT>
void
ShellVoxelCleaner<DistTreeT>::runSerial()
{
    (*this)(mLeafs.getRange());
    mDistTree.pruneInactive();
    mIndexTree.pruneInactive();
}

template<typename DistTreeT>
ShellVoxelCleaner<DistTreeT>::ShellVoxelCleaner(
    DistTreeT& distTree,
    DistArrayT& leafs,
    IndexTreeT& indexTree,
    StencilTreeT& intersectionTree)
    : mDistTree(distTree)
    , mLeafs(leafs)
    , mDistAccessor(mDistTree)
    , mIndexTree(indexTree)
    , mIndexAccessor(mIndexTree)
    , mIntersectionTree(intersectionTree)
    , mIntersectionAccessor(mIntersectionTree)
{
}

template<typename DistTreeT>
ShellVoxelCleaner<DistTreeT>::ShellVoxelCleaner(
    const ShellVoxelCleaner<DistTreeT> &rhs)
    : mDistTree(rhs.mDistTree)
    , mLeafs(rhs.mLeafs)
    , mDistAccessor(mDistTree)
    , mIndexTree(rhs.mIndexTree)
    , mIndexAccessor(mIndexTree)
    , mIntersectionTree(rhs.mIntersectionTree)
    , mIntersectionAccessor(mIntersectionTree)
{
}

template<typename DistTreeT>
void
ShellVoxelCleaner<DistTreeT>::operator()(
    const tbb::blocked_range<size_t>& range) const
{

    Coord ijk, m_ijk;
    bool turnOff;
    DistValueT value;
    
    const DistValueT distC = -0.86602540378443861;
    const DistValueT distBG = mDistTree.getBackground();
    const Int32 indexBG = mIntersectionTree.getBackground();

    typename DistTreeT::LeafNodeType::ValueOnCIter iter;
    for (size_t n = range.begin(); n < range.end(); ++n) {

        typename DistTreeT::LeafNodeType& leaf = mLeafs.leaf(n);
        iter = leaf.cbeginValueOn();
        for (; iter; ++iter) {

            value = iter.getValue();
            if(value > 0.0) continue;

            ijk = iter.getCoord();
            if (mIntersectionAccessor.isValueOn(ijk)) continue;

            turnOff = true;
            for (Int32 m = 0; m < 18; ++m) {
                m_ijk = ijk + util::COORD_OFFSETS[m];

                if (mIntersectionAccessor.isValueOn(m_ijk)) {
                    turnOff = false;
                    break;
                }
            }

            if (turnOff) {
                leaf.setValueOff(ijk, distBG);

                const_cast<ShellVoxelCleaner<DistTreeT> *>(this)->
                    mIndexAccessor.setValueOff(ijk, indexBG);

            } else {
                if (value > distC) leaf.setValue(ijk, distC);
            }
        }
    }
}


//////////

/// Dilate
/// @brief TBB Class object to dilate the level-set narrow-band.
/// @note The interior and exterior widths should be in world space units and squared.

template<typename DistTreeT>
class Dilate
{
public:

    /// @{
    /// @brief Custom types
    typedef typename DistTreeT::ValueType DistValueT;
    typedef typename tree::ValueAccessor<DistTreeT> DistAccessorT;
    typedef typename DistTreeT::template ValueConverter<Int32>::Type IndexTreeT;
    typedef typename tree::ValueAccessor<IndexTreeT> IndexAccessorT;
    typedef typename DistTreeT::template ValueConverter<bool>::Type StencilTreeT;
    typedef typename tree::ValueAccessor<StencilTreeT> StencilAccessorT;
    typedef tree::LeafManager<StencilTreeT> StencilArrayT;
    /// @}

    Dilate(const std::vector<Vec3s>& pointList, const std::vector<Vec4I>& polygonList,
        DistTreeT& distTree, IndexTreeT& indexTree, StencilTreeT& maskTree, StencilArrayT& leafs,
        DistValueT exteriorBandWidth, DistValueT interiorBandWidth, DistValueT voxelSize);

    Dilate(const Dilate<DistTreeT>& rhs, tbb::split);

    ~Dilate() {}

    void runParallel();
    void runSerial();
    
    void operator()(const tbb::blocked_range<size_t>&);
    void join(Dilate<DistTreeT>&);

private:
    void operator=(const Dilate<DistTreeT>&) {}

    void evalVoxel(const Coord& ijk);
    Vec3d getClosestPoint(const Coord& ijk) const;
    double getDistToPrim(const Coord& ijk, const Int32 polyIdx) const;
    void postProcess();

    std::vector<Vec3s> const * const mPointList;
    std::vector<Vec4I> const * const mPolygonList;

    DistTreeT& mDistTree;
    DistAccessorT mDistAccessor;

    IndexTreeT& mIndexTree;
    IndexAccessorT mIndexAccessor;

    StencilTreeT& mMaskTree;
    StencilAccessorT mMaskAccessor;
    StencilArrayT& mLeafs;

    const DistValueT mExteriorBandWidth, mInteriorBandWidth, mVoxelSize;

    // New trees
    DistTreeT mNewDistTree;
    DistAccessorT mNewDistAccessor;

    IndexTreeT mNewIndexTree;
    IndexAccessorT mNewIndexAccessor;
};

template<typename DistTreeT>
void
Dilate<DistTreeT>::runParallel()
{
    tbb::parallel_reduce(mLeafs.getRange(), *this);
    postProcess();
}

template<typename DistTreeT>
void
Dilate<DistTreeT>::runSerial()
{
    (*this)(mLeafs.getRange());
    postProcess();
}

template<typename DistTreeT>
Dilate<DistTreeT>::Dilate(
    const std::vector<Vec3s>& pointList,
    const std::vector<Vec4I>& polygonList,
    DistTreeT& distTree,
    IndexTreeT& indexTree,
    StencilTreeT& maskTree,
    StencilArrayT& leafs,
    DistValueT exteriorBandWidth, DistValueT interiorBandWidth,
    DistValueT voxelSize)
    : mPointList(&pointList)
    , mPolygonList(&polygonList)
    , mDistTree(distTree)
    , mDistAccessor(mDistTree)
    , mIndexTree(indexTree)
    , mIndexAccessor(mIndexTree)
    , mMaskTree(maskTree)
    , mMaskAccessor(mMaskTree)
    , mLeafs(leafs)
    , mExteriorBandWidth(exteriorBandWidth)
    , mInteriorBandWidth(interiorBandWidth)
    , mVoxelSize(voxelSize)
    , mNewDistTree(std::numeric_limits<DistValueT>::max())
    , mNewDistAccessor(mNewDistTree)
    , mNewIndexTree(Int32(util::INVALID_IDX))
    , mNewIndexAccessor(mNewIndexTree)
{
}

template<typename DistTreeT>
Dilate<DistTreeT>::Dilate(const Dilate<DistTreeT>& rhs, tbb::split)
    : mPointList(rhs.mPointList)
    , mPolygonList(rhs.mPolygonList)
    , mDistTree(rhs.mDistTree)
    , mDistAccessor(mDistTree)
    , mIndexTree(rhs.mIndexTree)
    , mIndexAccessor(mIndexTree)
    , mMaskTree(rhs.mMaskTree)
    , mMaskAccessor(mMaskTree)
    , mLeafs(rhs.mLeafs)
    , mExteriorBandWidth(rhs.mExteriorBandWidth)
    , mInteriorBandWidth(rhs.mInteriorBandWidth)
    , mVoxelSize(rhs.mVoxelSize)
    , mNewDistTree(std::numeric_limits<DistValueT>::max())
    , mNewDistAccessor(mNewDistTree)
    , mNewIndexTree(Int32(util::INVALID_IDX))
    , mNewIndexAccessor(mNewIndexTree)
{
}

template<typename DistTreeT>
void
Dilate<DistTreeT>::operator()(const tbb::blocked_range<size_t>& range)
{
    Coord ijk, n_ijk;
    typename StencilTreeT::LeafNodeType::ValueOnCIter iter;

    for (size_t n = range.begin(); n < range.end(); ++n) {

        iter = mLeafs.leaf(n).cbeginValueOn();
        for (; iter; ++iter) {

            ijk = iter.getCoord();

            for (Int32 n = 0; n < 6; ++n) {
                n_ijk = ijk + util::COORD_OFFSETS[n];

                if (!mDistAccessor.isValueOn(n_ijk) && !mNewDistAccessor.isValueOn(n_ijk)) {
                    evalVoxel(n_ijk);
                }
            }
        }
    }
}

template<typename DistTreeT>
void
Dilate<DistTreeT>::evalVoxel(const Coord& ijk)
{
    Vec3d voxelCenter(ijk[0], ijk[1], ijk[2]);

    double nDist, dist = std::numeric_limits<double>::max();

    bool hasExteriorMarkedNeighbour = false;
    Int32 closestPolyIndex = 0;

    // Find neighbour with closest face point
    Coord n_ijk;
    for (Int32 n = 0; n < 18; ++n) {
        n_ijk = ijk + util::COORD_OFFSETS[n];

        if (mIndexAccessor.isValueOn(n_ijk)) {
            if (mDistAccessor.isValueOn(n_ijk) && mDistAccessor.getValue(n_ijk) > DistValueT(0.0))
                hasExteriorMarkedNeighbour = true;

            nDist = (voxelCenter - getClosestPoint(n_ijk)).lengthSqr();

            if (nDist < dist) {
                dist = nDist;
                closestPolyIndex = mIndexAccessor.getValue(n_ijk);
            }
        }
    }

    // Calc. this voxels distance to the closest primitive.
    DistValueT value(getDistToPrim(ijk, closestPolyIndex));

    if (hasExteriorMarkedNeighbour && !(value > mExteriorBandWidth)) {
        mNewDistAccessor.setValue(ijk, value);
        mNewIndexAccessor.setValue(ijk, closestPolyIndex);
    } else if (!hasExteriorMarkedNeighbour && !(value > mInteriorBandWidth)) {
        mNewDistAccessor.setValue(ijk, -value);
        mNewIndexAccessor.setValue(ijk, closestPolyIndex);
    }
}

template<typename DistTreeT>
Vec3d
Dilate<DistTreeT>::getClosestPoint(const Coord& ijk) const
{
    Vec3d voxelCenter(ijk[0], ijk[1], ijk[2]);
    Vec4I prim = (*mPolygonList)[mIndexAccessor.getValue(ijk)];

    // Grab the first triangle's points
    Vec3d p0((*mPointList)[prim[0]]);
    Vec3d p1((*mPointList)[prim[1]]);
    Vec3d p2((*mPointList)[prim[2]]);

    Vec2d uv;
    double dist = math::sTri3ToPointDistSqr(p0, p1, p2, voxelCenter, uv);

    // Split-up quad into a second triangle and calac distance.
    if (prim[3] != util::INVALID_IDX) {
        Vec3d p3((*mPointList)[prim[3]]);

        Vec2d uv2;
        double dist2 = math::sTri3ToPointDistSqr(p0, p3, p2, voxelCenter, uv2);

        if (dist2 < dist) {
            p1 = p3;
            uv = uv2;
        }
    }

    Vec3d closestPoint = p0 * uv[0] +
                         p1 * uv[1] +
                         p2 * (1 - uv(0) - uv(1));

    return closestPoint;
}

template<typename DistTreeT>
double
Dilate<DistTreeT>::getDistToPrim(const Coord& ijk, const Int32 polyIdx) const
{
    Vec3d voxelCenter(ijk[0], ijk[1], ijk[2]);
    Vec4I verts = (*mPolygonList)[polyIdx];

    // Grab the triangle's points
    Vec3d p0((*mPointList)[verts[0]]);
    Vec3d p1((*mPointList)[verts[1]]);
    Vec3d p2((*mPointList)[verts[2]]);

    double dist = math::triToPtnDistSqr(p0, p1, p2, voxelCenter);

    // Split-up quad into a second triangle and calac distance.
    if (util::INVALID_IDX != verts[3]) {
        p1 = Vec3d((*mPointList)[verts[3]]);

        double secondDist = math::triToPtnDistSqr(p0, p1, p2, voxelCenter);
        if (secondDist < dist) dist = secondDist;
    }

    return std::sqrt(dist) * double(mVoxelSize);
}

template<typename DistTreeT>
void
Dilate<DistTreeT>::join(Dilate<DistTreeT>& rhs)
{
    mNewDistTree.merge(rhs.mNewDistTree);
    mNewIndexTree.merge(rhs.mNewIndexTree);
}

template<typename DistTreeT>
void
Dilate<DistTreeT>::postProcess()
{
    mMaskTree.clear();

    // Copy the topology of the newly created SDF grid.
    if (mNewDistTree.activeVoxelCount() > 0) {
        const bool inactiveValue = false, activeValue = true;
        StencilTreeT newMask(mNewDistTree, inactiveValue, activeValue, TopologyCopy());
        mMaskTree.merge(newMask);
    }

    mDistTree.merge(mNewDistTree);
    mIndexTree.merge(mNewIndexTree);
}

//////////

/// Helper methods

/// @brief Surface tracing method that flips the sign of interior marked voxels,
/// will not cross the boundary defined by the intersecting voxels.
///
/// @param seed             the coordinates of a interior marked seed point
/// @param distTree         the distance field to operate on
/// @param intersectionTree tree that defines the surface boundary
template<typename DistTreeT>
inline void
surfaceTracer(const Coord &seed, DistTreeT& distTree,
    typename DistTreeT::template ValueConverter<bool>::Type& intersectionTree)
{
    typedef typename DistTreeT::template ValueConverter<bool>::Type StencilTreeT;
    typedef typename tree::ValueAccessor<StencilTreeT> StencilAccessorT;
    typedef typename tree::ValueAccessor<DistTreeT> DistAccessorT;
    typedef typename DistTreeT::ValueType DistValueT;

    StencilAccessorT intrAccessor(intersectionTree);
    DistAccessorT distAccessor(distTree);

    std::deque<Coord> coordList;
    coordList.push_back(seed);
    Coord ijk, n_ijk;

    while (!coordList.empty()) {
        ijk = coordList.back();
        coordList.pop_back();

        if (!distAccessor.isValueOn(ijk)) continue;

        DistValueT& dist = const_cast<DistValueT&>(distAccessor.getValue(ijk));
        if (!(dist < 0.0)) continue;
        dist = -dist; // flip sign

        for (int n = 0; n < 6; ++n) {
            n_ijk = ijk + util::COORD_OFFSETS[n];

            if (!intrAccessor.isValueOn(n_ijk)) {    // Don't cross the interface
                if (distAccessor.isValueOn(n_ijk)) {                 // Is part of the narrow band
                    if (distAccessor.getValue(n_ijk) < 0.0) {        // Marked as outside.
                        coordList.push_back(n_ijk);
                    }
                }
            }

        } // END neighbour voxel loop.
    } // END coordList loop.
}


/// @brief Does a sparse iteration on the SDF grid to find regions with inconsistent sign
/// information. The surfaceTracer method is then used to resolve the sign inconsistency
/// in these regions.
///
/// @param distTree         Signed distance field to operate on
/// @param intersectionTree tree that defines the surface boundary for the surface tracer
template<typename DistTreeT>
inline void
propagateSign(DistTreeT& distTree,
    typename DistTreeT::template ValueConverter<bool>::Type& intersectionTree)
{
    typedef typename DistTreeT::template ValueConverter<bool>::Type StencilTreeT;
    typedef typename tree::ValueAccessor<StencilTreeT> StencilAccessorT;
    typedef typename tree::ValueAccessor<DistTreeT> DistAccessorT;
    typedef typename DistTreeT::ValueType DistValueT;

    StencilAccessorT intrAccessor(intersectionTree);
    DistAccessorT distAccessor(distTree);
    Coord ijk, n_ijk;

    typename DistTreeT::LeafIter leafIter = distTree.beginLeaf();
    for (; leafIter; leafIter.next()) {

        typename DistTreeT::LeafNodeType::ValueOnIter iter = leafIter->beginValueOn();
        for (; iter; iter.next()) {

            ijk = iter.getCoord();

            // Ignore intersecting voxels.
            if (intrAccessor.isValueOn(ijk)) continue;

            if (iter.getValue() < 0.0) {
                for (Int32 n = 0; n < 6; ++n) {
                    n_ijk = ijk + util::COORD_OFFSETS[n];

                    if (distAccessor.isValueOn(n_ijk) && distAccessor.getValue(n_ijk) > 0.0) {
                        surfaceTracer(ijk, distTree, intersectionTree);
                        break;
                    }
                }
            }

        } // END voxel iteration
    } // END leaf iteration
}


template<typename ValueType>
struct SqrtAndScaleOp {

    SqrtAndScaleOp(ValueType voxelSize, bool unsignedDist = false)
        : mVoxelSize(voxelSize) 
        , mUnsigned(unsignedDist) 
    {
    }
 
    template <typename LeafNodeType>
    void operator()(LeafNodeType &leaf, size_t/*leafIndex*/) const
    {
        ValueType w[2];
        w[0] = mVoxelSize;
        w[1] = -mVoxelSize;

        typename LeafNodeType::ValueOnIter iter = leaf.beginValueOn();
        for (; iter; ++iter) {
            ValueType& val = const_cast<ValueType&>(iter.getValue());
            val = w[!mUnsigned && int(val < ValueType(0.0))] * std::sqrt(std::abs(val));
        }
    }

private:
    ValueType mVoxelSize;
    const bool mUnsigned;
};




template<typename ValueType>
struct VoxelSignOp {

    VoxelSignOp(ValueType exBandWidth, ValueType inBandWidth)
        : mExBandWidth(exBandWidth)
        , mInBandWidth(inBandWidth)
    {
    }
 
    template <typename LeafNodeType>
    void operator()(LeafNodeType &leaf, size_t/*leafIndex*/) const
    {
        ValueType bgValues[2];
        bgValues[0] = mExBandWidth;
        bgValues[1] = -mInBandWidth;

        typename LeafNodeType::ValueOffIter iter = leaf.beginValueOff();

        for (; iter; ++iter) {
            ValueType& val = const_cast<ValueType&>(iter.getValue()); 
            val = bgValues[int(val < ValueType(0.0))];
        }
    }

private:
    ValueType mExBandWidth, mInBandWidth;
};


template<typename ValueType>
struct TrimOp {

    TrimOp(ValueType exBandWidth, ValueType inBandWidth)
        : mExBandWidth(exBandWidth)
        , mInBandWidth(inBandWidth)
    {
    }
 
    template <typename LeafNodeType>
    void operator()(LeafNodeType &leaf, size_t/*leafIndex*/) const
    {
        typename LeafNodeType::ValueOffIter iter = leaf.beginValueOff();

        for (; iter; ++iter) {
            ValueType val = iter.getValue(); 
            if (val < -mInBandWidth) {
                iter.setValue(-mInBandWidth);
                iter.setValueOff();
            } else if (val > mExBandWidth) {
                iter.setValue(mExBandWidth);
                iter.setValueOff();
            }
        }
    }

private:
    ValueType mExBandWidth, mInBandWidth;
};


template<typename ValueType>
struct OffsetOp {

    OffsetOp(ValueType offset): mOffset(offset) {}

    void resetOffset(ValueType offset) { mOffset = offset; }
 
    template <typename LeafNodeType>
    void operator()(LeafNodeType &leaf, size_t/*leafIndex*/) const
    {
        typename LeafNodeType::ValueOnIter iter = leaf.beginValueOn();
        for (; iter; ++iter) {
            ValueType& val = const_cast<ValueType&>(iter.getValue());
            val += mOffset;
        }
    }

private:
    ValueType mOffset;
};


template<typename GridType, typename ValueType>
struct RenormOp {

    typedef math::BIAS_SCHEME<math::FIRST_BIAS> Scheme;
    typedef typename Scheme::template ISStencil<GridType>::StencilType Stencil;
    typedef tree::LeafManager<typename GridType::TreeType> LeafManagerType;
    typedef typename LeafManagerType::BufferType BufferType;

    RenormOp(GridType& grid, LeafManagerType& leafs, ValueType voxelSize, ValueType cfl = 1.0)
        : mGrid(grid)
        , mLeafs(leafs)
        , mVoxelSize(voxelSize)
        , mCFL(cfl)
    {
    }

    void resetCFL(ValueType cfl) { mCFL = cfl; }

    template <typename LeafNodeType>
    void operator()(LeafNodeType &leaf, size_t leafIndex) const
    {
        const ValueType dt = mCFL * mVoxelSize, one(1.0), invDx = one / mVoxelSize;
        Stencil stencil(mGrid);

        BufferType& buffer = mLeafs.getBuffer(leafIndex, 1);

        typename LeafNodeType::ValueOnIter iter = leaf.beginValueOn();
        for (; iter; ++iter) {
            stencil.moveTo(iter);

            const ValueType normSqGradPhi =
                math::ISGradientNormSqrd<math::FIRST_BIAS>::result(stencil);

            const ValueType phi0 = stencil.getValue();
            const ValueType diff = math::Sqrt(normSqGradPhi) * invDx - one;
            const ValueType S = phi0 / (math::Sqrt(math::Pow2(phi0) + normSqGradPhi));

            buffer.setValue(iter.pos(), phi0 - dt * S * diff);
        }
    }

private:
    GridType& mGrid;
    LeafManagerType& mLeafs;
    ValueType mVoxelSize, mCFL;
};


template<typename TreeType, typename ValueType>
struct MinOp {

    typedef tree::LeafManager<TreeType> LeafManagerType;
    typedef typename LeafManagerType::BufferType BufferType;

    MinOp(LeafManagerType& leafs): mLeafs(leafs) {}

    template <typename LeafNodeType>
    void operator()(LeafNodeType &leaf, size_t leafIndex) const
    {
        BufferType& buffer = mLeafs.getBuffer(leafIndex, 1);

        typename LeafNodeType::ValueOnIter iter = leaf.beginValueOn();
        for (; iter; ++iter) {
            ValueType& val = const_cast<ValueType&>(iter.getValue());
            val = std::min(val, buffer.getValue(iter.pos()));
        }
    }

private:
    LeafManagerType& mLeafs;
};

template<typename TreeType, typename ValueType>
struct MergeBufferOp {

    typedef tree::LeafManager<TreeType> LeafManagerType;
    typedef typename LeafManagerType::BufferType BufferType;

    MergeBufferOp(LeafManagerType& leafs, size_t bufferIndex = 1)
        : mLeafs(leafs)
        , mBufferIndex(bufferIndex)
    {
    }

    template <typename LeafNodeType>
    void operator()(LeafNodeType &leaf, size_t leafIndex) const
    {
        BufferType& buffer = mLeafs.getBuffer(leafIndex, mBufferIndex);

        typename LeafNodeType::ValueOnIter iter = leaf.beginValueOn();
        for (; iter; ++iter) {
            leaf.setValueOnly(iter.pos(), buffer.getValue(iter.pos()));
        }
    }

private:
    LeafManagerType& mLeafs;
    const size_t mBufferIndex;
};

} // internal namespace


////////////////////////////////////////


// MeshToVolume

template<typename DistGridT>
MeshToVolume<DistGridT>::MeshToVolume(
    openvdb::math::Transform::Ptr& transform, int conversionFlags, int signSweeps)
    : mTransform(transform)
    , mConversionFlags(conversionFlags)
    , mSignSweeps(signSweeps)
{
    clear();
    mSignSweeps = std::min(mSignSweeps, 1);
}

template<typename DistGridT>
void
MeshToVolume<DistGridT>::clear()
{
    mDistGrid = DistGridT::create(std::numeric_limits<DistValueT>::max());
    mIndexGrid = IndexGridT::create(Int32(util::INVALID_IDX));
    mIntersectingVoxelsGrid = StencilGridT::create(false);
    mHermiteGrid = HermiteGridT::create(Hermite());
}

template<typename DistGridT>
void
MeshToVolume<DistGridT>::convert(
    const std::vector<Vec3s>& pointList,
    const std::vector<Vec4I>& polygonList,
    DistValueT exBandWidth,
    DistValueT inBandWidth)
{
    const bool unsignedDistField = mConversionFlags & CFLAG_UNSIGNED;

    mDistGrid->setTransform(mTransform);
    mIndexGrid->setTransform(mTransform);

    // Voxelize mesh
    { 
        internal::MeshVoxelizer<DistTreeT> voxelizer(pointList, polygonList);
        voxelizer.runParallel();

        mDistGrid->tree().merge(voxelizer.sqrDistTree());
        mIndexGrid->tree().merge(voxelizer.primIndexTree());
        mIntersectingVoxelsGrid->tree().merge(voxelizer.intersectionTree());
    }

    if (!unsignedDistField) {

        // Determine the inside/outside state for the narrow band of voxels.
        {
            // Slices up the volume and label the exterior contour of each slice in parallel.
            internal::ContourTracer<DistTreeT> trace(
                mDistGrid->tree(), mIntersectingVoxelsGrid->tree());

            for (int i = 0; i < mSignSweeps; ++i) {

                trace.runParallel();

                // Propagate sign information between the slices.
                internal::propagateSign<DistTreeT>(mDistGrid->tree(), mIntersectingVoxelsGrid->tree());
            }
        }

        {
            tree::LeafManager<StencilTreeT> leafs(mIntersectingVoxelsGrid->tree());

            // Determine the sign of the mesh intersecting voxels.
            internal::IntersectingVoxelSign<DistTreeT> sign(pointList, polygonList, 
                mDistGrid->tree(), mIndexGrid->tree(), mIntersectingVoxelsGrid->tree(), leafs);

            sign.runParallel();
        
            // Remove mesh intersecting voxels that where set by rasterizing
            // self-intersecting portions of the mesh.
            internal::IntersectingVoxelCleaner<DistTreeT> cleaner(mDistGrid->tree(),
                mIndexGrid->tree(), mIntersectingVoxelsGrid->tree(), leafs);

            cleaner.runParallel();
        }

        {
            // Remove shell voxels that where set by rasterizing
            // self-intersecting portions of the mesh.

            tree::LeafManager<DistTreeT> leafs(mDistGrid->tree());

            internal::ShellVoxelCleaner<DistTreeT> cleaner(mDistGrid->tree(),
                leafs, mIndexGrid->tree(), mIntersectingVoxelsGrid->tree());

            cleaner.runParallel();
        }

        mDistGrid->setGridClass(GRID_LEVEL_SET);

    } else { // if unsigned dist. field

        inBandWidth = DistValueT(0.0);
        mDistGrid->setGridClass(GRID_UNKNOWN);
    }

    if (mDistGrid->activeVoxelCount() == 0) return;
    
    const DistValueT voxelSize(mTransform->voxelDimensions()[0]);

    // Transform values (world space scaling etc.)
    {
        typedef internal::SqrtAndScaleOp<DistValueT> XForm;
        tree::LeafManager<DistTreeT> leafs(mDistGrid->tree());
        XForm op(voxelSize, unsignedDistField);
        LeafTransformer<DistTreeT, XForm> transform(leafs, op);
        transform.runParallel();
    }

    // Narrow-band dilation
    if (inBandWidth > voxelSize || exBandWidth > voxelSize) {

        // Create the initial voxel mask.
        StencilTreeT maskTree(false);
        maskTree.topologyUnion(mDistGrid->tree());
        // todo: diff topology of mIntersectingVoxelsGrid->tree()

        mIntersectingVoxelsGrid->clear();

        while (maskTree.activeVoxelCount() > 0) {

            tree::LeafManager<StencilTreeT> leafs(maskTree);

            internal::Dilate<DistTreeT> dilate(pointList, polygonList, mDistGrid->tree(),
                mIndexGrid->tree(), maskTree, leafs, exBandWidth, inBandWidth, voxelSize);

            dilate.runParallel();
        }
    }

    if (unsignedDistField) return;

    // Propagate sign information to inactive values.
    mDistGrid->tree().signedFloodFill();
    
    // Update the background value (inactive values)
    {
        tree::LeafManager<DistTreeT> leafs(mDistGrid->tree());

        typedef internal::VoxelSignOp<DistValueT> SignXForm;
        SignXForm op(exBandWidth, inBandWidth);

        LeafTransformer<DistTreeT, SignXForm> transform(leafs, op);
        transform.runParallel();

        DistValueT bgValues[2];
        bgValues[0] = exBandWidth;
        bgValues[1] = -inBandWidth;

        typename DistTreeT::ValueAllIter tileIt(mDistGrid->tree());
        tileIt.setMaxDepth(DistTreeT::ValueAllIter::LEAF_DEPTH - 1);

        for ( ; tileIt; ++tileIt) {
            DistValueT& val = const_cast<DistValueT&>(tileIt.getValue());
            val = bgValues[int(val < DistValueT(0.0))];
        }

        // fast bg value swap
        typename DistTreeT::Ptr newTree(new DistTreeT(/*background=*/exBandWidth));
        newTree->merge(mDistGrid->tree());
        mDistGrid->setTree(newTree);   
    }


    // Smooth out bumps caused by self-intersecting and
    // overlapping portions of the mesh and renormalize the level set.

    typedef internal::OffsetOp<DistValueT> OffsetOp;
    typedef internal::RenormOp<DistGridT, DistValueT> RenormOp;
    typedef internal::MinOp<DistTreeT, DistValueT> MinOp;
    typedef internal::MergeBufferOp<DistTreeT, DistValueT> MergeBufferOp;
    typedef internal::TrimOp<DistValueT> TrimOp;

    tree::LeafManager<DistTreeT> leafs(mDistGrid->tree(), 1);
    
    const DistValueT offset = 0.8 * voxelSize;

    OffsetOp offsetOp(-offset);
    LeafTransformer<DistTreeT, OffsetOp> offsetXform(leafs, offsetOp);
    offsetXform.runParallel();

    RenormOp renormOp(*mDistGrid, leafs, voxelSize);
    LeafTransformer<DistTreeT, RenormOp> renormXform(leafs, renormOp);
    renormXform.runParallel();
    
    MinOp minOp(leafs);
    LeafTransformer<DistTreeT, MinOp> minXform(leafs, minOp);
    minXform.runParallel();

    offsetOp.resetOffset(offset);
    offsetXform.runParallel();

    renormXform.runParallel();
    MergeBufferOp mergeOp(leafs);
    LeafTransformer<DistTreeT, MergeBufferOp> mergeXform(leafs, mergeOp);
    mergeXform.runParallel();

    // If the narrow band was not expanded, we might need to trim off
    // some of the active voxels in order to respect the narrow band limits.
    // (The mesh voxelization step generates some extra 'shell' voxels)
    if (!(exBandWidth > voxelSize)) {
        
        TrimOp op(exBandWidth, inBandWidth);
        LeafTransformer<DistTreeT, TrimOp> transform(leafs, op);
        transform.runParallel();

        tree::LevelSetPrune<DistValueT> prune;
        mDistGrid->tree().pruneOp(prune);
    }
}


} // namespace tools
} // namespace OPENVDB_VERSION_NAME
} // namespace openvdb

#endif // OPENVDB_TOOLS_MESH_TO_VOLUME_HAS_BEEN_INCLUDED

// Copyright (c) 2012 DreamWorks Animation LLC
// All rights reserved. This software is distributed under the
// Mozilla Public License 2.0 ( http://www.mozilla.org/MPL/2.0/ )
