///////////////////////////////////////////////////////////////////////////
//
// Copyright (c) 2012 DreamWorks Animation LLC
//
// All rights reserved. This software is distributed under the
// Mozilla Public License 2.0 ( http://www.mozilla.org/MPL/2.0/ )
//
// Redistributions of source code must retain the above copyright
// and license notice and the following restrictions and disclaimer.
//
// *     Neither the name of DreamWorks Animation nor the names of
// its contributors may be used to endorse or promote products derived
// from this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
// OWNER OR CONTRIBUTORS BE LIABLE FOR ANY INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
// IN NO EVENT SHALL THE COPYRIGHT HOLDERS' AND CONTRIBUTORS' AGGREGATE
// LIABILITY FOR ALL CLAIMS REGARDLESS OF THEIR BASIS EXCEED US$250.00.
//
///////////////////////////////////////////////////////////////////////////
//
/// @author Ken Museth
/// @file MeanCurvature.h

#ifndef OPENVDB_TOOLS_MEAN_CURVATURE_HAS_BEEN_INCLUDED
#define OPENVDB_TOOLS_MEAN_CURVATURE_HAS_BEEN_INCLUDED

#include <openvdb/Grid.h>
#include <openvdb/math/Math.h>
#include <openvdb/math/Operators.h>
#include <tbb/parallel_for.h>


namespace openvdb {
OPENVDB_USE_VERSION_NAMESPACE
namespace OPENVDB_VERSION_NAME {
namespace tools {

/// @brief Computes the mean curvature of a scalar grid
///
/// @note The current implementation assumes all the input vector
/// values are represented by leaf voxels and not tiles. In the
/// future we will expand this class to also handle tile values.
template<typename GridT, typename MapType, math::DDScheme S2, math::DScheme S1>
class MeanCurvatureImpl
{
public:
    typedef GridT                                GridType;
    typedef typename GridType::TreeType          TreeType;
    typedef typename GridType::ValueType         Real;
    typedef typename GridType::ConstAccessor     InAccessorType;
    typedef typename tree::LeafArray<TreeType,0> ArrayType;
    typedef typename ArrayType::IterRangeType    RangeType;

    MeanCurvatureImpl(const GridType& grid, const MapType& map):
        mInputGrid(&grid),
        mInputAccessor(grid.getConstAccessor()),
        mMap(&map)
    {
        const Real backg = 0;// Divergence is zero in constant regions!
        // output tree = topology copy of input tree!
        typename TreeType::Ptr outTree(new TreeType(grid.tree(), backg, TopologyCopy()));
        // create grid with output tree and unit transform
        mOutputGrid = typename GridType::Ptr(new GridType(outTree));
        // transform of output grid = tranform of input grid
        mOutputGrid->setTransform(grid.transform().copy());
    }

    /// Constructor called by tbb::parallel_for threads
    MeanCurvatureImpl(const MeanCurvatureImpl& other):
        mInputGrid(other.mInputGrid),//shallow copy
        mOutputGrid(other.mOutputGrid),//shallow copy
        mInputAccessor(mInputGrid->getConstAccessor()),//local instance
        mMap(other.mMap)//shallow copy
    {
    }

    ~MeanCurvatureImpl() {}

    typename GridType::Ptr process(bool threaded=true)
    {
        ArrayType leafArray(mOutputGrid->tree());
        if (threaded) {
            tbb::parallel_for(leafArray.getRange(), *this);
        } else {
            (*this)(leafArray.getRange());
        }
        return mOutputGrid;
    }

    void operator()(const RangeType& range) const
    {
        typedef math::MeanCurvature<MapType, S2, S1> MCT;
        typename ArrayType::IterType aIter, aEnd = range.end(); // leaf nodes
        typename TreeType::LeafNodeType::ValueOnIter vIter; // leaf values
        for (aIter = range.begin(); aIter != aEnd; ++aIter) {
            for (vIter = aIter->leaf->beginValueOn(); vIter; ++vIter) {
                vIter.setValue(MCT::result(*mMap, mInputAccessor, vIter.getCoord()));
            }
        }
    }

protected:
    const GridType*         mInputGrid;
    typename GridType::Ptr  mOutputGrid;
    mutable InAccessorType  mInputAccessor;
    const MapType*          mMap;
}; // end of MeanCurvatureImpl class



template<typename GridT>
class MeanCurvatureFunctor
{
public:
    typedef GridT GridType;

    MeanCurvatureFunctor(const GridType& grid, bool threaded):
        mInputGrid(&grid), mThreaded(threaded)
    {}

    template<typename MapType>
    void operator()(const MapType& map)
    {
        MeanCurvatureImpl<GridType, MapType, math::CD_SECOND, math::CD_2ND>
            meancurveImpl(*mInputGrid, map);
        mOutputGrid = meancurveImpl.process(mThreaded); // cache the result
    }

    // for output of the result
    typename GridType::Ptr result() { return mOutputGrid; }

private:
    const GridType*         mInputGrid;
    typename GridType::Ptr  mOutputGrid;
    bool                    mThreaded;
}; // class LaplacianFunctor


/// @brief Computes the mean curvature of a scalar grid
template<typename GridT>
class MeanCurvature
{
public:
    typedef GridT GridType;

    MeanCurvature(const GridType& grid): mInputGrid(&grid) {}

    ~MeanCurvature() {}

    typename GridType::Ptr process(bool threaded = true)
    {
        MeanCurvatureFunctor<GridT> meancurvatureFunctor(*mInputGrid, threaded);
        processTypedMap(mInputGrid->transform(), meancurvatureFunctor);
        return meancurvatureFunctor.result();
    }

protected:
    const GridType* mInputGrid;
}; // end of MeanCurvature class

} // namespace tools
} // namespace OPENVDB_VERSION_NAME
} // namespace openvdb

#endif // OPENVDB_TOOLS_MEAN_CURVATURE_HAS_BEEN_INCLUDED

// Copyright (c) 2012 DreamWorks Animation LLC
// All rights reserved. This software is distributed under the
// Mozilla Public License 2.0 ( http://www.mozilla.org/MPL/2.0/ )
