///////////////////////////////////////////////////////////////////////////
//
// Copyright (c) 2012 DreamWorks Animation LLC
//
// All rights reserved. This software is distributed under the
// Mozilla Public License 2.0 ( http://www.mozilla.org/MPL/2.0/ )
//
// Redistributions of source code must retain the above copyright
// and license notice and the following restrictions and disclaimer.
//
// *     Neither the name of DreamWorks Animation nor the names of
// its contributors may be used to endorse or promote products derived
// from this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
// OWNER OR CONTRIBUTORS BE LIABLE FOR ANY INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
// IN NO EVENT SHALL THE COPYRIGHT HOLDERS' AND CONTRIBUTORS' AGGREGATE
// LIABILITY FOR ALL CLAIMS REGARDLESS OF THEIR BASIS EXCEED US$250.00.
//
///////////////////////////////////////////////////////////////////////////

#ifndef OPENVDB_TOOLS_LEVEL_SET_UTIL_HAS_BEEN_INCLUDED
#define OPENVDB_TOOLS_LEVEL_SET_UTIL_HAS_BEEN_INCLUDED


#include <openvdb/openvdb.h>
#include <openvdb/tools/ValueTransformer.h>
#include <openvdb/tools/Filter.h>
#include <openvdb/tree/LeafManager.h>


namespace openvdb {
OPENVDB_USE_VERSION_NAMESPACE
namespace OPENVDB_VERSION_NAME {
namespace tools {


/// @brief  Method to convert a sparse level-set/SDF into a sparse fog volume.
/// @note   The negative/interior portion of the narrow-band is transformed
///         into a smooth [0 to 1] gradient, the remaining interior values
///         are set to 1. All exterior values including the background are
///         set to 0 with inactive states. The interior is still a sparse
///         representations but the values are now active.
///
/// @param grid            Level set / SDF grid to transform.
/// @param cutOffDistance  Optional world space cutoff distance for the gradient
///                        (automatically clamped if greater than the interior
///                        narrow band width)
template <class GridType>
inline void
sdfToFogVolume(GridType& grid, typename GridType::ValueType cutOffDistance
    = std::numeric_limits<typename GridType::ValueType>::max());

//////////


/// @brief Threaded operation to find the min and max active voxel values.
template <class TreeType>
class MinMaxVoxel
{
public:
    typedef tree::LeafManager<TreeType> LeafArray;
    typedef typename TreeType::ValueType ValueType;

    /// LeafArray = openvdb::tree::LeafManager<TreeType> leafs(myTree)
    MinMaxVoxel(LeafArray&);
    
    void runParallel();
    void runSerial();
    
    const ValueType& minVoxel() const { return mMin; }
    const ValueType& maxVoxel() const { return mMax; }
    

    inline MinMaxVoxel(const MinMaxVoxel<TreeType>&, tbb::split);
    inline void operator()(const tbb::blocked_range<size_t>&);
    inline void join(const MinMaxVoxel<TreeType>&);

private:
    LeafArray& mLeafArray;
    ValueType mMin, mMax;
};


//////////


/// @brief Threaded leaf-node transformation scheme.
template <class TreeType, class LeafOp>
class LeafTransformer
{
public:
    typedef tree::LeafManager<TreeType> LeafArray;
    typedef typename TreeType::ValueType ValueType;

    /// LeafArray = openvdb::tree::LeafManager<TreeType> leafs(myTree)
    LeafTransformer(LeafArray&, LeafOp&);
    
    void runParallel();
    void runSerial();


    inline void operator()(const tbb::blocked_range<size_t>&) const;
    inline LeafTransformer(const LeafTransformer<TreeType, LeafOp>&);

private:
    LeafArray& mLeafArray;
    LeafOp& mLeafOp;
};


////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////

// Internal utility objects and implementation details 


namespace internal {


template<typename ValueType>
struct FogVolumeOp {

    FogVolumeOp(ValueType cutOffDistance)
        : mWeight(ValueType(1.0) / cutOffDistance)
    {
    }

    // cutOff has to be < 0.0
    template <typename LeafNodeType>
    void operator()(LeafNodeType &leaf, size_t/*leafIndex*/) const
    {
        typename LeafNodeType::ValueAllIter iter =
            leaf.beginValueAll();

        const ValueType zeroVal(0.0);

        for (; iter; ++iter) {
            ValueType& value = const_cast<ValueType&>(iter.getValue());
            if (value > zeroVal) {
                value = zeroVal;
                iter.setValueOff();
            } else {
                value = std::min(ValueType(1.0), value * mWeight);
                iter.setValueOn(value > zeroVal);
            }
        }
    }

private:
    ValueType mWeight;
};

} // namespace internal


////////////////////////////////////////


template <class TreeType>
MinMaxVoxel<TreeType>::MinMaxVoxel(LeafArray& leafs)
    : mLeafArray(leafs)
    , mMin(std::numeric_limits<ValueType>::max())
    , mMax(-mMin)
{
}

template <class TreeType>
inline
MinMaxVoxel<TreeType>::MinMaxVoxel(
    const MinMaxVoxel<TreeType>& rhs, tbb::split)
    : mLeafArray(rhs.mLeafArray)
    , mMin(rhs.mMin)
    , mMax(rhs.mMax)
{
}

template <class TreeType>
void
MinMaxVoxel<TreeType>::runParallel()
{
    tbb::parallel_reduce(mLeafArray.getRange(), *this);
}


template <class TreeType>
void
MinMaxVoxel<TreeType>::runSerial()
{
    (*this)(mLeafArray.getRange());
}


template <class TreeType>
inline void
MinMaxVoxel<TreeType>::operator()(const tbb::blocked_range<size_t>& range)
{
    typename TreeType::LeafNodeType::ValueOnCIter iter;

    for (size_t n = range.begin(); n < range.end(); ++n) {
        iter = mLeafArray.leaf(n).cbeginValueOn();
        for (; iter; ++iter) {
            const ValueType value = iter.getValue();
            mMin = std::min(mMin, value);
            mMax = std::max(mMax, value);
        }
    }
}

template <class TreeType>
inline void
MinMaxVoxel<TreeType>::join(const MinMaxVoxel<TreeType>& rhs)
{
    mMin = std::min(mMin, rhs.mMin);
    mMax = std::max(mMax, rhs.mMax);
}


////////////////////////////////////////


template <class TreeType, class LeafOp>
LeafTransformer<TreeType, LeafOp>::
    LeafTransformer(LeafArray& leafs, LeafOp& leafOp)
    : mLeafArray(leafs)
    , mLeafOp(leafOp)
{
}

template <class TreeType, class LeafOp>
inline
LeafTransformer<TreeType, LeafOp>::LeafTransformer(
    const LeafTransformer<TreeType, LeafOp>& rhs)
    : mLeafArray(rhs.mLeafArray)
    , mLeafOp(rhs.mLeafOp)
{
}

template <class TreeType, class LeafOp>
void
LeafTransformer<TreeType, LeafOp>::runParallel()
{
    tbb::parallel_for(mLeafArray.getRange(), *this);
}

template <class TreeType, class LeafOp>
void
LeafTransformer<TreeType, LeafOp>::runSerial()
{
    (*this)(mLeafArray.getRange());
}

template <class TreeType, class LeafOp>
inline void
LeafTransformer<TreeType, LeafOp>::operator()(const tbb::blocked_range<size_t>& range) const
{
    for (size_t n = range.begin(); n < range.end(); ++n) {
        mLeafOp(mLeafArray.leaf(n), n);
    }
}


////////////////////////////////////////


template <class GridType>
inline void
sdfToFogVolume(GridType& grid, typename GridType::ValueType cutOffDistance)
{
    typedef typename GridType::TreeType TreeType;
    typedef typename GridType::ValueType ValueType;

    cutOffDistance = -std::abs(cutOffDistance);

    TreeType& tree = const_cast<TreeType&>(grid.tree());

    { // Transform all voxels (parallel, over leaf nodes)
        tree::LeafManager<TreeType> leafs(tree);

        MinMaxVoxel<TreeType> minmax(leafs);
        minmax.runParallel();

        // Clamp to the interior band width.
        if (minmax.minVoxel() > cutOffDistance) {
            cutOffDistance = minmax.minVoxel();
        }

        internal::FogVolumeOp<ValueType> op(cutOffDistance);
        LeafTransformer<TreeType, internal::FogVolumeOp<ValueType> > transform(leafs, op);
        transform.runParallel();
    }

    // Transform all tile values (serial, but the iteration
    // is constrained from descending into leaf nodes)
    const ValueType zeroVal(0.0);
    typename TreeType::ValueAllIter iter(tree);
    iter.setMaxDepth(TreeType::ValueAllIter::LEAF_DEPTH - 1);

    for ( ; iter; ++iter) {
        ValueType& value = const_cast<ValueType&>(iter.getValue());

        if (value > zeroVal) {
            value = zeroVal;
            iter.setValueOff();
        } else {
            value = ValueType(1.0);
            iter.setActiveState(true);
        }
    }

    // Update the tree background value.

    typename TreeType::Ptr newTree(new TreeType(/*background=*/zeroVal));
    newTree->merge(tree);
    // This is faster than calling Tree::setBackground, since we only need
    // to update the value that is returned for coordinates that don't fall
    // inside an allocated node. All inactive tiles and voxels have already
    // been updated in the previous step so the Tree::setBackground method
    // will in this case do a redundant traversal of the tree to update the
    // inactive values once more.

    //newTree->pruneInactive();
    grid.setTree(newTree);

    grid.setGridClass(GRID_FOG_VOLUME);
}


} // namespace tools
} // namespace OPENVDB_VERSION_NAME
} // namespace openvdb

#endif // OPENVDB_TOOLS_LEVEL_SET_UTIL_HAS_BEEN_INCLUDED

// Copyright (c) 2012 DreamWorks Animation LLC
// All rights reserved. This software is distributed under the
// Mozilla Public License 2.0 ( http://www.mozilla.org/MPL/2.0/ )
