///////////////////////////////////////////////////////////////////////////
//
// Copyright (c) 2012 DreamWorks Animation LLC
//
// All rights reserved. This software is distributed under the
// Mozilla Public License 2.0 ( http://www.mozilla.org/MPL/2.0/ )
//
// Redistributions of source code must retain the above copyright
// and license notice and the following restrictions and disclaimer.
//
// *     Neither the name of DreamWorks Animation nor the names of
// its contributors may be used to endorse or promote products derived
// from this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
// OWNER OR CONTRIBUTORS BE LIABLE FOR ANY INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
// IN NO EVENT SHALL THE COPYRIGHT HOLDERS' AND CONTRIBUTORS' AGGREGATE
// LIABILITY FOR ALL CLAIMS REGARDLESS OF THEIR BASIS EXCEED US$250.00.
//
///////////////////////////////////////////////////////////////////////////

#ifndef OPENVDB_TOOLS_LEVEL_SET_MESHER_HAS_BEEN_INCLUDED
#define OPENVDB_TOOLS_LEVEL_SET_MESHER_HAS_BEEN_INCLUDED

#include <openvdb/openvdb.h>
#include <openvdb/tree/ValueAccessor.h>
#include <openvdb/tools/ValueTransformer.h>
#include <openvdb/util/Util.h>
#include <openvdb/math/Operators.h>

#include <boost/scoped_array.hpp>
#include <boost/intrusive/slist.hpp>

#include <tbb/mutex.h>
#include <tbb/tick_count.h>
#include <tbb/blocked_range.h>

#include <vector>


namespace openvdb {
OPENVDB_USE_VERSION_NAMESPACE
namespace OPENVDB_VERSION_NAME {
namespace tools {


/// @brief Collection of quads
class QuadPool
{
public:
    QuadPool() {}

    void reset(size_t poolSize)
    {
        mPoolSize = poolSize;
        mQuads.reset(new openvdb::Vec4I[mPoolSize]);
    }

    const size_t& size() const { return mPoolSize; }
    openvdb::Vec4I& operator[](size_t n) { return mQuads[n]; }

private:
    size_t mPoolSize;
    boost::scoped_array<openvdb::Vec4I> mQuads;
};

/// @{
/// @brief Point and primitive list types.
typedef boost::scoped_array<openvdb::Vec3s> PointList;
typedef boost::scoped_array<QuadPool> QuadPoolList;
/// @}


////////////////////////////////////////


/// @brief LevelSet meshing
class LevelSetMesher
{
public:

    /// @param isoValue Determines which iso-surface to extract.
    /// @param adaptivity
    LevelSetMesher(double isoValue = 0, double adaptivity = 0):
        mPointListSize(0), mQuadPoolListSize(0),
        mIsoValue(1e-4 + isoValue), mAdaptivity(adaptivity) {}

    /// @note call with scalar typed grid.
    template<typename GridT>
    void operator()(const GridT&);

    PointList& pointList() { return mPoints; }
    const size_t& pointListSize() const { return mPointListSize; }

    QuadPoolList& quadPoolList() { return mQuads; }
    const size_t& quadPoolListSize() const { return mQuadPoolListSize; }

private:
    PointList mPoints;
    QuadPoolList mQuads;
    size_t mPointListSize, mQuadPoolListSize;
    double mIsoValue, mAdaptivity;
};


////////////////////////////////////////


// Internal utility objects


namespace internal {


// Bit-flags
enum { INSIDE = 0x1, XEDGE = 0x2, YEDGE = 0x4,  ZEDGE = 0x8 };


const bool sAmbiguous[256] =
   {0,0,0,0,0,1,0,0,
    0,0,1,0,0,0,0,0,
    0,0,1,0,1,1,1,0,
    1,0,1,0,1,0,1,0,
    0,1,0,0,1,1,0,0,
    1,1,1,0,1,1,0,0,
    0,0,0,0,1,1,0,0,
    1,0,1,0,1,1,1,0,
    0,1,1,1,0,1,0,0,
    1,1,1,1,0,0,0,0,
    1,1,1,1,1,1,1,1,
    1,1,1,1,1,1,1,1,
    0,1,0,0,0,1,0,0,
    1,1,1,1,0,1,0,0,
    0,0,0,0,0,1,0,0,
    1,1,1,1,1,1,1,0,
    0,1,1,1,1,1,1,1,
    0,0,1,0,0,0,0,0,
    0,0,1,0,1,1,1,1,
    0,0,1,0,0,0,1,0,
    1,1,1,1,1,1,1,1,
    1,1,1,1,1,1,1,1,
    0,0,0,0,1,1,1,1,
    0,0,1,0,1,1,1,0,
    0,1,1,1,0,1,0,1,
    0,0,1,1,0,0,0,0,
    0,0,1,1,0,1,1,1,
    0,0,1,1,0,0,1,0,
    0,1,0,1,0,1,0,1,
    0,1,1,1,0,1,0,0,
    0,0,0,0,0,1,0,0,
    0,0,1,0,0,0,0,0};


template<class AccessorT>
inline bool isAmbiguous(const AccessorT& accessor, const Coord& ijk,
    typename AccessorT::ValueType isoValue, int dim)
{
    unsigned signs = 0;
    Coord coord = ijk; // i, j, k
    if (accessor.getValue(coord) < isoValue) signs |= 1u;
    coord[0] += dim; // i+dim, j, k
    if (accessor.getValue(coord) < isoValue) signs |= 2u;
    coord[2] += dim; // i+dim, j, k+dim
    if (accessor.getValue(coord) < isoValue) signs |= 4u;
    coord[0] = ijk[0]; // i, j, k+dim
    if (accessor.getValue(coord) < isoValue) signs |= 8u;
    coord[1] += dim; coord[2] = ijk[2]; // i, j+dim, k
    if (accessor.getValue(coord) < isoValue) signs |= 16u;
    coord[0] += dim; // i+dim, j+dim, k
    if (accessor.getValue(coord) < isoValue) signs |= 32u;
    coord[2] += dim; // i+dim, j+dim, k+dim
    if (accessor.getValue(coord) < isoValue) signs |= 64u;
    coord[0] = ijk[0]; // i, j+dim, k+dim
    if (accessor.getValue(coord) < isoValue) signs |= 128u;
    return sAmbiguous[signs];
}


template<class AccessorT>
inline bool
isNonManifold(const AccessorT& accessor, const Coord& ijk,
    typename AccessorT::ValueType isoValue, const int dim)
{
    int hDim = dim >> 1;
    bool m, p[8]; // Corner signs

    Coord coord = ijk; // i, j, k
    p[0] = accessor.getValue(coord) < isoValue;
    coord[0] += dim; // i+dim, j, k
    p[1] = accessor.getValue(coord) < isoValue;
    coord[2] += dim; // i+dim, j, k+dim
    p[2] = accessor.getValue(coord) < isoValue;
    coord[0] = ijk[0]; // i, j, k+dim
    p[3] = accessor.getValue(coord) < isoValue;
    coord[1] += dim; coord[2] = ijk[2]; // i, j+dim, k
    p[4] = accessor.getValue(coord) < isoValue;
    coord[0] += dim; // i+dim, j+dim, k
    p[5] = accessor.getValue(coord) < isoValue;
    coord[2] += dim; // i+dim, j+dim, k+dim
    p[6] = accessor.getValue(coord) < isoValue;
    coord[0] = ijk[0]; // i, j+dim, k+dim
    p[7] = accessor.getValue(coord) < isoValue;

    // Check if the corner sign configuration is ambiguous
    unsigned signs = 0;
    if (p[0]) signs |= 1u;
    if (p[1]) signs |= 2u;
    if (p[2]) signs |= 4u;
    if (p[3]) signs |= 8u;
    if (p[4]) signs |= 16u;
    if (p[5]) signs |= 32u;
    if (p[6]) signs |= 64u;
    if (p[7]) signs |= 128u;
    if (sAmbiguous[signs]) return true;

    // Manifold check

    // Evaluate edges
    int i = ijk[0], ip = ijk[0] + hDim, ipp = ijk[0] + dim;
    int j = ijk[1], jp = ijk[1] + hDim, jpp = ijk[1] + dim;
    int k = ijk[2], kp = ijk[2] + hDim, kpp = ijk[2] + dim;

    // edge 1
    coord.reset(ip, j, k);
    m = accessor.getValue(coord) < isoValue;
    if (p[0] != m && p[1] != m) return true;

    // edge 2
    coord.reset(ipp, j, kp);
    m = accessor.getValue(coord) < isoValue;
    if (p[1] != m && p[2] != m) return true;

    // edge 3
    coord.reset(ip, j, kpp);
    m = accessor.getValue(coord) < isoValue;
    if (p[2] != m && p[3] != m) return true;

    // edge 4
    coord.reset(i, j, kp);
    m = accessor.getValue(coord) < isoValue;
    if (p[0] != m && p[3] != m) return true;

    // edge 5
    coord.reset(ip, jpp, k);
    m = accessor.getValue(coord) < isoValue;
    if (p[4] != m && p[5] != m) return true;

    // edge 6
    coord.reset(ipp, jpp, kp);
    m = accessor.getValue(coord) < isoValue;
    if (p[5] != m && p[6] != m) return true;

    // edge 7
    coord.reset(ip, jpp, kpp);
    m = accessor.getValue(coord) < isoValue;
    if (p[6] != m && p[7] != m) return true;

    // edge 8
    coord.reset(i, jpp, kp);
    m = accessor.getValue(coord) < isoValue;
    if (p[7] != m && p[4] != m) return true;

    // edge 9
    coord.reset(i, jp, k);
    m = accessor.getValue(coord) < isoValue;
    if (p[0] != m && p[4] != m) return true;

    // edge 10
    coord.reset(ipp, jp, k);
    m = accessor.getValue(coord) < isoValue;
    if (p[1] != m && p[5] != m) return true;

    // edge 11
    coord.reset(ipp, jp, kpp);
    m = accessor.getValue(coord) < isoValue;
    if (p[2] != m && p[6] != m) return true;


    // edge 12
    coord.reset(i, jp, kpp);
    m = accessor.getValue(coord) < isoValue;
    if (p[3] != m && p[7] != m) return true;


    // evaluate faces

    // face 1
    coord.reset(ip, jp, k);
    m = accessor.getValue(coord) < isoValue;
    if (p[0] != m && p[1] != m && p[4] != m && p[5] != m) return true;

    // face 2
    coord.reset(ipp, jp, kp);
    m = accessor.getValue(coord) < isoValue;
    if (p[1] != m && p[2] != m && p[5] != m && p[6] != m) return true;

    // face 3
    coord.reset(ip, jp, kpp);
    m = accessor.getValue(coord) < isoValue;
    if (p[2] != m && p[3] != m && p[6] != m && p[7] != m) return true;

    // face 4
    coord.reset(i, jp, kp);
    m = accessor.getValue(coord) < isoValue;
    if (p[0] != m && p[3] != m && p[4] != m && p[7] != m) return true;


    // face 5
    coord.reset(ip, j, kp);
    m = accessor.getValue(coord) < isoValue;
    if (p[0] != m && p[1] != m && p[2] != m && p[3] != m) return true;

    // face 6
    coord.reset(ip, jpp, kp);
    m = accessor.getValue(coord) < isoValue;
    if (p[4] != m && p[5] != m && p[6] != m && p[7] != m) return true;

    // test cube center
    coord.reset(ip, jp, kp);
    m = accessor.getValue(coord) < isoValue;
    if (p[0] != m && p[1] != m && p[2] != m && p[3] != m &&
        p[4] != m && p[5] != m && p[6] != m && p[7] != m) return true;

    return false;
}


////////////////////////////////////////


template <class LeafType>
inline void
mergeVoxels(LeafType& leaf, const Coord& start, int dim, int regionId)
{
    Coord ijk, end = start;
    end[0] += dim;
    end[1] += dim;
    end[2] += dim;

    for (ijk[0] = start[0]; ijk[0] < end[0]; ++ijk[0]) {
        for (ijk[1] = start[1]; ijk[1] < end[1]; ++ijk[1]) {
            for (ijk[2] = start[2]; ijk[2] < end[2]; ++ijk[2]) {
                if(leaf.isValueOn(ijk)) leaf.setValue(ijk, regionId);
            }
        }
    }
}



// Note that we must use ValueType::value_type or else Visual C++ gets confused
// thinking that it is a constructor.
template <class LeafType>
inline bool
isMergable(LeafType& leaf, const Coord& start, int dim,
    typename LeafType::ValueType::value_type adaptivity)
{
    typedef typename LeafType::ValueType VecT;
    Coord ijk, end = start;
    end[0] += dim;
    end[1] += dim;
    end[2] += dim;

    std::vector<VecT> norms;
    for (ijk[0] = start[0]; ijk[0] < end[0]; ++ijk[0]) {
        for (ijk[1] = start[1]; ijk[1] < end[1]; ++ijk[1]) {
            for (ijk[2] = start[2]; ijk[2] < end[2]; ++ijk[2]) {

                if(!leaf.isValueOn(ijk)) continue;
                norms.push_back(leaf.getValue(ijk));
            }
        }
    }

    size_t N = norms.size();
    for (size_t ni = 0; ni < N; ++ni) {
        VecT n_i = norms[ni];
        for (size_t nj = 0; nj < N; ++nj) {
            VecT n_j = norms[nj];
            if ((1.0 - n_i.dot(n_j)) > adaptivity) return false;
        }
    }
    return true;
}


////////////////////////////////////////


template <class TreeT>
class LeafPtrList
{
public:
    typedef std::vector<const typename TreeT::LeafNodeType *> ListT;

    LeafPtrList(const TreeT& tree)
    {
        mLeafNodes.reserve(tree.leafCount());
        typename TreeT::LeafCIter iter = tree.cbeginLeaf();
        for ( ; iter; ++iter) mLeafNodes.push_back(iter.getLeaf());
    }

    size_t size() const { return mLeafNodes.size(); }

    const typename TreeT::LeafNodeType* operator[](size_t n) const
        { return mLeafNodes[n]; }

    tbb::blocked_range<size_t> getRange() const
        { return tbb::blocked_range<size_t>(0, mLeafNodes.size()); }

    const ListT& getList() const { return mLeafNodes; }

private:
    ListT mLeafNodes;
};


////////////////////////////////////////


template <class SourceTreeT>
class Count
{
public:
    Count(const LeafPtrList<SourceTreeT>&, std::vector<size_t>&);
    inline Count(const Count<SourceTreeT>&);

    void runParallel();
    void runSerial();

    inline void operator()(const tbb::blocked_range<size_t>&) const;
private:
    const LeafPtrList<SourceTreeT>& mLeafNodes;
    std::vector<size_t>& mLeafRegionCount;
};


template <class SourceTreeT>
Count<SourceTreeT>::Count(
    const LeafPtrList<SourceTreeT>& leafs,
    std::vector<size_t>& leafRegionCount):

    mLeafNodes(leafs),
    mLeafRegionCount(leafRegionCount)
{
}


template <class SourceTreeT>
inline
Count<SourceTreeT>::Count(const Count<SourceTreeT>& rhs):

    mLeafNodes(rhs.mLeafNodes),
    mLeafRegionCount(rhs.mLeafRegionCount)
{
}


template <class SourceTreeT>
void
Count<SourceTreeT>::runParallel()
{
    tbb::parallel_for(mLeafNodes.getRange(), *this);
}


template <class SourceTreeT>
void
Count<SourceTreeT>::runSerial()
{
    (*this)(mLeafNodes.getRange());
}


template <class SourceTreeT>
inline void
Count<SourceTreeT>::operator()(const tbb::blocked_range<size_t>& range) const
{
    for (size_t n = range.begin(); n != range.end(); ++n) {
        mLeafRegionCount[n] = size_t(mLeafNodes[n]->onVoxelCount());
    }
}


////////////////////////////////////////


template <class SourceTreeT, class AuxTreeT>
class Merge
{
public:
    typedef typename SourceTreeT::ValueType ValueType;

    Merge(const SourceTreeT&, const LeafPtrList<AuxTreeT>&, std::vector<size_t>&,
        const double iso = 0.0, const double adaptivity = 0.0);
    inline Merge(const Merge<SourceTreeT, AuxTreeT>&);

    void runParallel();
    void runSerial();

    inline void operator()(const tbb::blocked_range<size_t>&) const;

private:
    const SourceTreeT& mSourceTree;
    const LeafPtrList<AuxTreeT>& mLeafNodes;
    std::vector<size_t>& mLeafRegionCount;
    const ValueType mIsoValue, mAdaptivity;
};


template <class SourceTreeT, class AuxTreeT>
Merge<SourceTreeT, AuxTreeT>::Merge(
    const SourceTreeT& tree,
    const LeafPtrList<AuxTreeT>& leafs,
    std::vector<size_t>& leafRegionCount,
    const double iso,
    const double adaptivity):

    mSourceTree(tree),
    mLeafNodes(leafs),
    mLeafRegionCount(leafRegionCount),
    mIsoValue(ValueType(iso)),
    mAdaptivity(ValueType(adaptivity))
{
}


template <class SourceTreeT, class AuxTreeT>
inline
Merge<SourceTreeT, AuxTreeT>::Merge(const Merge<SourceTreeT, AuxTreeT>& rhs):
    mSourceTree(rhs.mSourceTree),
    mLeafNodes(rhs.mLeafNodes),
    mLeafRegionCount(rhs.mLeafRegionCount),
    mIsoValue(rhs.mIsoValue),
    mAdaptivity(rhs.mAdaptivity)
{
}


template <class SourceTreeT, class AuxTreeT>
void
Merge<SourceTreeT, AuxTreeT>::runParallel()
{
    tbb::parallel_for(mLeafNodes.getRange(), *this);
}


template <class SourceTreeT, class AuxTreeT>
void
Merge<SourceTreeT, AuxTreeT>::runSerial()
{
    (*this)(mLeafNodes.getRange());
}


template <class SourceTreeT, class AuxTreeT>
inline void
Merge<SourceTreeT, AuxTreeT>::operator()(const tbb::blocked_range<size_t>& range) const
{
    typedef math::Vec3<ValueType>  Vec3Type;

    typedef typename AuxTreeT::LeafNodeType AuxLeafT;
    typedef typename SourceTreeT::LeafNodeType::template ValueConverter<bool>::Type BoolLeafT;
    typedef typename SourceTreeT::LeafNodeType::template ValueConverter<Vec3Type>::Type Vec3LeafT;

    typedef typename AuxTreeT::LeafNodeType::ValueOnIter AuxIterT;
    typedef typename SourceTreeT::LeafNodeType::ValueOnIter SourceIterT;


    //////////
    const int LeafDim = int(BoolLeafT::DIM);
    tree::ValueAccessor<const SourceTreeT> acc(mSourceTree);
    Coord ijk, coord, origin, end;

    // Allocate reusable leaf buffers
    BoolLeafT mask;
    Vec3LeafT gradientBuffer;

    // for each auxiliary leaf
    for (size_t n = range.begin(); n != range.end(); ++n) {

        AuxLeafT& auxLeaf = *const_cast<AuxLeafT* >(mLeafNodes[n]);

        origin = auxLeaf.getOrigin();
        end[0] = origin[0] + LeafDim;
        end[1] = origin[1] + LeafDim;
        end[2] = origin[2] + LeafDim;

        mask.setValuesOff();

        /*for (AuxIterT it = auxLeaf.beginValueOn(); it; ++it) {
            ijk = it.getCoord();
            coord[0] = ijk[0] - (ijk[0] % 2);
            coord[1] = ijk[1] - (ijk[1] % 2);
            coord[2] = ijk[2] - (ijk[2] % 2);
            if(mask.isValueOn(coord)) continue;
            mask.setActiveState(coord, isAmbiguous(acc, ijk, mIsoValue, 1));
        }*/

        int dim = 2;
        // Mask off topologically ambiguous 2x2x2 voxel sub-blocks
        for (ijk[0] = origin[0]; ijk[0] < end[0]; ijk[0] += dim) {
            for (ijk[1] = origin[1]; ijk[1] < end[1]; ijk[1] += dim) {
                for (ijk[2] = origin[2]; ijk[2] < end[2]; ijk[2] += dim) {
                    if (isNonManifold(acc, ijk, mIsoValue, dim)) {
                        mask.setActiveState(ijk, true);
                    }
                }
            }
        }


        // Compute the gradient for the remaining voxels
        gradientBuffer.setValuesOff();

        for (AuxIterT it = auxLeaf.beginValueOn(); it; ++it) {

            ijk = it.getCoord();
            coord[0] = ijk[0] - (ijk[0] % dim);
            coord[1] = ijk[1] - (ijk[1] % dim);
            coord[2] = ijk[2] - (ijk[2] % dim);
            if(mask.isValueOn(coord)) continue;

            Vec3Type norm(math::ISGradient<math::CD_2ND>::result(acc, ijk));
            // Normalize (Vec3's normalize uses isApproxEqual, which uses abs and does more work)
            ValueType length = norm.length();
            if (length > ValueType(1.0e-7)) {
                norm *= ValueType(1.0) / length;
            }
            gradientBuffer.setValue(ijk, norm);
        }


        int regionId = 1, next_dim = dim << 1;

        // Process the first adaptivity level.
        for (ijk[0] = 0; ijk[0] < LeafDim; ijk[0] += dim) {
            coord[0] = ijk[0] - (ijk[0] % next_dim);
            for (ijk[1] = 0; ijk[1] < LeafDim; ijk[1] += dim) {
                coord[1] = ijk[1] - (ijk[1] % next_dim);
                for (ijk[2] = 0; ijk[2] < LeafDim; ijk[2] += dim) {
                    coord[2] = ijk[2] - (ijk[2] % next_dim);
                    if(mask.isValueOn(ijk) || !isMergable(gradientBuffer, ijk, dim, mAdaptivity)) {
                        mask.setActiveState(coord, true);
                        continue;
                    }
                    mergeVoxels(auxLeaf, ijk, dim, regionId++);
                }
            }
        }

        // Process remaining adaptivity levels
        for (dim = 4; dim < LeafDim; dim = dim << 1) {
            next_dim = dim << 1;
            coord[0] = ijk[0] - (ijk[0] % next_dim);
            for (ijk[0] = origin[0]; ijk[0] < end[0]; ijk[0] += dim) {
                coord[1] = ijk[1] - (ijk[1] % next_dim);
                for (ijk[1] = origin[1]; ijk[1] < end[1]; ijk[1] += dim) {
                    coord[2] = ijk[2] - (ijk[2] % next_dim);
                    for (ijk[2] = origin[2]; ijk[2] < end[2]; ijk[2] += dim) {

                        if (mask.isValueOn(ijk) || isNonManifold(acc, ijk, mIsoValue, dim) ||
                            !isMergable(gradientBuffer, ijk, dim, mAdaptivity)) {
                            mask.setActiveState(coord, true);
                            continue;
                        }
                        mergeVoxels(auxLeaf, ijk, dim, regionId++);
                    }
                }
            }
        }

        if (!(mask.isValueOn(origin) || isNonManifold(acc, origin, mIsoValue, LeafDim))
            && isMergable(gradientBuffer, origin, LeafDim, mAdaptivity)) {
            mergeVoxels(auxLeaf, origin, LeafDim, regionId++);
        }

        // Count unique regions
        size_t numVoxels = 0;
        AuxLeafT tmpLeaf(auxLeaf);
        for (AuxIterT it = tmpLeaf.beginValueOn(); it; ++it) {
            if(it.getValue() == 0) {
                it.setValueOff();
                ++numVoxels;
            }
        }

        while(tmpLeaf.onVoxelCount() > 0) {
            ++numVoxels;
            AuxIterT it = tmpLeaf.beginValueOn();
            regionId = it.getValue();
            for (; it; ++it) {
                if(it.getValue() == regionId) it.setValueOff();
            }
        }

        mLeafRegionCount[n] = numVoxels;
    }
}


////////////////////////////////////////


template <class SourceTreeT, class AuxTreeT>
class PointGen
{
public:
    PointGen(const LeafPtrList<AuxTreeT>&,
        std::vector<size_t>&,
        const SourceTreeT&,
        const openvdb::math::Transform&,
        PointList&,
        double iso = 0.0);

    PointGen(const PointGen&);

    void runParallel();
    void runSerial();

    void operator()(const tbb::blocked_range<size_t>&) const;
private:
    const LeafPtrList<AuxTreeT>& mLeafNodes;
    const std::vector<size_t>& mLeafIdx;

    const SourceTreeT& mTree;
    const openvdb::math::Transform& mTransform;

    const PointList& mPoints;
    const double mIsoValue;

    double root(double v0, double v1) const { return (mIsoValue - v0) / (v1 - v0); }
    void calcAvgPoint(const tree::ValueAccessor<const SourceTreeT>&, const Coord&,
        openvdb::Vec3d&) const;
};


template <class SourceTreeT, class AuxTreeT>
PointGen<SourceTreeT, AuxTreeT>::PointGen(
    const LeafPtrList<AuxTreeT>& leafs,
    std::vector<size_t>& leafIdx,
    const SourceTreeT& tree,
    const openvdb::math::Transform& transform,
    PointList& points,
    double iso):

    mLeafNodes(leafs),
    mLeafIdx(leafIdx),
    mTree(tree),
    mTransform(transform),
    mPoints(points),
    mIsoValue(iso)
{
}


template <class SourceTreeT, class AuxTreeT>
PointGen<SourceTreeT, AuxTreeT>::PointGen(const PointGen& rhs):

    mLeafNodes(rhs.mLeafNodes),
    mLeafIdx(rhs.mLeafIdx),
    mTree(rhs.mTree),
    mTransform(rhs.mTransform),
    mPoints(rhs.mPoints),
    mIsoValue(rhs.mIsoValue)
{
}


template <class SourceTreeT, class AuxTreeT>
void
PointGen<SourceTreeT, AuxTreeT>::runParallel()
{
    tbb::parallel_for(mLeafNodes.getRange(), *this);
}


template <class SourceTreeT, class AuxTreeT>
void
PointGen<SourceTreeT, AuxTreeT>::runSerial()
{
    (*this)(mLeafNodes.getRange());
}


template <class SourceTreeT, class AuxTreeT>
void
PointGen<SourceTreeT, AuxTreeT>::operator()(const tbb::blocked_range<size_t>& range) const
{
    openvdb::tree::ValueAccessor<const SourceTreeT> acc(mTree);

    typename AuxTreeT::LeafNodeType::ValueOnIter iter;

    Coord ijk;
    openvdb::Vec3d avg, tmp;

    for (size_t n = range.begin(); n != range.end(); ++n) {

        size_t idx = mLeafIdx[n];
        typename AuxTreeT::LeafNodeType& leaf =
            *const_cast<typename AuxTreeT::LeafNodeType*>(mLeafNodes[n]);
        iter = leaf.beginValueOn();

        for (; iter; ++iter) {

            if(iter.getValue() == 0) {

                iter.setValue(idx);
                iter.setValueOff();
                ijk = iter.getCoord();

                calcAvgPoint(acc, ijk, avg);

                avg = mTransform.indexToWorld(avg);

                openvdb::Vec3s& ptn = mPoints[idx];
                ptn[0] = float(avg[0]);
                ptn[1] = float(avg[1]);
                ptn[2] = float(avg[2]);

                ++idx;
            }
        }

        while(leaf.onVoxelCount() > 0) {

            avg[0] = 0;
            avg[1] = 0;
            avg[2] = 0;

            iter = leaf.beginValueOn();
            int regionId = iter.getValue(), points = 0;

            for (; iter; ++iter) {

                if(iter.getValue() == regionId) {

                    iter.setValue(idx);
                    iter.setValueOff();
                    ijk = iter.getCoord();

                    calcAvgPoint(acc, ijk, tmp);

                    avg += mTransform.indexToWorld(tmp);
                    ++points;
                }
            }

            if (points > 1) {
                double w = 1.0 / double(points);
                avg[0] *= w;
                avg[1] *= w;
                avg[2] *= w;
            }

            openvdb::Vec3s& ptn = mPoints[idx];
            ptn[0] = float(avg[0]);
            ptn[1] = float(avg[1]);
            ptn[2] = float(avg[2]);
            ++idx;
        }
    }
}


template <class SourceTreeT, class AuxTreeT>
void
PointGen<SourceTreeT, AuxTreeT>::calcAvgPoint(const tree::ValueAccessor<const SourceTreeT>& acc,
    const Coord& ijk, openvdb::Vec3d& avg) const
{
    double values[8];
    bool signMask[8];
    Coord coord;

    // Sample corner values
    coord = ijk;
    values[0] = double(acc.getValue(coord)); // i, j, k

    coord[0] += 1;
    values[1] = double(acc.getValue(coord)); // i+1, j, k

    coord[2] += 1;
    values[2] = double(acc.getValue(coord)); // i+i, j, k+1

    coord[0] = ijk[0];
    values[3] = double(acc.getValue(coord)); // i, j, k+1

    coord[1] += 1; coord[2] = ijk[2];
    values[4] = double(acc.getValue(coord)); // i, j+1, k

    coord[0] += 1;
    values[5] = double(acc.getValue(coord)); // i+1, j+1, k

    coord[2] += 1;
    values[6] = double(acc.getValue(coord)); // i+1, j+1, k+1

    coord[0] = ijk[0];
    values[7] = double(acc.getValue(coord)); // i, j+1, k+1

    // init sign mask
    for (int n = 0; n < 8; ++n) signMask[n] = (values[n] < mIsoValue);

    int samples = 0;
    avg[0] = 0.0;
    avg[1] = 0.0;
    avg[2] = 0.0;

    if (signMask[0] != signMask[1]) { // Edged: 0 - 1
        avg[0] += root(values[0], values[1]);
        ++samples;
    }

    if (signMask[1] != signMask[2]) { // Edged: 1 - 2
        avg[0] += 1.0;
        avg[2] += root(values[1], values[2]);
        ++samples;
    }

    if (signMask[3] != signMask[2]) { // Edged: 3 - 2
        avg[0] += root(values[3], values[2]);
        avg[2] += 1.0;
        ++samples;
    }

    if (signMask[0] != signMask[3]) { // Edged: 0 - 3
        avg[2] += root(values[0], values[3]);
        ++samples;
    }

    if (signMask[4] != signMask[5]) { // Edged: 4 - 5
        avg[0] += root(values[4], values[5]);
        avg[1] += 1.0;
        ++samples;
    }

    if (signMask[5] != signMask[6]) { // Edged: 5 - 6
        avg[0] += 1.0;
        avg[1] += 1.0;
        avg[2] += root(values[5], values[6]);
        ++samples;
    }

    if (signMask[7] != signMask[6]) { // Edged: 7 - 6
        avg[0] += root(values[7], values[6]);
        avg[1] += 1.0;
        avg[2] += 1.0;
        ++samples;
    }

    if (signMask[4] != signMask[7]) { // Edged: 4 - 7
        avg[1] += 1.0;
        avg[2] += root(values[4], values[7]);
        ++samples;
    }

    if (signMask[0] != signMask[4]) { // Edged: 0 - 4
        avg[1] += root(values[0], values[4]);
        ++samples;
    }

    if (signMask[1] != signMask[5]) { // Edged: 1 - 5
        avg[0] += 1.0;
        avg[1] += root(values[1], values[5]);
        ++samples;
    }

    if (signMask[2] != signMask[6]) { // Edged: 2 - 6
        avg[0] += 1.0;
        avg[1] += root(values[2], values[6]);
        avg[2] += 1.0;
        ++samples;
    }

    if (signMask[3] != signMask[7]) { // Edged: 3 - 7
        avg[1] += root(values[3], values[7]);
        avg[2] += 1.0;
        ++samples;
    }

    if (samples > 1) {
        double w = 1.0 / double(samples);
        avg[0] *= w;
        avg[1] *= w;
        avg[2] *= w;
    }

    // offset by cell-origin
    avg[0] += double(ijk[0]);
    avg[1] += double(ijk[1]);
    avg[2] += double(ijk[2]);
}


////////////////////////////////////////


struct QuadMeshOp
{
    QuadMeshOp(): mIdx(0), mQuadPool(NULL) {}

    void init(const size_t upperBound, QuadPool& quadPool)
    {
        mQuadPool = &quadPool;
        mQuadPool->reset(upperBound);
        mIdx = 0;
    }

    void addPrim(const Vec4I& verts, bool reverse)
    {
        if (!reverse) {
            (*mQuadPool)[mIdx] = verts;
        } else {
            Vec4I& quad = (*mQuadPool)[mIdx];
            quad[0] = verts[3];
            quad[1] = verts[2];
            quad[2] = verts[1];
            quad[3] = verts[0];
        }
        ++mIdx;
    }

    void done() {}

private:
    size_t mIdx;
    QuadPool* mQuadPool;
};


struct AdaptiveMeshOp
{
    AdaptiveMeshOp(): mIdx(0), mQuadPool(NULL) {}

    void init(const size_t upperBound, QuadPool& quadPool)
    {
        mQuadPool = &quadPool;
        mTmpQuadPool.reset(upperBound);
        mIdx = 0;
    }

    void addPrim(const Vec4I& verts, bool reverse)
    {
        if (verts[0] != verts[1] && verts[0] != verts[2] && verts[0] != verts[3]
            && verts[1] != verts[2] && verts[1] != verts[3] && verts[2] != verts[3]) {
            addQuad(verts, reverse);
        } else if (
            verts[0] == verts[3] &&
            verts[1] != verts[2] &&
            verts[1] != verts[0] &&
            verts[2] != verts[0]) {
            addTriangle(verts[0], verts[1], verts[2], reverse);
        } else if (
            verts[1] == verts[2] &&
            verts[0] != verts[3] &&
            verts[0] != verts[1] &&
            verts[3] != verts[1]) {
            addTriangle(verts[0], verts[1], verts[3], reverse);
        } else if (
            verts[0] == verts[1] &&
            verts[2] != verts[3] &&
            verts[2] != verts[0] &&
            verts[3] != verts[0]) {
            addTriangle(verts[0], verts[2], verts[3], reverse);
        } else if (
            verts[2] == verts[3] &&
            verts[0] != verts[1] &&
            verts[0] != verts[2] &&
            verts[1] != verts[2]) {
            addTriangle(verts[0], verts[1], verts[2], reverse);
        }
    }


    void done()
    {
        mQuadPool->reset(mIdx);
        for (size_t  i = 0; i < mIdx; ++i) {
            (*mQuadPool)[i] = mTmpQuadPool[i];
        }
    }

private:

    void addQuad(const Vec4I& verts, bool reverse)
    {
        if (!reverse) {
            mTmpQuadPool[mIdx] = verts;
        } else {
            Vec4I& quad = mTmpQuadPool[mIdx];
            quad[0] = verts[3];
            quad[1] = verts[2];
            quad[2] = verts[1];
            quad[3] = verts[0];
        }
        ++mIdx;
    }

    void addTriangle(unsigned v0, unsigned v1, unsigned v2, bool reverse)
    {
        Vec4I& prim = mTmpQuadPool[mIdx];

        prim[1] = v1;
        prim[3] = util::INVALID_IDX;

        if (!reverse) {
            prim[0] = v0;
            prim[2] = v2;
        } else {
            prim[0] = v2;
            prim[2] = v0;
        }
        ++mIdx;
    }


    size_t mIdx;
    QuadPool *mQuadPool;
    QuadPool mTmpQuadPool;
};


////////////////////////////////////////


template<class EdgeTreeT, class AuxTreeT, class MeshingOp>
class MeshGen
{
public:
    MeshGen(const LeafPtrList<EdgeTreeT>&, const AuxTreeT&, QuadPoolList&);
    MeshGen(const MeshGen&);

    void runParallel();
    void runSerial();

    void operator()(const tbb::blocked_range<size_t>&) const;

private:
    const LeafPtrList<EdgeTreeT>& mLeafNodes;
    const AuxTreeT& mAuxTree;
    const QuadPoolList& mQuadPoolList;
    size_t mID;
};


template<class EdgeTreeT, class AuxTreeT, class MeshingOp>
MeshGen<EdgeTreeT, AuxTreeT, MeshingOp>::MeshGen(
    const LeafPtrList<EdgeTreeT>& leafs,
    const AuxTreeT& auxTree,
    QuadPoolList& quadPoolList):

    mLeafNodes(leafs),
    mAuxTree(auxTree),
    mQuadPoolList(quadPoolList)
{
}


template<class EdgeTreeT, class AuxTreeT, class MeshingOp>
MeshGen<EdgeTreeT, AuxTreeT, MeshingOp>::MeshGen(const MeshGen& rhs):

    mLeafNodes(rhs.mLeafNodes),
    mAuxTree(rhs.mAuxTree),
    mQuadPoolList(rhs.mQuadPoolList)
{
}


template<class EdgeTreeT, class AuxTreeT, class MeshingOp>
void
MeshGen<EdgeTreeT, AuxTreeT, MeshingOp>::runParallel()
{
    tbb::parallel_for(mLeafNodes.getRange(), *this);
}


template<class EdgeTreeT, class AuxTreeT, class MeshingOp>
void
MeshGen<EdgeTreeT, AuxTreeT, MeshingOp>::runSerial()
{
    (*this)(mLeafNodes.getRange());
}


template<class EdgeTreeT, class AuxTreeT, class MeshingOp>
void
MeshGen<EdgeTreeT, AuxTreeT, MeshingOp>::operator()(
    const tbb::blocked_range<size_t>& range) const
{
    openvdb::tree::ValueAccessor<const AuxTreeT> acc(mAuxTree);

    typename EdgeTreeT::LeafNodeType::ValueOnCIter iter;
    Coord ijk, coord;
    char edgeFlags;
    openvdb::Vec4I quad;
    size_t edgeCount;

    MeshingOp mesher;

    for (size_t n = range.begin(); n != range.end(); ++n) {

        // Get an upper bound on the number of .
        edgeCount = 0;
        iter = mLeafNodes[n]->cbeginValueOn();
        for (; iter; ++iter) {
            edgeFlags = iter.getValue() >> 1;
            edgeCount += edgeFlags & 0x1;

            edgeFlags = edgeFlags >> 1;
            edgeCount += edgeFlags & 0x1;

            edgeFlags = edgeFlags >> 1;
            edgeCount += edgeFlags & 0x1;
        }

        mesher.init(edgeCount, mQuadPoolList[n]);

        iter = mLeafNodes[n]->cbeginValueOn();
        for (; iter; ++iter) {
            ijk = iter.getCoord();
            edgeFlags = iter.getValue();

            const bool isInside = edgeFlags & INSIDE;
            int v0 = acc.getValue(ijk);

            if (edgeFlags & XEDGE) {

                quad[0] = v0;
                coord[0] = ijk[0]; coord[1] = ijk[1]-1; coord[2] = ijk[2]; // i, j-1, k
                quad[1] = acc.getValue(coord);
                coord[2] -= 1; // i, j-1, k-1
                quad[2] = acc.getValue(coord);
                coord[1] = ijk[1]; // i, j, k-1
                quad[3] = acc.getValue(coord);

                mesher.addPrim(quad, isInside);
            }


            if (edgeFlags & YEDGE) {
                quad[0] = v0;

                coord[0] = ijk[0]; coord[1] = ijk[1]; coord[2] = ijk[2]-1; // i, j, k-1
                quad[1] = acc.getValue(coord);
                coord[0] -= 1; // i-1, j, k-1
                quad[2] = acc.getValue(coord);
                coord[2] = ijk[2]; // i-1, j, k
                quad[3] = acc.getValue(coord);

                mesher.addPrim(quad, isInside);
            }

            if (edgeFlags & ZEDGE) {
                quad[0] = v0;

                coord[0] = ijk[0]; coord[1] = ijk[1]-1; coord[2] = ijk[2]; // i, j-1, k
                quad[1] = acc.getValue(coord);
                coord[0] -= 1; // i-1, j-1, k
                quad[2] = acc.getValue(coord);
                coord[1] = ijk[1]; // i, j, k
                quad[3] = acc.getValue(coord);

                mesher.addPrim(quad, !isInside);
            }
        }

        mesher.done();
    }
}


////////////////////////////////////////


template<class SourceTreeT>
class AuxiliaryData
{
public:
    typedef openvdb::tree::ValueAccessor<const SourceTreeT> SourceAccessorT;
    typedef typename SourceTreeT::ValueType ValueT;

    typedef typename SourceTreeT::template ValueConverter<char>::Type EdgeTreeT;
    typedef openvdb::tree::ValueAccessor<EdgeTreeT> EdgeAccessorT;

    typedef typename SourceTreeT::template ValueConverter<int>::Type AuxTreeT;
    typedef openvdb::tree::ValueAccessor<AuxTreeT> AuxAccessorT;

    AuxiliaryData(const SourceTreeT&, const LeafPtrList<SourceTreeT>&, double iso = 0.0);
    AuxiliaryData(AuxiliaryData&, tbb::split);

    void runParallel();
    void runSerial();

    typename EdgeTreeT::Ptr edgeTree() const { return mEdgeTree; }
    typename AuxTreeT::Ptr auxTree() const { return mAuxTree; }

    void operator()(const tbb::blocked_range<size_t>&);

    void join(const AuxiliaryData& rhs)
    {
        mEdgeTree->merge(*rhs.mEdgeTree);
        mAuxTree->merge(*rhs.mAuxTree);
    }

private:

    const LeafPtrList<SourceTreeT>& mLeafNodes;
    const SourceTreeT& mSourceTree;
    SourceAccessorT mSourceAccessor;

    typename EdgeTreeT::Ptr mEdgeTree;
    EdgeAccessorT mEdgeAccessor;

    typename AuxTreeT::Ptr mAuxTree;
    AuxAccessorT mAuxAccessor;

    const double mIsoValue;
};

template<class SourceTreeT>
AuxiliaryData<SourceTreeT>::AuxiliaryData(
    const SourceTreeT& tree,
    const LeafPtrList<SourceTreeT>& leafNodes,
    double iso):
    mLeafNodes(leafNodes),
    mSourceTree(tree),
    mSourceAccessor(mSourceTree),
    mEdgeTree(new EdgeTreeT(0)),
    mEdgeAccessor(*mEdgeTree),
    mAuxTree(new AuxTreeT(0)),
    mAuxAccessor(*mAuxTree),
    mIsoValue(iso)
{
}

template<class SourceTreeT>
AuxiliaryData<SourceTreeT>::AuxiliaryData(AuxiliaryData& rhs, tbb::split):
    mLeafNodes(rhs.mLeafNodes),
    mSourceTree(rhs.mSourceTree),
    mSourceAccessor(mSourceTree),
    mEdgeTree(new EdgeTreeT(0)),
    mEdgeAccessor(*mEdgeTree),
    mAuxTree(new AuxTreeT(0)),
    mAuxAccessor(*mAuxTree),
    mIsoValue(rhs.mIsoValue)
{
}



template<class SourceTreeT>
void
AuxiliaryData<SourceTreeT>::runParallel()
{
    tbb::parallel_reduce(mLeafNodes.getRange(), *this);
}

template<class SourceTreeT>
void
AuxiliaryData<SourceTreeT>::runSerial()
{
    (*this)(mLeafNodes.getRange());
}

template<class SourceTreeT>
void
AuxiliaryData<SourceTreeT>::operator()(const tbb::blocked_range<size_t>& range)
{
    typedef openvdb::Coord Coord;
    typename SourceTreeT::LeafNodeType::ValueOnCIter iter;
    Coord ijk, n_ijk, coord;
    ValueT value;
    bool thisInside, otherInside;
    int edgeFlags;

    for (size_t n = range.begin(); n != range.end(); ++n) {

        iter = mLeafNodes[n]->cbeginValueOn();
        for (; iter; ++iter) {

            ijk = iter.getCoord();

            value = iter.getValue();
            thisInside = (value < mIsoValue);
            edgeFlags = 0;

            // Eval upwind x-edge
            n_ijk = ijk; ++n_ijk[0];
            otherInside = (mSourceAccessor.getValue(n_ijk) < mIsoValue);
            if (otherInside != thisInside) {

                edgeFlags = XEDGE;

                mAuxAccessor.setActiveState(ijk, true);

                coord[0] = ijk[0]; coord[1] = ijk[1]-1; coord[2] = ijk[2];
                mAuxAccessor.setActiveState(coord, true);

                coord[0] = ijk[0]; coord[1] = ijk[1]-1; coord[2] = ijk[2]-1;
                mAuxAccessor.setActiveState(coord, true);

                coord[0] = ijk[0]; coord[1] = ijk[1]; coord[2] = ijk[2]-1;
                mAuxAccessor.setActiveState(coord, true);
            }

            // Eval upwind y-edge
            n_ijk[0] = ijk[0]; ++n_ijk[1];
            otherInside = (mSourceAccessor.getValue(n_ijk) < mIsoValue);
            if (otherInside != thisInside) {

                edgeFlags |= YEDGE;

                mAuxAccessor.setActiveState(ijk, true);

                coord[0] = ijk[0]; coord[1] = ijk[1]; coord[2] = ijk[2]-1;
                mAuxAccessor.setActiveState(coord, true);

                coord[0] = ijk[0]-1; coord[1] = ijk[1]; coord[2] = ijk[2];
                mAuxAccessor.setActiveState(coord, true);

                coord[0] = ijk[0]-1; coord[1] = ijk[1]; coord[2] = ijk[2]-1;
                mAuxAccessor.setActiveState(coord, true);
            }

            // Eval upwind z-edge
            n_ijk[1] = ijk[1]; ++n_ijk[2];
            otherInside = (mSourceAccessor.getValue(n_ijk) < mIsoValue);
            if (otherInside != thisInside) {

                edgeFlags |= ZEDGE;

                mAuxAccessor.setActiveState(ijk, true);

                coord[0] = ijk[0]; coord[1] = ijk[1]-1; coord[2] = ijk[2];
                mAuxAccessor.setActiveState(coord, true);

                coord[0] = ijk[0]-1; coord[1] = ijk[1]; coord[2] = ijk[2];
                mAuxAccessor.setActiveState(coord, true);

                coord[0] = ijk[0]-1; coord[1] = ijk[1]-1; coord[2] = ijk[2];
                mAuxAccessor.setActiveState(coord, true);
            }

            if (edgeFlags != 0) {
                edgeFlags |= int(thisInside);
                mEdgeAccessor.setValue(ijk, char(edgeFlags));
            }
        }
    }
}

} // namespace internal


////////////////////////////////////////


template<typename GridT>
inline void
LevelSetMesher::operator()(const GridT& sourceGrid)
{
    typedef typename GridT::TreeType SourceTreeT;
    typedef typename GridT::TreeType SourceValueT;
    typedef typename internal::AuxiliaryData<SourceTreeT>::EdgeTreeT EdgeTreeT;
    typedef typename internal::AuxiliaryData<SourceTreeT>::AuxTreeT AuxTreeT;

    typename EdgeTreeT::Ptr edgeTree; // upwind-edge flags
    typename AuxTreeT::Ptr auxTree; // auxiliary data

    const openvdb::math::Transform& transform = sourceGrid.transform();
    const SourceTreeT& sourceTree = sourceGrid.tree();

    // Collect auxiliary data
    {
        internal::LeafPtrList<SourceTreeT> sourceLeafs(sourceTree);

        internal::AuxiliaryData<SourceTreeT> collect(sourceTree, sourceLeafs, mIsoValue);
        collect.runParallel();

        edgeTree = collect.edgeTree();
        auxTree = collect.auxTree();
    }


    // Process auxiliary data
    {
        internal::LeafPtrList<AuxTreeT> auxLeafs(*auxTree);
        std::vector<size_t> regions(auxLeafs.size(), 0);

        {
            if (mAdaptivity < 1e-4) {
                internal::Count<AuxTreeT> count(auxLeafs, regions);
                count.runParallel();
            } else {
                internal::Merge<SourceTreeT, AuxTreeT>
                    merge(sourceTree, auxLeafs, regions, mIsoValue, mAdaptivity);
                merge.runParallel();
            }

            mPointListSize = 0;
            size_t tmp = 0;
            for (size_t n = 0, N = regions.size(); n < N; ++n) {
                tmp = regions[n];
                regions[n] = mPointListSize;
                mPointListSize += tmp;
            }
        }

        // Generate the unique point list
        mPoints.reset(new openvdb::Vec3s[mPointListSize]);

        internal::PointGen<SourceTreeT, AuxTreeT>
            pointGen(auxLeafs, regions, sourceTree, transform, mPoints, mIsoValue);
        pointGen.runParallel();
    }

    // Generate quad mesh
    {
        internal::LeafPtrList<EdgeTreeT> edgeLeafs(*edgeTree);
        mQuadPoolListSize = edgeLeafs.size();

        mQuads.reset(new QuadPool[mQuadPoolListSize]);

        if (mAdaptivity < 1e-4) {
            internal::MeshGen<EdgeTreeT, AuxTreeT, internal::QuadMeshOp>
                meshGen(edgeLeafs, *auxTree, mQuads);
            meshGen.runParallel();
        } else {
            internal::MeshGen<EdgeTreeT, AuxTreeT, internal::AdaptiveMeshOp>
                meshGen(edgeLeafs, *auxTree, mQuads);
            meshGen.runParallel();
        }
    }
}


////////////////////////////////////////


} // namespace tools
} // namespace OPENVDB_VERSION_NAME
} // namespace openvdb

#endif // OPENVDB_TOOLS_LEVEL_SET_MESHER_HAS_BEEN_INCLUDED

// Copyright (c) 2012 DreamWorks Animation LLC
// All rights reserved. This software is distributed under the
// Mozilla Public License 2.0 ( http://www.mozilla.org/MPL/2.0/ )
