///////////////////////////////////////////////////////////////////////////
//
// Copyright (c) 2012 DreamWorks Animation LLC
//
// All rights reserved. This software is distributed under the
// Mozilla Public License 2.0 ( http://www.mozilla.org/MPL/2.0/ )
//
// Redistributions of source code must retain the above copyright
// and license notice and the following restrictions and disclaimer.
//
// *     Neither the name of DreamWorks Animation nor the names of
// its contributors may be used to endorse or promote products derived
// from this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
// OWNER OR CONTRIBUTORS BE LIABLE FOR ANY INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
// IN NO EVENT SHALL THE COPYRIGHT HOLDERS' AND CONTRIBUTORS' AGGREGATE
// LIABILITY FOR ALL CLAIMS REGARDLESS OF THEIR BASIS EXCEED US$250.00.
//
///////////////////////////////////////////////////////////////////////////
//
/// @author Ken Museth
///
/// @file LevelSetFilter.h
///
/// @brief Performs various types of level set deformations with
/// interface tracking. These unrestricted deformations include
/// surface smoothing (e.g., Laplacian flow), filtering (e.g., mean
/// value) and morphological operations (e.g., morphological opening).

#ifndef OPENVDB_TOOLS_LEVELSETFILTER_HAS_BEEN_INCLUDED
#define OPENVDB_TOOLS_LEVELSETFILTER_HAS_BEEN_INCLUDED

#include "LevelSetTracker.h"

namespace openvdb {
OPENVDB_USE_VERSION_NAMESPACE
namespace OPENVDB_VERSION_NAME {
namespace tools {

/// @brief Filtering (i.e. diffusion) of narrow-band level sets
template<typename _GridType,
         typename InterruptType=util::NullInterrupter>
class LevelSetFilter : public LevelSetTracker<_GridType, InterruptType>
{
public:
    typedef boost::shared_ptr<LevelSetFilter> Ptr;
    typedef LevelSetTracker<_GridType, InterruptType>  BaseType;
    typedef _GridType                                  GridType;
    typedef typename GridType::TreeType                TreeType;
    typedef typename TreeType::LeafNodeType            LeafType;
    typedef typename LeafType::ValueType               ValueType;
    typedef typename tree::LeafManager<TreeType>       ManagerType;
    typedef typename ManagerType::RangeType            RangeType;

    /// Main constructor
    LevelSetFilter(GridType& grid, InterruptType& interrupt);

    /// Shallow copy constructor called by tbb::parallel_for() threads during filtering
    LevelSetFilter(const LevelSetFilter& other);

    virtual ~LevelSetFilter() {};

    /// Used internally by tbb::parallel_for()
    void operator()(const RangeType& r) const
    {
        if (mTask) mTask(const_cast<LevelSetFilter*>(this), r);
        else OPENVDB_THROW(ValueError, "task is undefined - call offset(), etc");
    }

    /// @brief Perform mean-curvature flow of the level set
    void meanCurvature();

    /// @brief Perform laplacian flow of the level set
    void laplacian();

    /// @brief Offset the level set by the specified (world) distance
    void offset(ValueType offset);

    /// @brief Perform median-value flow of the level set
    void median(int width = 1);

    /// @brief Perform mean-value flow of the level set
    void mean(int width = 1);

private:
    typedef typename boost::function<void (LevelSetFilter*, const RangeType&)> FuncType;

    FuncType mTask;

    // Private cook method calling tbb::parallel_for
    void cook(int swapBuffer=0);

    // Private methods called by tbb::parallel_for threads
    void doMedian(const RangeType&, int);
    void doMean(const RangeType&, int);
    void doMeanCurvature(const RangeType&);
    void doLaplacian(const RangeType&);
    void doOffset(const RangeType&, ValueType);

}; // end of LevelSetFilter class


////////////////////////////////////////


template<typename _GridType, typename InterruptT>
LevelSetFilter<_GridType, InterruptT>::LevelSetFilter(_GridType& grid, InterruptT& interrupt):
    BaseType(grid, interrupt),
    mTask(0)
{
}


template<typename GridT, typename InterruptT>
LevelSetFilter<GridT, InterruptT>::LevelSetFilter(const LevelSetFilter& other):
    BaseType(other),
    mTask(other.mTask)
{
}


////////////////////////////////////////

template<typename GridT, typename InterruptT>
inline void
LevelSetFilter<GridT, InterruptT>::median(int width)
{
    mTask = boost::bind(&LevelSetFilter::doMedian, _1, _2, std::max(1, width));
    this->cook(1);// one auxiliary buffer required
}


template<typename GridT, typename InterruptT>
inline void
LevelSetFilter<GridT, InterruptT>::mean(int width)
{
    mTask = boost::bind(&LevelSetFilter::doMean, _1, _2, std::max(1, width));
    this->cook(1);// one auxiliary buffer required
}


template<typename GridT, typename InterruptT>
inline void
LevelSetFilter<GridT, InterruptT>::meanCurvature()
{
    mTask = boost::bind(&LevelSetFilter::doMeanCurvature, _1, _2);
    this->cook(1);// one auxiliary buffer required
}

template<typename GridT, typename InterruptT>
inline void
LevelSetFilter<GridT, InterruptT>::laplacian()
{
    mTask = boost::bind(&LevelSetFilter::doLaplacian, _1, _2);
    this->cook(1);;// one auxiliary buffer required
}


template<typename GridT, typename InterruptT>
inline void
LevelSetFilter<GridT, InterruptT>::offset(ValueType value)
{
    const ValueType CFL = ValueType(0.5) * BaseType::voxelSize(), offset = openvdb::math::Abs(value);
    ValueType dist = 0.0;
    while (offset-dist > ValueType(0.001)*CFL) {
        const ValueType delta = openvdb::math::Min(offset-dist, CFL);
        dist += delta;
        mTask = boost::bind(&LevelSetFilter::doOffset, _1, _2, copysign(delta,value));
        this->cook(0);// no auxiliary buffers required
    }
}


///////////////////////// PRIVATE METHODS //////////////////////


/// Private method to perform the propagation task (serial or threaded)
template<typename GridT, typename InterruptT>
inline void
LevelSetFilter<GridT, InterruptT>::cook(int swapBuffer)
{
    BaseType::startInterrupter("Propagating level set");

    BaseType::mLeafs->rebuildAuxBuffers(swapBuffer);

    if (BaseType::getGrainSize()>0) {
        tbb::parallel_for(BaseType::mLeafs->getRange(BaseType::getGrainSize()), *this);
    } else {
        (*this)(BaseType::mLeafs->getRange());
    }

    BaseType::mLeafs->swapLeafBuffer(swapBuffer, BaseType::getGrainSize()==0);

    BaseType::mLeafs->removeAuxBuffers();

    BaseType::track();
}

/// Performs parabolic mean-curvature diffusion
template<typename GridT, typename InterruptT>
inline void
LevelSetFilter<GridT, InterruptT>::doMeanCurvature(const RangeType& range)
{
    typedef typename LeafType::ValueOnCIter VoxelIterT;
    BaseType::checkInterrupter();
    const ValueType dx = BaseType::voxelSize(),dt = math::Pow2(dx) / ValueType(3.0);
    math::CurvatureStencil<GridType> stencil(BaseType::mGrid, dx);
    for (size_t n=range.begin(), e=range.end(); n != e; ++n) {
        typename BaseType::BufferType& buffer = BaseType::mLeafs->getBuffer(n,1);
        for (VoxelIterT iter = BaseType::mLeafs->leaf(n).cbeginValueOn(); iter; ++iter) {
            stencil.moveTo(iter);
            buffer.setValue(iter.pos(), stencil.getValue() + dt * stencil.meanCurvatureNormGrad());
        }
    }
}

/// Performs laplacian diffusion. Note if the grids contains a true
/// signed distance field (e.g. a solution to the Eikonal equation)
/// Laplacian diffusions (e.g. geometric heat equation) is actually
/// identical to mean curvature diffusion, yet less computationally
/// expensive! In other words if you're performing renormalization
/// anyway (e.g. rebuilding the narrow-band) you should consider
/// performing laplacian diffusion over mean curvature flow!
template<typename GridT, typename InterruptT>
inline void
LevelSetFilter<GridT, InterruptT>::doLaplacian(const RangeType& range)
{
    typedef typename LeafType::ValueOnCIter VoxelIterT;
    BaseType::checkInterrupter();
    const ValueType dx = BaseType::voxelSize(), dt = math::Pow2(dx) / ValueType(6.0);
    math::GradStencil<GridType> stencil(BaseType::mGrid, dx);
    for (size_t n=range.begin(), e=range.end(); n != e; ++n) {
        typename BaseType::BufferType& buffer = BaseType::mLeafs->getBuffer(n,1);
        for (VoxelIterT iter = BaseType::mLeafs->leaf(n).cbeginValueOn(); iter; ++iter) {
            stencil.moveTo(iter);
            buffer.setValue(iter.pos(), stencil.getValue() + dt * stencil.laplacian());
        }
    }
}

/// Offsets the values by a constant
template<typename GridT, typename InterruptT>
inline void
LevelSetFilter<GridT, InterruptT>::doOffset(const RangeType& range, ValueType value)
{
    BaseType::checkInterrupter();
    for (size_t n=range.begin(), e=range.end(); n != e; ++n)
        BaseType::mLeafs->leaf(n).addValue(value);
}

/// Performs simple but fast mean-value diffusion. Note this can be
/// shown to be related to Laplacian flow.
template<typename GridT, typename InterruptT>
inline void
LevelSetFilter<GridT, InterruptT>::doMean(const RangeType& range, int width)
{
    typedef typename LeafType::ValueOnCIter VoxelIterT;
    BaseType::checkInterrupter();
    math::DenseStencil<GridType> stencil(BaseType::mGrid, width);//creates local cache!
    for (size_t n=range.begin(), e=range.end(); n != e; ++n) {
        typename BaseType::BufferType& buffer = BaseType::mLeafs->getBuffer(n,1);
        for (VoxelIterT iter=BaseType::mLeafs->leaf(n).cbeginValueOn(); iter; ++iter) {
            stencil.moveTo(iter);
            buffer.setValue(iter.pos(), stencil.mean());
        }
    }
}

/// Performs simple but fast median-value diffusion
template<typename GridT, typename InterruptT>
inline void
LevelSetFilter<GridT, InterruptT>::doMedian(const RangeType& range, int width)
{
    typedef typename LeafType::ValueOnCIter VoxelIterT;
    BaseType::checkInterrupter();
    typename math::DenseStencil<GridType> stencil(BaseType::mGrid, width);//creates local cache!
    for (size_t n=range.begin(), e=range.end(); n != e; ++n) {
        typename BaseType::BufferType& buffer = BaseType::mLeafs->getBuffer(n,1);
        for (VoxelIterT iter=BaseType::mLeafs->leaf(n).cbeginValueOn(); iter; ++iter) {
            stencil.moveTo(iter);
            buffer.setValue(iter.pos(), stencil.median());
        }
    }
}

} // namespace tools
} // namespace OPENVDB_VERSION_NAME
} // namespace openvdb

#endif // OPENVDB_TOOLS_LEVELSETFILTER_HAS_BEEN_INCLUDED

// Copyright (c) 2012 DreamWorks Animation LLC
// All rights reserved. This software is distributed under the
// Mozilla Public License 2.0 ( http://www.mozilla.org/MPL/2.0/ )
