///////////////////////////////////////////////////////////////////////////
//
// Copyright (c) 2012 DreamWorks Animation LLC
//
// All rights reserved. This software is distributed under the
// Mozilla Public License 2.0 ( http://www.mozilla.org/MPL/2.0/ )
//
// Redistributions of source code must retain the above copyright
// and license notice and the following restrictions and disclaimer.
//
// *     Neither the name of DreamWorks Animation nor the names of
// its contributors may be used to endorse or promote products derived
// from this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
// OWNER OR CONTRIBUTORS BE LIABLE FOR ANY INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
// IN NO EVENT SHALL THE COPYRIGHT HOLDERS' AND CONTRIBUTORS' AGGREGATE
// LIABILITY FOR ALL CLAIMS REGARDLESS OF THEIR BASIS EXCEED US$250.00.
//
///////////////////////////////////////////////////////////////////////////
//
/// @file Maps.h

#ifndef OPENVDB_MATH_MAPS_HAS_BEEN_INCLUDED
#define OPENVDB_MATH_MAPS_HAS_BEEN_INCLUDED

#include "Mat4.h"
#include "Vec3.h"
#include "BBox.h"
#include "Coord.h"
#include <openvdb/util/Name.h>
#include <openvdb/Types.h>
#include <boost/shared_ptr.hpp>
#include <tbb/mutex.h>
#include <map>

namespace openvdb {
OPENVDB_USE_VERSION_NAMESPACE
namespace OPENVDB_VERSION_NAME {
namespace math {


////////////////////////////////////////

/// Forward declarations of the different map types

class MapBase;
class ScaleMap;
class TranslationMap;
class ScaleTranslateMap;
class UniformScaleMap;
class UniformScaleTranslateMap;
class AffineMap;
class UnitaryMap;
class NonlinearFrustumMap;

template<typename T1, typename T2> class CompoundMap;

typedef CompoundMap<UnitaryMap, TranslationMap>                     UnitaryAndTranslationMap;
typedef CompoundMap<CompoundMap<UnitaryMap, ScaleMap>, UnitaryMap>  SpectralDecomposedMap;
typedef SpectralDecomposedMap                                       SymmetricMap;
typedef CompoundMap<SymmetricMap, UnitaryAndTranslationMap>         FullyDecomposedMap;
typedef CompoundMap<SymmetricMap, UnitaryMap>                       PolarDecomposedMap;


////////////////////////////////////////

/// Map traits

template<typename T> struct is_linear                 { static const bool value = false; };
template<> struct is_linear<AffineMap>                { static const bool value = true; };
template<> struct is_linear<ScaleMap>                 { static const bool value = true; };
template<> struct is_linear<UniformScaleMap>          { static const bool value = true; };
template<> struct is_linear<UnitaryMap>               { static const bool value = true; };
template<> struct is_linear<TranslationMap>           { static const bool value = true; };
template<> struct is_linear<ScaleTranslateMap>        { static const bool value = true; };
template<> struct is_linear<UniformScaleTranslateMap> { static const bool value = true; };

template<typename T1, typename T2> struct is_linear<CompoundMap<T1, T2> > {
    static const bool value = is_linear<T1>::value && is_linear<T2>::value;
};


template<typename T> struct is_uniform_scale          { static const bool value = false; };
template<> struct is_uniform_scale<UniformScaleMap>   { static const bool value = true; };

template<typename T> struct is_uniform_scale_translate       { static const bool value = false; };
template<> struct is_uniform_scale_translate<TranslationMap> { static const bool value = true; };
template<> struct is_uniform_scale_translate<UniformScaleTranslateMap> {
    static const bool value = true;
};


template<typename T> struct is_scale                  { static const bool value = false; };
template<> struct is_scale<ScaleMap>                  { static const bool value = true; };

template<typename T> struct is_scale_translate        { static const bool value = false; };
template<> struct is_scale_translate<ScaleTranslateMap> { static const bool value = true; };


template<typename T> struct is_uniform_diagonal_jacobian {
    static const bool value = is_uniform_scale<T>::value || is_uniform_scale_translate<T>::value;
};

template<typename T> struct is_diagonal_jacobian {
    static const bool value = is_scale<T>::value || is_scale_translate<T>::value;
};


////////////////////////////////////////

/// Utility methods

/// @brief Create a SymmetricMap from a symmetric matrix.
/// Decomposes the map into Rotation Diagonal Rotation^T
OPENVDB_API boost::shared_ptr<SymmetricMap> createSymmetricMap(const Mat3d& m);


/// @brief General decomposition of a Matrix into a Unitary (e.g. rotation)
/// following a Symmetric (e.g. stretch & shear)
OPENVDB_API boost::shared_ptr<FullyDecomposedMap> createFullyDecomposedMap(const Mat4d& m);


/// @brief Decomposes a general linear into translation following polar decomposition.
///
/// T U S where:
///
///  T: Translation
///  U: Unitary (rotation or reflection)
///  S: Symmetric
///
/// @note: the Symmetric is automatically decomposed into Q D Q^T, where
/// Q is rotation and D is diagonal.
OPENVDB_API boost::shared_ptr<PolarDecomposedMap> createPolarDecomposedMap(const Mat3d& m);


/// @brief reduces an AffineMap to a ScaleMap or a ScaleTranslateMap when it can
OPENVDB_API boost::shared_ptr<MapBase> simplify(boost::shared_ptr<AffineMap> affine);


////////////////////////////////////////


/// @brief Abstract base class for maps
class OPENVDB_API MapBase
{
public:
    typedef boost::shared_ptr<MapBase>       Ptr;
    typedef boost::shared_ptr<const MapBase> ConstPtr;
    typedef Ptr (*MapFactory)();

    virtual ~MapBase(){}

    virtual boost::shared_ptr<AffineMap> getAffineMap() const = 0;

    /// Return the name of this map's concrete type (e.g., @c "AffineMap").
    virtual Name type() const = 0;

    /// Return @c true if this map is of concrete type @c MapT (e.g., AffineMap).
    template<typename MapT> bool isType() const { return this->type() == MapT::mapType(); }

    /// Return @c true if this map is equal to the given map.
    virtual bool isEqual(const MapBase& other) const = 0;

    /// Return @c true if this map is linear.
    virtual bool isLinear() const = 0;
    /// Return @c true if the spacing between the image of latice is uniform in all directions
    virtual bool hasUniformScale() const = 0;

    virtual Vec3d applyMap(const Vec3d& in) const = 0;
    virtual Vec3d applyInverseMap(const Vec3d& in) const = 0;

    virtual Vec3d applyIJT(const Vec3d& in) const = 0;
    virtual Vec3d applyIJT(const Vec3d& in, const Vec3d& pos) const = 0;
    virtual Mat3d applyIJC(const Mat3d& m) const = 0;
    virtual Mat3d applyIJC(const Mat3d& m, const Vec3d& v, const Vec3d& pos) const = 0;


    virtual double determinant() const = 0;
    virtual double determinant(const Vec3d&) const = 0;

    virtual Vec3d voxelDimensions() const = 0;
    virtual Vec3d voxelDimensions(const Vec3d&) const = 0;

    virtual void read(std::istream&) = 0;
    virtual void write(std::ostream&) const = 0;

    virtual std::string str() const = 0;

    virtual MapBase::Ptr copy() const = 0;
    virtual MapBase::Ptr rotate(double radians, Axis axis = X_AXIS) const = 0;
    virtual MapBase::Ptr translate(const Vec3d&) const = 0;
    virtual MapBase::Ptr scale(const Vec3d&) const = 0;
    virtual MapBase::Ptr shear(double shear, Axis axis0, Axis axis1) const = 0;

protected:
    MapBase() {}

    template<typename MapT>
    static bool isEqualBase(const MapT& self, const MapBase& other)
    {
        return other.isType<MapT>() && (self == *static_cast<const MapT*>(&other));
    }
};


////////////////////////////////////////


/// @brief Threadsafe singleton object for accessing the map type-name dictionary.
/// Associates a map type-name with a factory function.
class OPENVDB_API MapRegistry
{
public:
    typedef std::map<Name, MapBase::MapFactory> MapDictionary;
    typedef tbb::mutex Mutex;
    typedef Mutex::scoped_lock Lock;

    static MapRegistry* instance();

    /// Create a new map of the given (registered) type name.
    static MapBase::Ptr createMap(const Name&);

    /// Return @c true if the given map type name is registered.
    static bool isRegistered(const Name&);

    /// Register a map type along with a factory function.
    static void registerMap(const Name&, MapBase::MapFactory);

    /// Remove a map type from the registry.
    static void unregisterMap(const Name&);

    /// Clear the map type registry.
    static void clear();

private:
    MapRegistry() {}
    static MapRegistry* mInstance;

    mutable Mutex mMutex;
    MapDictionary mMap;
};


////////////////////////////////////////


/// @brief A general linear transform using homogeneous coordinates to perform
/// rotation, scaling, shear and translation
class OPENVDB_API AffineMap: public MapBase
{
public:
    typedef boost::shared_ptr<AffineMap>       Ptr;
    typedef boost::shared_ptr<const AffineMap> ConstPtr;

    AffineMap():
        mMatrix(Mat4d::identity()),
        mMatrixInv(Mat4d::identity()),
        mJacobianInv(Mat3d::identity()),
        mDeterminant(1),
        mVoxelDimensions(Vec3d(1,1,1)),
        mIsDiagonal(true),
        mIsIdentity(true)
        // the default constructor for translation is zero
    {
    }

    AffineMap(const Mat3d& m)
    {
        Mat4d mat4(Mat4d::identity());
        mat4.setMat3(m);
        mMatrix = mat4;
        updateAcceleration();
    }

    AffineMap(const Mat4d& m): mMatrix(m)
    {
        if (!isAffine(m)) {
            OPENVDB_THROW(ArithmeticError,
                "Tried to initialize an affine transform from a non-affine 4x4 matrix");
        }
        updateAcceleration();
    }

    AffineMap(const AffineMap& other):
        MapBase(other),
        mMatrix(other.mMatrix),
        mMatrixInv(other.mMatrixInv),
        mJacobianInv(other.mJacobianInv),
        mDeterminant(other.mDeterminant),
        mVoxelDimensions(other.mVoxelDimensions),
        mIsDiagonal(other.mIsDiagonal),
        mIsIdentity(other.mIsIdentity)
    {
    }

    /// @brief constructor that merges the matrixes for two affine maps
    AffineMap(const AffineMap& first, const AffineMap& second):
        mMatrix(first.mMatrix * second.mMatrix)
    {
        updateAcceleration();
    }

    ~AffineMap() {}

    /// Return a MapBase::Ptr to a new AffineMap
    static MapBase::Ptr create() { return MapBase::Ptr(new AffineMap()); }
    /// Return a MapBase::Ptr to a deep copy of this map
    MapBase::Ptr copy() const { return MapBase::Ptr(new AffineMap(*this)); }

    static bool isRegistered() { return MapRegistry::isRegistered(AffineMap::mapType()); }

    static void registerMap()
    {
        MapRegistry::registerMap(
            AffineMap::mapType(),
            AffineMap::create);
    }

    Name type() const { return mapType(); }
    static Name mapType() { return Name("AffineMap"); }

    /// Return @c true (an AffineMap is always linear).
    bool isLinear() const { return true; }

    /// Return @c false ( test if this is unitary with translation )
    bool hasUniformScale() const { return isUnitary(mMatrix.getMat3());}

    virtual bool isEqual(const MapBase& other) const { return isEqualBase(*this, other); }

    bool operator==(const AffineMap& other) const
    {
        // the Mat.eq() is approximate
        if (!mMatrix.eq(other.mMatrix)) { return false; }
        if (!mMatrixInv.eq(other.mMatrixInv))  { return false; }
        return true;
    }

    bool operator!=(const AffineMap& other) const { return !(*this == other); }

    AffineMap& operator=(const AffineMap& other)
    {
        mMatrix = other.mMatrix;
        mMatrixInv = other.mMatrixInv;

        mJacobianInv = other.mJacobianInv;
        mDeterminant = other.mDeterminant;
        mVoxelDimensions = other.mVoxelDimensions;
        mIsDiagonal  = other.mIsDiagonal;
        mIsIdentity  = other.mIsIdentity;
        return *this;
    }
    /// Return the image of @c in under the map
    Vec3d applyMap(const Vec3d& in) const { return in * mMatrix; }
    /// Return the pre-image of @c in under the map
    Vec3d applyInverseMap(const Vec3d& in) const {return in * mMatrixInv; }

    /// Return the transpose of the inverse Jacobian of the map applied to @a in.
    Vec3d applyIJT(const Vec3d& in, const Vec3d&) const { return applyIJT(in); }
    /// Return the transpose of the inverse Jacobian of the map applied to @c in
    Vec3d applyIJT(const Vec3d& in) const { return in * mJacobianInv; }
    /// Return the Jacobian Curvature: zero for a linear map
    Mat3d applyIJC(const Mat3d& m) const {
        return mJacobianInv.transpose()* m * mJacobianInv;
    }
    Mat3d applyIJC(const Mat3d& in, const Vec3d& , const Vec3d& ) const {
        return applyIJC(in);
    }
    /// Return the determinant of the Jacobian, ignores argument
    double determinant(const Vec3d& ) const { return determinant(); }
    /// Return the determinant of the Jacobian
    double determinant() const { return mDeterminant; }

    /// @brief Return the lengths of the images of the segments
    /// (0,0,0)-(1,0,0), (0,0,0)-(0,1,0) and (0,0,0)-(0,0,1).
    Vec3d voxelDimensions() const { return mVoxelDimensions; }
    Vec3d voxelDimensions(const Vec3d&) const { return voxelDimensions(); }

    /// Return @c true if the underlying matrix is approximately an identity
    bool isIdentity() const { return mIsIdentity; }
    /// Return @c true  if the underylying matrix is diagonal
    bool isDiagonal() const { return mIsDiagonal; }
    /// Return @c true if the map is equivalent to a ScaleMap
    bool isScale() const { return isDiagonal(); }
    /// Return @c true if the map is equivalent to a ScaleTranslateMap
    bool isScaleTranslate() const { return math::isDiagonal(mMatrix.getMat3()); }


    // Methods that modify the existing affine map
    void accumRotation(Axis axis, double radians)
    {
        mMatrix.accumRotation(axis, radians);
        updateAcceleration();
    }

    /// Accumulates scale in the current AffineMap by pre-multiplying
    void accumScale(const Vec3d& v)
    {
        // this pre-multiplies by a diagonal matrix with the elements of v
        // on the diagonal:  D * M
        mMatrix.accumScale(v);
        updateAcceleration();
    }

    /// Accumulate image-space translation in the current AffineMap
    void accumTranslation(const Vec3d& v)
    {
        // this: 'mMatrix.accumTranslation(v)' would accumulate translation in index space.
        Vec3d old_translation = mMatrix.getTranslation();
        mMatrix.setTranslation(v + old_translation);
        updateAcceleration();
    }
    /// Accumulate shear in the current AffineMap
    void accumShear(Axis axis0, Axis axis1, double shear)
    {
        mMatrix.accumShear(axis0, axis1, shear);
        updateAcceleration();
    }

    /// read serialization
    void read(std::istream& is)
    {
        mMatrix.read(is);
        updateAcceleration();
    }

    /// write serialization
    void write(std::ostream& os) const
    {
        mMatrix.write(os);
    }

    /// string serialization, useful for debugging
    std::string str() const
    {
        std::ostringstream buffer;
        buffer << " - mat4:\n" << mMatrix.str() << std::endl;
        buffer << " - voxel dimensions: " << mVoxelDimensions << std::endl;
        return buffer.str();
    }

    /// on-demand decomposition of the affine map
    boost::shared_ptr<FullyDecomposedMap> createDecomposedMap()
    {
        return createFullyDecomposedMap(mMatrix);
    }

    /// Return AffineMap::Ptr to  a deep copy of the current AffineMap
    AffineMap::Ptr getAffineMap() const { return AffineMap::Ptr(new AffineMap(*this)); }

    /// Return AffineMap::Ptr to the inverse of this map
    AffineMap::Ptr inverse() const { return AffineMap::Ptr(new AffineMap(mMatrixInv)); }


    /// Return a MapBase::Ptr to a new map that is the result of accumulating rotation on this map
    MapBase::Ptr rotate(double radians, Axis axis = X_AXIS) const
    {
        AffineMap::Ptr affineMap = getAffineMap();
        affineMap->accumRotation(axis, radians);
        return simplify(affineMap);
    }

    /// @brief Return a MapBase::Ptr to a new map that is the result of accumulating
    /// translation on this map
    MapBase::Ptr translate(const Vec3d& t) const
    {
        AffineMap::Ptr affineMap = getAffineMap();
        affineMap->accumTranslation(t);
        return boost::static_pointer_cast<MapBase, AffineMap>(affineMap);
    }

    /// Return a MapBase::Ptr to a new Map that is the result of accumulating scale on this map
    MapBase::Ptr scale(const Vec3d& s) const
    {
        AffineMap::Ptr affineMap = getAffineMap();
        affineMap->accumScale(s);
        return boost::static_pointer_cast<MapBase, AffineMap>(affineMap);
    }

    /// Return a MapBase::Ptr to a new Map that is the result of accumulating shear on this map
    MapBase::Ptr shear(double shear, Axis axis0, Axis axis1) const
    {
        AffineMap::Ptr affineMap = getAffineMap();
        affineMap->accumShear(axis0, axis1, shear);
        return simplify(affineMap);
    }

    /// Return the matrix representation of this AffineMap
    Mat4d getMat4() const { return mMatrix; }

private:
    void updateAcceleration() {
        mDeterminant = mMatrix.getMat3().det();

        if (std::abs(mDeterminant) < (3.0 * tolerance<double>::value())) {
            OPENVDB_THROW(ArithmeticError,
                "Tried to initialize an affine transform from a nearly signular matrix");
        }
        mMatrixInv = mMatrix.inverse();
        mJacobianInv = mMatrixInv.getMat3().transpose();
        mIsDiagonal = math::isDiagonal(mMatrix);
        mIsIdentity = math::isIdentity(mMatrix);
        Vec3d pos = applyMap(Vec3d(0,0,0));
        mVoxelDimensions(0) = (applyMap(Vec3d(1,0,0)) - pos).length();
        mVoxelDimensions(1) = (applyMap(Vec3d(0,1,0)) - pos).length();
        mVoxelDimensions(2) = (applyMap(Vec3d(0,0,1)) - pos).length();
    }

    // the underlying matrix
    Mat4d  mMatrix;

    // stored for acceleration
    Mat4d  mMatrixInv;
    Mat3d  mJacobianInv;
    double mDeterminant;
    Vec3d  mVoxelDimensions;
    bool mIsDiagonal, mIsIdentity;
}; // class AffineMap


////////////////////////////////////////


/// @brief A specialized Affine transform that scales along the principal axis
/// the scaling need not be uniform in the three-directions
class OPENVDB_API ScaleMap: public MapBase
{
public:
    typedef boost::shared_ptr<ScaleMap>       Ptr;
    typedef boost::shared_ptr<const ScaleMap> ConstPtr;

    ScaleMap(): MapBase(), mScaleValues(Vec3d(1,1,1)), mVoxelDimensions(Vec3d(1,1,1)),
                mInvScaleSqr(1,1,1), mInvTwiceScale(0.5,0.5,0.5){}

    ScaleMap(const Vec3d& scale):
        MapBase(),
        mScaleValues(scale),
        mVoxelDimensions(Vec3d(std::abs(scale(0)),std::abs(scale(1)), std::abs(scale(2))))
    {
        if (scale.eq(Vec3d(0.0), 1.0e-10)) {
            OPENVDB_THROW(ArithmeticError, "Non-zero scale values required");
        }
        mScaleValuesInverse = 1.0 / mScaleValues;
        mInvScaleSqr = mScaleValuesInverse * mScaleValuesInverse;
        mInvTwiceScale = mScaleValuesInverse / 2;
    }

    ScaleMap(const ScaleMap& other):
        MapBase(),
        mScaleValues(other.mScaleValues),
        mVoxelDimensions(other.mVoxelDimensions),
        mScaleValuesInverse(other.mScaleValuesInverse),
        mInvScaleSqr(other.mInvScaleSqr),
        mInvTwiceScale(other.mInvTwiceScale)
    {
    }

    ~ScaleMap() {}

    /// Return a MapBase::Ptr to a new ScaleMap
    static MapBase::Ptr create() { return MapBase::Ptr(new ScaleMap()); }
    /// Return a MapBase::Ptr to a deep copy of this map
    MapBase::Ptr copy() const { return MapBase::Ptr(new ScaleMap(*this)); }

    static bool isRegistered() { return MapRegistry::isRegistered(ScaleMap::mapType()); }

    static void registerMap()
    {
        MapRegistry::registerMap(
            ScaleMap::mapType(),
            ScaleMap::create);
    }

    Name type() const { return mapType(); }
    static Name mapType() { return Name("ScaleMap"); }

    /// Return @c true (a ScaleMap is always linear).
    bool isLinear() const { return true; }

    /// Return @c true if the values have the same magitude (eg. -1, 1, -1 would be a rotation).
    bool hasUniformScale() const {
        bool value;
        value =          isApproxEqual(std::abs(mScaleValues.x()), std::abs(mScaleValues.y()));
        value = value && isApproxEqual(std::abs(mScaleValues.x()), std::abs(mScaleValues.z()));
        return value; }

    /// Return the image of @c in under the map
    Vec3d applyMap(const Vec3d& in) const
    {
        return Vec3d(
            in.x() * mScaleValues.x(),
            in.y() * mScaleValues.y(),
            in.z() * mScaleValues.z());
    }
    /// Return the pre-image of @c in under the map
    Vec3d applyInverseMap(const Vec3d& in) const
    {
        return Vec3d(
            in.x() * mScaleValuesInverse.x(),
            in.y() * mScaleValuesInverse.y(),
            in.z() * mScaleValuesInverse.z());
    }
    /// @brief Return the transpose of the inverse Jacobian of the map applied to @a in.
    /// @details Ignores second argument
    Vec3d applyIJT(const Vec3d& in, const Vec3d& ) const { return applyIJT(in);}
    /// Return the transpose of the inverse Jacobian of the map applied to @c in
    Vec3d applyIJT(const Vec3d& in) const { return applyInverseMap(in); }
    /// Return the Jacobian Curvature: zero for a linear map
    Mat3d applyIJC(const Mat3d& in) const {
        Mat3d tmp;
        for (int i=0; i<3; i++){
            tmp.setRow(i, in.row(i)*mScaleValuesInverse(i));
        }
        for (int i=0; i<3; i++){
            tmp.setCol(i, tmp.col(i)*mScaleValuesInverse(i));
        }
        return tmp;
    }
    Mat3d applyIJC(const Mat3d& in, const Vec3d&, const Vec3d& ) const {
        return applyIJC(in);
    }
    /// Return the product of the scale values, ignores argument
    double determinant(const Vec3d& ) const { return determinant(); }
    /// Return the product of the scale values
    double determinant() const { return mScaleValues.x()*mScaleValues.y()*mScaleValues.z(); }

    /// Return the scale values that define the map
    const Vec3d& getScale() const {return mScaleValues;}

    /// Return the square of the scale.  Used to optimize some finite difference calculations
    const Vec3d& getInvScaleSqr() const {return mInvScaleSqr;}
    /// Return 1/(2 scale). Used to optimize some finite difference calculations
    const Vec3d& getInvTwiceScale() const {return mInvTwiceScale;}
    /// Return 1/(scale)
    const Vec3d& getInvScale() const {return mScaleValuesInverse; }


    /// @brief Returns the lengths of the images
    /// of the segments
    /// \f$(0,0,0)-(1,0,0)\f$, \f$(0,0,0)-(0,1,0)\f$, \f$(0,0,0)-(0,0,1)\f$
    /// this is equivalent to the absolute values of the scale values
    Vec3d voxelDimensions() const { return mVoxelDimensions; }
    Vec3d voxelDimensions(const Vec3d&) const { return voxelDimensions();}

    /// read serialization
    void read(std::istream& is)
    {
        mScaleValues.read(is);
        mVoxelDimensions.read(is);
        mScaleValuesInverse.read(is);
        mInvScaleSqr.read(is);
        mInvTwiceScale.read(is);
    }
    /// write serialization
    void write(std::ostream& os) const
    {
        mScaleValues.write(os);
        mVoxelDimensions.write(os);
        mScaleValuesInverse.write(os);
        mInvScaleSqr.write(os);
        mInvTwiceScale.write(os);
    }
    /// string serialization, useful for debuging
    std::string str() const
    {
        std::ostringstream buffer;
        buffer << " - scale: " << mScaleValues << std::endl;
        buffer << " - voxel dimensions: " << mVoxelDimensions << std::endl;
        return buffer.str();
    }

    virtual bool isEqual(const MapBase& other) const { return isEqualBase(*this, other); }

    bool operator==(const ScaleMap& other) const
    {
        // ::eq() uses a tolerance
        if (!mScaleValues.eq(other.mScaleValues)) { return false; }
        return true;
    }

    bool operator!=(const ScaleMap& other) const { return !(*this == other); }

    /// Return a AffineMap equivalent to this map
    AffineMap::Ptr getAffineMap() const
    {
        return AffineMap::Ptr(new AffineMap(math::scale<Mat4d>(mScaleValues)));
    }
    /// Return a MapBase::Ptr to a new map that is the result of accumulating rotation on this map
    MapBase::Ptr rotate(double radians, Axis axis) const
    {
        AffineMap::Ptr affineMap = getAffineMap();
        affineMap->accumRotation(axis, radians);
        return simplify(affineMap);
    }
    /// @brief Return a MapBase::Ptr to a ScaleTraslateMap that is the result of
    /// accumulating translation on this map
    MapBase::Ptr translate(const Vec3d& tr) const;
    /// @brief Return a MapBase::Ptr to a ScaleMap that is the result of accumulating
    /// scale on this map
    MapBase::Ptr scale(const Vec3d& v) const;
    /// @brief Return a MapBase::Ptr to a new map that is the result of accumulating
    /// translation on this map
    MapBase::Ptr shear(double shear, Axis axis0, Axis axis1) const
    {
        AffineMap::Ptr affineMap = getAffineMap();
        affineMap->accumShear(axis0, axis1, shear);
        return simplify(affineMap);
    }

private:
    Vec3d mScaleValues, mVoxelDimensions, mScaleValuesInverse, mInvScaleSqr, mInvTwiceScale;
}; // class ScaleMap


/// @brief A specialized Affine transform that scales along the principal axis
/// the scaling is uniform in the three-directions
class OPENVDB_API UniformScaleMap: public ScaleMap
{
public:
    typedef boost::shared_ptr<UniformScaleMap>       Ptr;
    typedef boost::shared_ptr<const UniformScaleMap> ConstPtr;

    UniformScaleMap(): ScaleMap(Vec3d(1,1,1)) {}
    UniformScaleMap(double scale): ScaleMap(Vec3d(scale, scale, scale)) {}
    UniformScaleMap(const UniformScaleMap& other): ScaleMap(other) {}
    ~UniformScaleMap() {}

    /// Return a MapBase::Ptr to a new UniformScaleMap
    static MapBase::Ptr create() { return MapBase::Ptr(new UniformScaleMap()); }
    /// Return a MapBase::Ptr to a deep copy of this map
    MapBase::Ptr copy() const { return MapBase::Ptr(new UniformScaleMap(*this)); }

    static bool isRegistered() { return MapRegistry::isRegistered(UniformScaleMap::mapType()); }
    static void registerMap()
    {
        MapRegistry::registerMap(
            UniformScaleMap::mapType(),
            UniformScaleMap::create);
    }

    Name type() const { return mapType(); }
    static Name mapType() { return Name("UniformScaleMap"); }

    virtual bool isEqual(const MapBase& other) const { return isEqualBase(*this, other); }

    bool operator==(const UniformScaleMap& other) const { return ScaleMap::operator==(other); }
    bool operator!=(const UniformScaleMap& other) const { return !(*this == other); }

    /// Return a MapBase::Ptr to a UniformScaleTraslateMap that is the result of
    /// accumulating translation on this map
    MapBase::Ptr translate(const Vec3d& tr) const;
}; // class UniformScaleMap

////////////////////////////////////////


inline MapBase::Ptr
ScaleMap::scale(const Vec3d& v) const
{
    Vec3d new_scale(v * mScaleValues);
    if (isApproxEqual(new_scale[0],new_scale[1]) && isApproxEqual(new_scale[0],new_scale[2])) {
        return MapBase::Ptr(new UniformScaleMap(new_scale[0]));
    } else {
        return MapBase::Ptr(new ScaleMap(new_scale));
    }
}


/// @brief A specialized linear transform that performs a translation
class OPENVDB_API TranslationMap: public MapBase
{
public:
    typedef boost::shared_ptr<TranslationMap>       Ptr;
    typedef boost::shared_ptr<const TranslationMap> ConstPtr;

    // default constructor is a translation by zero.
    TranslationMap(): MapBase(),mTranslation(Vec3d(0,0,0)) {}

    TranslationMap(const Vec3d& t): MapBase(),mTranslation(t)
    {
    }

    TranslationMap(const TranslationMap& other): MapBase(),mTranslation(other.mTranslation)
    {
    }

    ~TranslationMap() {}

    /// Return a MapBase::Ptr to a new TranslationMap
    static MapBase::Ptr create() { return MapBase::Ptr(new TranslationMap()); }
    /// Return a MapBase::Ptr to a deep copy of this map
    MapBase::Ptr copy() const { return MapBase::Ptr(new TranslationMap(*this)); }

    static bool isRegistered() { return MapRegistry::isRegistered(TranslationMap::mapType()); }

    static void registerMap()
    {
        MapRegistry::registerMap(
            TranslationMap::mapType(),
            TranslationMap::create);
    }

    Name type() const { return mapType(); }
    static Name mapType() { return Name("TranslationMap"); }

    /// Return @c true (a TranslationMap is always linear).
    bool isLinear() const { return true; }

    /// Return @c false (by convention true)
    bool hasUniformScale() const { return true; }

    /// Return the image of @c in under the map
    Vec3d applyMap(const Vec3d& in) const { return in + mTranslation; }
    /// Return the pre-image of @c in under the map
    Vec3d applyInverseMap(const Vec3d& in) const { return in - mTranslation; }

    /// @brief Return the transpose of the inverse Jacobian (Identity for TranslationMap)
    /// of the map applied to @c in, ignores second argument
    Vec3d applyIJT(const Vec3d& in, const Vec3d& ) const { return applyIJT(in);}
    /// @brief Return the transpose of the inverse Jacobian (Identity for TranslationMap)
    /// of the map applied to @c in
    Vec3d applyIJT(const Vec3d& in) const {return in;}
    /// Return the Jacobian Curvature: zero for a linear map
    Mat3d applyIJC(const Mat3d& mat) const {return mat;}
    Mat3d applyIJC(const Mat3d& mat, const Vec3d&, const Vec3d&) const {
        return applyIJC(mat);
    }

    /// Return @c 1
    double determinant(const Vec3d& ) const { return determinant(); }
    /// Return @c 1
    double determinant() const { return 1.0; }

    /// Return \f$ (1,1,1) \f$
    Vec3d voxelDimensions() const { return Vec3d(1,1,1);}
    /// Return \f$ (1,1,1) \f$
    Vec3d voxelDimensions(const Vec3d&) const { return voxelDimensions();}

    /// Return the translation vector
    const Vec3d& getTranslation() const { return mTranslation; }
    /// read serialization
    void read(std::istream& is) { mTranslation.read(is); }
    /// write serialization
    void write(std::ostream& os) const { mTranslation.write(os); }

    /// string serialization, useful for debuging
    std::string str() const
    {
        std::ostringstream buffer;
        buffer << " - translation: " << mTranslation << std::endl;
        return buffer.str();
    }

    virtual bool isEqual(const MapBase& other) const { return isEqualBase(*this, other); }

    bool operator==(const TranslationMap& other) const
    {
        // ::eq() uses a tolerance
        if (!mTranslation.eq(other.mTranslation)) { return false; }
        return true;
    }

    bool operator!=(const TranslationMap& other) const { return !(*this == other); }

    /// Return AffineMap::Ptr to an AffineMap equivalent to *this
    AffineMap::Ptr getAffineMap() const
    {
        Mat4d matrix(Mat4d::identity());
        matrix.setTranslation(mTranslation);

        AffineMap::Ptr affineMap(new AffineMap(matrix));
        return affineMap;
    }

    /// Return MapBase::Ptr to a new map that is the result of accumulating rotation on this map
    MapBase::Ptr rotate(double radians, Axis axis) const
    {
        AffineMap::Ptr affineMap = getAffineMap();
        affineMap->accumRotation(axis, radians);
        return simplify(affineMap);

    }
    /// @brief Return MapBase::Ptr to a new TranlationMap that is the result of
    /// accumulating translation on this map
    MapBase::Ptr translate(const Vec3d& t) const
    {
        return MapBase::Ptr(new TranslationMap(t + mTranslation));
    }
    /// @brief Return a MapBase::Ptr to a ScaleTranslateMap that is the result of
    /// accumulating scale on this map
    MapBase::Ptr scale(const Vec3d& v) const;
    /// Return a MapBase::Ptr to a new map that is the result of accumulating shear on this map
    MapBase::Ptr shear(double shear, Axis axis0, Axis axis1) const
    {
        AffineMap::Ptr affineMap = getAffineMap();
        affineMap->accumShear(axis0, axis1, shear);
        return simplify(affineMap);
    }

private:
    Vec3d mTranslation;
}; // class TranslationMap


////////////////////////////////////////


/// @brief A specialized Affine transform that scales along the principal axis
/// the scaling need not be uniform in the three-directions, and then
/// translates the result.
class OPENVDB_API ScaleTranslateMap: public MapBase
{
public:
    typedef boost::shared_ptr<ScaleTranslateMap>       Ptr;
    typedef boost::shared_ptr<const ScaleTranslateMap> ConstPtr;

    ScaleTranslateMap():
        MapBase(),
        mTranslation(Vec3d(0,0,0)),
        mScaleValues(Vec3d(1,1,1)),
        mVoxelDimensions(Vec3d(1,1,1)),
        mScaleValuesInverse(Vec3d(1,1,1)),
        mInvScaleSqr(1,1,1),
        mInvTwiceScale(0.5,0.5,0.5)
    {
    }

    ScaleTranslateMap(const Vec3d& scale, const Vec3d& translate):
        MapBase(),
        mTranslation(translate),
        mScaleValues(scale),
        mVoxelDimensions(std::abs(scale(0)), std::abs(scale(1)), std::abs(scale(2)))
    {
        if (scale.eq(Vec3d(0.0), 1.0e-10)) {
            OPENVDB_THROW(ArithmeticError, "Non-zero scale values required");
        }
        mScaleValuesInverse = 1.0 / mScaleValues;
        mInvScaleSqr = mScaleValuesInverse * mScaleValuesInverse;
        mInvTwiceScale = mScaleValuesInverse / 2;
    }

    ScaleTranslateMap(const ScaleMap& scale, const TranslationMap& translate):
        MapBase(),
        mTranslation(translate.getTranslation()),
        mScaleValues(scale.getScale()),
        mVoxelDimensions(std::abs(mScaleValues(0)),
                         std::abs(mScaleValues(1)),
                         std::abs(mScaleValues(2))),
        mScaleValuesInverse(1.0 / scale.getScale())
    {
        mInvScaleSqr = mScaleValuesInverse * mScaleValuesInverse;
        mInvTwiceScale = mScaleValuesInverse / 2;
    }

    ScaleTranslateMap(const ScaleTranslateMap& other):
        MapBase(),
        mTranslation(other.mTranslation),
        mScaleValues(other.mScaleValues),
        mVoxelDimensions(other.mVoxelDimensions),
        mScaleValuesInverse(other.mScaleValuesInverse),
        mInvScaleSqr(other.mInvScaleSqr),
        mInvTwiceScale(other.mInvTwiceScale)
    {}

    ~ScaleTranslateMap() {}

    /// Return a MapBase::Ptr to a new ScaleTranslateMap
    static MapBase::Ptr create() { return MapBase::Ptr(new ScaleTranslateMap()); }
    /// Return a MapBase::Ptr to a deep copy of this map
    MapBase::Ptr copy() const { return MapBase::Ptr(new ScaleTranslateMap(*this)); }

    static bool isRegistered() { return MapRegistry::isRegistered(ScaleTranslateMap::mapType()); }

    static void registerMap()
    {
        MapRegistry::registerMap(
            ScaleTranslateMap::mapType(),
            ScaleTranslateMap::create);
    }

    Name type() const { return mapType(); }
    static Name mapType() { return Name("ScaleTranslateMap"); }

    /// Return @c true (a ScaleTranslateMap is always linear).
    bool isLinear() const { return true; }

    /// @brief Return @c true if the scale values have the same magnitude
    /// (eg. -1, 1, -1 would be a rotation).
    bool hasUniformScale() const {
        bool value;
        value =          isApproxEqual(std::abs(mScaleValues.x()), std::abs(mScaleValues.y()));
        value = value && isApproxEqual(std::abs(mScaleValues.x()), std::abs(mScaleValues.z()));
        return value;
    }

    /// Return the image of @c under the map
    Vec3d applyMap(const Vec3d& in) const
    {
        return Vec3d(
            in.x() * mScaleValues.x() + mTranslation.x(),
            in.y() * mScaleValues.y() + mTranslation.y(),
            in.z() * mScaleValues.z() + mTranslation.z());
    }
    /// Return the pre-image of @c under the map
    Vec3d applyInverseMap(const Vec3d& in) const
    {
        return Vec3d(
            (in.x() - mTranslation.x() ) / mScaleValues.x(),
            (in.y() - mTranslation.y() ) / mScaleValues.y(),
            (in.z() - mTranslation.z() ) / mScaleValues.z());
    }
    /// @brief Return the transpose of the inverse Jacobian of the map applied to @a in
    /// @details Ignores second argument
    Vec3d applyIJT(const Vec3d& in, const Vec3d& ) const { return applyIJT(in);}
    /// Return the transpose of the inverse Jacobian of the map applied to @c in
    Vec3d applyIJT(const Vec3d& in) const
    {
        return Vec3d(
            in.x() / mScaleValues.x(),
            in.y() / mScaleValues.y(),
            in.z() / mScaleValues.z());
    }
    /// Return the Jacobian Curvature: zero for a linear map
    Mat3d applyIJC(const Mat3d& in) const {
        Mat3d tmp;
        for (int i=0; i<3; i++){
            tmp.setRow(i, in.row(i)*mScaleValuesInverse(i));
        }
        for (int i=0; i<3; i++){
            tmp.setCol(i, tmp.col(i)*mScaleValuesInverse(i));
        }
        return tmp;
    }
    Mat3d applyIJC(const Mat3d& in, const Vec3d&, const Vec3d& ) const {
        return applyIJC(in);
    }

    /// Return the product of the scale values, ignores argument
    double determinant(const Vec3d& ) const { return determinant(); }
    /// Return the product of the scale values
    double determinant() const { return mScaleValues.x()*mScaleValues.y()*mScaleValues.z(); }
    /// Return the absolute values of the scale values
    Vec3d voxelDimensions() const { return mVoxelDimensions;}
    /// Return the absolute values of the scale values, ignores argument
    Vec3d voxelDimensions(const Vec3d&) const { return voxelDimensions();}

    /// Returns the scale values
    const Vec3d& getScale() const { return mScaleValues; }
    /// Returns the translation
    const Vec3d& getTranslation() const { return mTranslation; }

    /// Return the square of the scale.  Used to optimize some finite difference calculations
    const Vec3d& getInvScaleSqr() const {return mInvScaleSqr;}
    /// Return 1/(2 scale). Used to optimize some finite difference calculations
    const Vec3d& getInvTwiceScale() const {return mInvTwiceScale;}
    /// Return 1/(scale)
    const Vec3d& getInvScale() const {return mScaleValuesInverse; }

    /// read serialization
    void read(std::istream& is)
    {
        mTranslation.read(is);
        mScaleValues.read(is);
        mVoxelDimensions.read(is);
        mScaleValuesInverse.read(is);
        mInvScaleSqr.read(is);
        mInvTwiceScale.read(is);
    }
    /// write serialization
    void write(std::ostream& os) const
    {
        mTranslation.write(os);
        mScaleValues.write(os);
        mVoxelDimensions.write(os);
        mScaleValuesInverse.write(os);
        mInvScaleSqr.write(os);
        mInvTwiceScale.write(os);
    }
    /// string serialization, useful for debuging
    std::string str() const
    {
        std::ostringstream buffer;
        buffer << " - translation: " << mTranslation << std::endl;
        buffer << " - scale: " << mScaleValues << std::endl;
        buffer << " - voxel dimensions: " << mVoxelDimensions << std::endl;
        return buffer.str();
    }

    virtual bool isEqual(const MapBase& other) const { return isEqualBase(*this, other); }

    bool operator==(const ScaleTranslateMap& other) const
    {
        // ::eq() uses a tolerance
        if (!mScaleValues.eq(other.mScaleValues)) { return false; }
        if (!mTranslation.eq(other.mTranslation)) { return false; }
        return true;
    }

    bool operator!=(const ScaleTranslateMap& other) const { return !(*this == other); }

    /// Return AffineMap::Ptr to an AffineMap equivalent to *this
    AffineMap::Ptr getAffineMap() const
    {
        AffineMap::Ptr affineMap(new AffineMap(math::scale<Mat4d>(mScaleValues)));
        affineMap->accumTranslation(mTranslation);
        return affineMap;
    }

    /// Return a MapBase::Ptr to a new map that is the result of accumulating rotation on this map
    MapBase::Ptr rotate(double radians, Axis axis) const
    {
        AffineMap::Ptr affineMap = getAffineMap();
        affineMap->accumRotation(axis, radians);
        return simplify(affineMap);
    }
    /// @brief Return a MapBase::Ptr to a ScaleTranslateMap that is the result of
    /// accumulating translation on this map
    MapBase::Ptr translate(const Vec3d& t) const
    {
        return MapBase::Ptr( new ScaleTranslateMap(mScaleValues, mTranslation + t));
    }
    /// Return a MapBase::Ptr to a ScaleTranslateMap or UniformScaleTranslateMap
    ///that is the result of accumulating translation on this map
    MapBase::Ptr scale(const Vec3d& v) const;

    /// Return a MapBase::Ptr to a new map that is the result of accumulating shear on this map
    MapBase::Ptr shear(double shear, Axis axis0, Axis axis1) const
    {
        AffineMap::Ptr affineMap = getAffineMap();
        affineMap->accumShear(axis0, axis1, shear);
        return simplify(affineMap);
    }

private:
    Vec3d mTranslation, mScaleValues, mVoxelDimensions, mScaleValuesInverse,
        mInvScaleSqr, mInvTwiceScale;
}; // class ScaleTanslateMap


inline MapBase::Ptr
ScaleMap::translate(const Vec3d& t) const
{
    return MapBase::Ptr(new ScaleTranslateMap(mScaleValues, t));
}

/// @brief A specialized Affine transform that uniformaly scales along the principal axis
/// and then translates the result.
class OPENVDB_API UniformScaleTranslateMap: public ScaleTranslateMap
{
public:
    typedef boost::shared_ptr<UniformScaleTranslateMap>       Ptr;
    typedef boost::shared_ptr<const UniformScaleTranslateMap> ConstPtr;

    UniformScaleTranslateMap():ScaleTranslateMap(Vec3d(1,1,1), Vec3d(0,0,0)) {}
    UniformScaleTranslateMap(double scale, const Vec3d& translate):
        ScaleTranslateMap(Vec3d(scale,scale,scale), translate) {}
    UniformScaleTranslateMap(const UniformScaleMap& scale, const TranslationMap& translate):
        ScaleTranslateMap(scale.getScale(), translate.getTranslation()) {}

    UniformScaleTranslateMap(const UniformScaleTranslateMap& other):ScaleTranslateMap(other) {}
    ~UniformScaleTranslateMap() {}

    /// Return a MapBase::Ptr to a new UniformScaleTranslateMap
    static MapBase::Ptr create() { return MapBase::Ptr(new UniformScaleTranslateMap()); }
    /// Return a MapBase::Ptr to a deep copy of this map
    MapBase::Ptr copy() const { return MapBase::Ptr(new UniformScaleTranslateMap(*this)); }

    static bool isRegistered()
    {
        return MapRegistry::isRegistered(UniformScaleTranslateMap::mapType());
    }

    static void registerMap()
    {
        MapRegistry::registerMap(
            UniformScaleTranslateMap::mapType(),
            UniformScaleTranslateMap::create);
    }

    Name type() const { return mapType(); }
    static Name mapType() { return Name("UniformScaleTranslateMap"); }

    virtual bool isEqual(const MapBase& other) const { return isEqualBase(*this, other); }

    bool operator==(const UniformScaleTranslateMap& other) const
    {
        return ScaleTranslateMap::operator==(other);
    }
    bool operator!=(const UniformScaleTranslateMap& other) const { return !(*this == other); }

    /// @brief Return a MapBase::Ptr to a UniformScaleTranslateMap that is
    /// the result of accumulating translation on this map
    MapBase::Ptr translate(const Vec3d& t) const
    {
        double scale = this->getScale().x();
        return MapBase::Ptr( new UniformScaleTranslateMap(scale, this->getTranslation() + t));
    }

}; // class UniformScaleTanslateMap


inline MapBase::Ptr
UniformScaleMap::translate(const Vec3d& t) const
{
    double scale = this->getScale().x();
    return MapBase::Ptr(new UniformScaleTranslateMap(scale, t));
}

inline MapBase::Ptr
TranslationMap::scale(const Vec3d& v) const
{
    if (isApproxEqual(v[0],v[1]) && isApproxEqual(v[0],v[2])) {
        return MapBase::Ptr(new UniformScaleTranslateMap(v[0], mTranslation));
    } else {
        return MapBase::Ptr(new ScaleTranslateMap(v, mTranslation));
    }
}

inline MapBase::Ptr
ScaleTranslateMap::scale(const Vec3d& v) const
{
    Vec3d new_scale( v * mScaleValues );
    if (isApproxEqual(new_scale[0],new_scale[1]) && isApproxEqual(new_scale[0],new_scale[2])) {
        return MapBase::Ptr( new UniformScaleTranslateMap(new_scale[0], mTranslation));
    } else {
        return MapBase::Ptr( new ScaleTranslateMap(new_scale, mTranslation));
    }
}


////////////////////////////////////////

/// @brief A specialized linear transform that performs a unitary maping
/// i.e. rotation  and or reflection.
class OPENVDB_API UnitaryMap: public MapBase
{
public:
    typedef boost::shared_ptr<UnitaryMap>       Ptr;
    typedef boost::shared_ptr<const UnitaryMap> ConstPtr;

    /// default constructor makes an Idenity.
    UnitaryMap(): mAffineMap(Mat4d::identity())
    {
    }

    UnitaryMap(const Vec3d& axis, double radians)
    {
        Mat3d matrix;
        matrix.setToRotation(axis, radians);
        mAffineMap = AffineMap(matrix);
    }

    UnitaryMap(Axis axis, double radians)
    {
        Mat4d matrix;
        matrix.setToRotation(axis, radians);
        mAffineMap = AffineMap(matrix);
    }

    UnitaryMap(const Mat3d& m)
    {
        // test that the mat3 is a rotation || reflection
        if (!isUnitary(m)) {
            OPENVDB_THROW(ArithmeticError, "Matrix initializing unitary map was not unitary");
        }

        Mat4d matrix(Mat4d::identity());
        matrix.setMat3(m);
        mAffineMap = AffineMap(matrix);
    }

    UnitaryMap(const Mat4d& m)
    {
        if (!isInvertible(m)) {
            OPENVDB_THROW(ArithmeticError,
                "4x4 Matrix initializing unitary map was not unitary: not invertible");
        }

        if (!isAffine(m)) {
            OPENVDB_THROW(ArithmeticError,
                "4x4 Matrix initializing unitary map was not unitary: not affine");
        }

        if (hasTranslation(m)) {
            OPENVDB_THROW(ArithmeticError,
                "4x4 Matrix initializing unitary map was not unitary: had translation");
        }

        if (!isUnitary(m.getMat3())) {
            OPENVDB_THROW(ArithmeticError,
                "4x4 Matrix initializing unitary map was not unitary");
        }

        mAffineMap = AffineMap(m);
    }

    UnitaryMap(const UnitaryMap& other):
        MapBase(other),
        mAffineMap(other.mAffineMap)
    {
    }

    UnitaryMap(const UnitaryMap& first, const UnitaryMap& second):
        mAffineMap(*(first.getAffineMap()), *(second.getAffineMap()))
    {
    }

    ~UnitaryMap() {}
    /// Return a MapBase::Ptr to a new UnitaryMap
    static MapBase::Ptr create() { return MapBase::Ptr(new UnitaryMap()); }
    /// Returns a MapBase::Ptr to a deep copy of *this
    MapBase::Ptr copy() const { return MapBase::Ptr(new UnitaryMap(*this)); }

    static bool isRegistered() { return MapRegistry::isRegistered(UnitaryMap::mapType()); }

    static void registerMap()
    {
        MapRegistry::registerMap(
            UnitaryMap::mapType(),
            UnitaryMap::create);
    }

    /// Return @c UnitaryMap
    Name type() const { return mapType(); }
    /// Return @c UnitaryMap
    static Name mapType() { return Name("UnitaryMap"); }

    /// Return @c true (a UnitaryMap is always linear).
    bool isLinear() const { return true; }

    /// Return @c false (by convention true)
    bool hasUniformScale() const { return true; }

    virtual bool isEqual(const MapBase& other) const { return isEqualBase(*this, other); }

    bool operator==(const UnitaryMap& other) const
    {
        // compare underlying linear map.
        if (mAffineMap!=other.mAffineMap)  return false;
        return true;
    }

    bool operator!=(const UnitaryMap& other) const { return !(*this == other); }
    /// Return the image of @c in under the map
    Vec3d applyMap(const Vec3d& in) const { return mAffineMap.applyMap(in); }
    /// Return the pre-image of @c in under the map
    Vec3d applyInverseMap(const Vec3d& in) const { return mAffineMap.applyInverseMap(in); }
    /// @brief Return the transpose of the inverse Jacobian of the map applied to @a in
    /// @details Ignores second argument
    Vec3d applyIJT(const Vec3d& in, const Vec3d& ) const { return applyIJT(in);}
    /// Return the transpose of the inverse Jacobian of the map applied to @c in
    Vec3d applyIJT(const Vec3d& in) const { return mAffineMap.applyIJT(in); }
       /// Return the Jacobian Curvature: zero for a linear map
    Mat3d applyIJC(const Mat3d& in) const {
        return mAffineMap.applyIJC(in);
    }
    Mat3d applyIJC(const Mat3d& in, const Vec3d&, const Vec3d& ) const {
        return applyIJC(in);
    }
    /// Return the determinant of the Jacobian, ignores argument
    double determinant(const Vec3d& ) const { return determinant(); }
    /// Return the determinant of the Jacobian
    double determinant() const { return mAffineMap.determinant(); }


    /// @brief Returns the lengths of the images
    /// of the segments
    /// \f$(0,0,0)-(1,0,0)\f$, \f$(0,0,0)-(0,1,0)\f$, \f$(0,0,0)-(0,0,1)\f$
    Vec3d voxelDimensions() const { return mAffineMap.voxelDimensions();}
    Vec3d voxelDimensions(const Vec3d&) const { return voxelDimensions();}

    /// read serialization
    void read(std::istream& is)
    {
        mAffineMap.read(is);
    }

    /// write serialization
    void write(std::ostream& os) const
    {
        mAffineMap.write(os);
    }
    /// string serialization, useful for debuging
    std::string str() const
    {
        std::ostringstream buffer;
        buffer << mAffineMap.str();
        return buffer.str();
    }
    /// Return AffineMap::Ptr to an AffineMap equivalent to *this
    AffineMap::Ptr getAffineMap() const { return AffineMap::Ptr(new AffineMap(mAffineMap)); }

    /// Return MapBase::Ptr to a UnitaryMap that is the result of accumulating rotation on this map
    MapBase::Ptr rotate(double radians, Axis axis) const
    {
        UnitaryMap second(axis, radians);
        UnitaryMap::Ptr unitaryMap(new UnitaryMap(*this, second));
        return boost::static_pointer_cast<MapBase, UnitaryMap>(unitaryMap);
    }
    /// @brief Return a MapBase::Ptr to a new map that is the result of accumulating
    /// translation on this map
    MapBase::Ptr translate(const Vec3d& t) const
    {
        AffineMap::Ptr affineMap = getAffineMap();
        affineMap->accumTranslation(t);
        return simplify(affineMap);
    }
    /// Return a MapBase::Ptr to a new map that is the result of accumulating scale on this map
    MapBase::Ptr scale(const Vec3d& v) const
    {
        AffineMap::Ptr affineMap = getAffineMap();
        affineMap->accumScale(v);
        return simplify(affineMap);
    }
    /// Return a MapBase::Ptr to a new map that is the result of accumulating shear on this map
    MapBase::Ptr shear(double shear, Axis axis0, Axis axis1) const
    {
        AffineMap::Ptr affineMap = getAffineMap();
        affineMap->accumShear(axis0, axis1, shear);
        return simplify(affineMap);
    }

private:
    AffineMap  mAffineMap;
}; // class UnitaryMap


////////////////////////////////////////


/// @brief  This map is composed of three steps.
/// Frist it will take a box of size (Lx X  Ly X Lz) defined by an member data bounding box
/// and map it into a frustum with near plane (1 X Ly/Lx) and precribed depth
/// Then this frustum is transformed by an internal second map: most often a uniform scale,
/// but other affects can be achieved by accumulating translation, shear and rotation: these
/// are all applied to the second map
class OPENVDB_API NonlinearFrustumMap: public MapBase
{
public:
    typedef boost::shared_ptr<NonlinearFrustumMap>       Ptr;
    typedef boost::shared_ptr<const NonlinearFrustumMap> ConstPtr;

    NonlinearFrustumMap():
        MapBase(),
        mBBox(Coord(0), Coord(0)),
        mTaper(1),
        mDepth(1),
        mSecondMap(new ScaleMap())
    {
        init();
    }

    /// @brief Constructor that takes an index-space bounding box
    /// to be mapped into a frustum with a given @a depth and @a taper
    /// (defined as ratio of nearplane/farplane).
    NonlinearFrustumMap(const CoordBBox& bb, double taper, double depth):
        MapBase(),mBBox(bb), mTaper(taper), mDepth(depth), mSecondMap(new ScaleMap())
    {
        init();
    }


    /// @brief Constructor that takes an index-space bounding box
    /// to be mapped into a frustum with a given @a depth and @a taper
    /// (defined as ratio of nearplane/farplane).
    /// @details This frustum is further modifed by the @a secondMap,
    /// intended to be a simple translation and rotation and uniform scale
    NonlinearFrustumMap(const CoordBBox& bb, double taper, double depth,
        const MapBase::Ptr& secondMap):
        mBBox(bb), mTaper(taper), mDepth(depth), mSecondMap(secondMap)
    {
        if (!mSecondMap->isLinear() ) {
              OPENVDB_THROW(ArithmeticError,
                "The second map in the Frustum transfrom must be linear");
        }
        init();
    }

    NonlinearFrustumMap(const NonlinearFrustumMap& other):
        MapBase(),
        mBBox(other.mBBox),
        mTaper(other.mTaper),
        mDepth(other.mDepth),
        mSecondMap(other.mSecondMap),  /// @todo reimplement using deep-copy
        mHasSimpleAffine(other.mHasSimpleAffine)
    {
        init();
    }


    /// @brief Constructor from a camera frustum
    ///
    /// @param position the tip of the frustum (i.e., the camera's position).
    /// @param direction a vector pointing from @a position toward the near plane.
    /// @param up a non-unit vector describing the direction and extent of
    ///     the frustum's intersection on the near plane.  Together,
    ///     @a up must be orthogonal to @a direction.
    /// @param aspect the aspect ratio of the frustum intersection with near plane
    ///     defined as width / height
    /// @param z_near,depth the distance from @a position along @a direction to the
    ///     near and far planes of the frustum.
    /// @param x_count the number of voxels, aligned with @a left,
    ///     across the face of the frustum
    /// @param z_count the number of voxels, aligned with @a direction,
    ///     between the near and far planes
    NonlinearFrustumMap(const Vec3d& position,
                        const Vec3d& direction,
                        const Vec3d& up,
                        double aspect /* width / height */,
                        double z_near, double depth,
                        Coord::ValueType x_count, Coord::ValueType z_count) {

        /// @todo check that depth > 0
        /// @todo check up.length > 0
        /// @todo check that direction dot up = 0
        if (!(depth > 0)) {
            OPENVDB_THROW(ArithmeticError,
                "The frustum depth must be non-zero and positive");
        }
        if (!(up.length() > 0)) {
            OPENVDB_THROW(ArithmeticError,
                "The frustum height must be non-zero and positive");
        }
        if (!(aspect > 0)) {
            OPENVDB_THROW(ArithmeticError,
                "The frustum aspect ratio  must be non-zero and positive");
        }
        if (!(isApproxEqual(up.dot(direction), 0.))) {
            OPENVDB_THROW(ArithmeticError,
                "The frustum up orientation must be perpendicular to into-frustum direction");
        }

        double near_plane_height = 2 * up.length();
        double near_plane_width = aspect * near_plane_height;

        Coord::ValueType y_count = static_cast<int>(Round(x_count / aspect));

        mBBox = CoordBBox(Coord(0,0,0), Coord(x_count, y_count, z_count));
        mDepth = depth / near_plane_width;  // depth non-dimensionalized on width
        double gamma = near_plane_width / z_near;
        mTaper = 1./(mDepth*gamma + 1.);

        Vec3d direction_unit = direction;
        direction_unit.normalize();

        Mat4d r1(Mat4d::identity());
        r1.setToRotation(/*from*/Vec3d(0,0,1), /*to */direction_unit);
        Mat4d r2(Mat4d::identity());
        Vec3d temp = r1.inverse().transform(up);
        r2.setToRotation(/*from*/Vec3d(0,1,0), /*to*/temp );
        Mat4d scale = math::scale<Mat4d>(
            Vec3d(near_plane_width, near_plane_width, near_plane_width));

        // move the near plane to origin, rotate to align with axis, and scale down
        // T_inv * R1_inv * R2_inv * scale_inv
        Mat4d mat = scale * r2 * r1;
        mat.setTranslation(position + z_near*direction_unit);
        AffineMap map(mat);
        mSecondMap = simplify(map.getAffineMap());

        init();
    }

    ~NonlinearFrustumMap(){}
    /// Return a MapBase::Ptr to a new NonlinearFrustumMap
    static MapBase::Ptr create() { return MapBase::Ptr(new NonlinearFrustumMap()); }
    /// Return a MapBase::Ptr to a deep copy of this map
    MapBase::Ptr copy() const { return MapBase::Ptr(new NonlinearFrustumMap(*this)); }

    static bool isRegistered() { return MapRegistry::isRegistered(NonlinearFrustumMap::mapType()); }

    static void registerMap()
    {
        MapRegistry::registerMap(
            NonlinearFrustumMap::mapType(),
            NonlinearFrustumMap::create);
    }
    /// Return @c NonlinearFrustumMap
    Name type() const { return mapType(); }
    /// Return @c NonlinearFrustumMap
    static Name mapType() { return Name("NonlinearFrustumMap"); }

    /// Return @c false (a NonlinearFrustumMap is never linear).
    bool isLinear() const { return false; }

    /// Return @c false (by convention false)
    bool hasUniformScale() const { return false; }

    virtual bool isEqual(const MapBase& other) const { return isEqualBase(*this, other); }

    bool operator==(const NonlinearFrustumMap& other) const
    {
        if (mBBox!=other.mBBox) return false;
        if (!isApproxEqual(mTaper, other.mTaper)) return false;
        if (!isApproxEqual(mDepth, other.mDepth)) return false;
        if (!mSecondMap->isLinear() || !other.mSecondMap->isLinear()) return false;

        // Two linear transforms are equivalent iff they have the same translation
        // and have the same affects on orthongal spanning basis check translation
        Vec3d e(0,0,0);
        if (!mSecondMap->applyMap(e).eq(other.mSecondMap->applyMap(e))) return false;
        /// check spanning vectors
        e(0) = 1;
        if (!mSecondMap->applyMap(e).eq(other.mSecondMap->applyMap(e))) return false;
        e(0) = 0;
        e(1) = 1;
        if (!mSecondMap->applyMap(e).eq(other.mSecondMap->applyMap(e))) return false;
        e(1) = 0;
        e(2) = 1;
        if (!mSecondMap->applyMap(e).eq(other.mSecondMap->applyMap(e))) return false;
        return true;
    }

    bool operator!=(const NonlinearFrustumMap& other) const { return !(*this == other); }

    /// Return the image of @c in under the map
    Vec3d applyMap(const Vec3d& in) const
    {
        return mSecondMap->applyMap(applyFrustumMap(in));
    }

    /// Return the pre-image of @c in under the map
    Vec3d applyInverseMap(const Vec3d& in) const
    {
        return applyFrustumInverseMap(mSecondMap->applyInverseMap(in));
    }

    /// Return the transpose of the inverse Jacobian of the linear second map applied to @c in
    Vec3d applyIJT(const Vec3d& in) const { return mSecondMap->applyIJT(in); }

    // the Jacobian of the nonlinear part of the transform is a sparse matrix
    // Jacobian^(-T) =
    //
    //    (Lx)(  1/s               0              0 )
    //        (  0                1/s             0 )
    //        (  -(x-xo)g/(sLx)   -(y-yo)g/(sLx)  Lz/(Depth Lx)   )
    /// Return the transpose of the inverse Jacobain (at @c locW applied to @c in.
    /// @c locI is the location in the pre-image space (e.g. index space)
    Vec3d applyIJT(const Vec3d& in, const Vec3d& loc) const
    {
        const double s = mGamma*loc.z() + 1.;
        double factor = mGamma / (s * mLx);

        Vec3d out;

        out.x() = (mLx/s) * in.x();
        out.y() = (mLx/s) * in.y();
        out.z() =
            -(loc.x() - mXo) * factor * in.x()
            -(loc.y() - mYo) * factor * in.y()
            + (mLz / (mDepth * mLx)) * in.z();

        if (mSecondMap->isLinear()) {
            return mSecondMap->applyIJT(out);
        } else {
            return mSecondMap->applyIJT(out, applyFrustumMap(loc));
        }
    }

    /// Return the Jacobian Curvature for the linear second map
    Mat3d applyIJC(const Mat3d& in) const { return mSecondMap->applyIJC(in); }
    /// Return the Jacobian Curvature
    /// @todo fix this for the frustum
    Mat3d applyIJC(const Mat3d& mat, const Vec3d& /*v*/, const Vec3d& /*pos*/) const
    {
        // applyIJC for nonlinear part,
        Mat3d resultOfNonlinear(mat);
        // next applyIJC for linear part
        return mSecondMap->applyIJC(resultOfNonlinear);
    }

    /// Return the determinant of the Jacobian of linear second map
    double determinant() const {return mSecondMap->determinant();} // no implementation

    /// Return the determinate of the Jacobian evaluated at @c loc
    /// @c loc is a location in the pre-image space (e.g., index space)
    double determinant(const Vec3d& loc) const
    {
        double s = mGamma * loc.z() + 1.0;
        double frustum_determinant = s * s * mDepthOnLzLxLx;
        return mSecondMap->determinant() * frustum_determinant;
    }

    /// Return the voxel dimensions of the linear second map
    Vec3d voxelDimensions() const { return mSecondMap->voxelDimensions(); }

    /// @brief Returns the lengths of the images of the three segments
    /// from @a loc to @a loc + (1,0,0), from @a loc to @a loc + (0,1,0)
    /// and from @a loc to @a loc + (0,0,1)
    /// @param loc  a location in the pre-image space (e.g., index space)
    Vec3d voxelDimensions(const Vec3d& loc) const {
        Vec3d out, pos = applyMap(loc);
        out(0) = (applyMap(loc + Vec3d(1,0,0)) - pos).length();
        out(1) = (applyMap(loc + Vec3d(0,1,0)) - pos).length();
        out(2) = (applyMap(loc + Vec3d(0,0,1)) - pos).length();
        return out;
    }


    AffineMap::Ptr getAffineMap()  const
    {
        return mSecondMap->getAffineMap();
    }

    /// set the taper value, the ratio of nearplane width / far plane width
    void setTaper(double t) { mTaper = t; init();}
    /// Return the taper value.
    double getTaper() const { return mTaper; }
    /// set the frustum depth: distance between near and far plane = frustm depth * frustm x-width
    void setDepth(double d) { mDepth = d; init();}
    /// Return the unscaled frustm depth
    double getDepth() const { return mDepth; }
    // gamma a non-dimensional  number:  nearplane x-width / camera to near plane distance
    double getGamma() const { return mGamma; }

    /// Set a bounding box in pre-image space (e.g. index space) that is converted to the frustm
    /// in image space
    void setBBox(const CoordBBox& bb) { mBBox = bb; }
    /// Return the bounding box that defines the frustum in pre-image space
    const CoordBBox& getBBox() const { return mBBox; }

    /// Return MapBase::Ptr& to the second map
    const MapBase::Ptr& linearMap() const { return mSecondMap; }

    /// Return @c true if the  the bounding box in index space that defines the region that
    /// is maped into the frustum is non-zero, otherwise @c false
    bool isValid() const { return !mBBox.empty();}

    /// Return @c true if the second map is a uniform scale, Rotation and translation
    bool hasSimpleAffine() const { return mHasSimpleAffine; }

    /// read serialization
    void read(std::istream& is)
    {
        mBBox.read(is);
        is.read(reinterpret_cast<char*>(&mTaper), sizeof(double));
        is.read(reinterpret_cast<char*>(&mDepth), sizeof(double));

        // Read the second maps type.
        Name type = readString(is);

        // Check if the map has been registered.
        if(!MapRegistry::isRegistered(type)) {
            OPENVDB_THROW(KeyError, "Map " << type << " is not registered");
        }

        // Create the second map of the type and then read it in.
        mSecondMap =  math::MapRegistry::createMap(type);
        mSecondMap->read(is);

        init();
    }

    /// write serialization
    void write(std::ostream& os) const
    {
        mBBox.write(os);
        os.write(reinterpret_cast<const char*>(&mTaper), sizeof(double));
        os.write(reinterpret_cast<const char*>(&mDepth), sizeof(double));

        // Write out the second map.
        if (!mSecondMap) {
            OPENVDB_THROW(IoError, "The NonlinearFrustumMap does not have a second map");
        }

        writeString(os, mSecondMap->type());
        mSecondMap->write(os);
    }

    /// string serialization, useful for debuging
    std::string str() const
    {
        std::ostringstream buffer;
        buffer << " - taper: " << mTaper << std::endl;
        buffer << " - depth: " << mDepth << std::endl;
        buffer << " SecondMap: "<< mSecondMap->type() << std::endl;
        buffer << mSecondMap->str() << std::endl;
        return buffer.str();
    }

    /// Return a MapBase::Ptr to a new map that is the result of accumulating rotation on this map
    MapBase::Ptr rotate(double radians, Axis axis = X_AXIS) const
    {
        return MapBase::Ptr(
            new NonlinearFrustumMap(mBBox, mTaper, mDepth, mSecondMap->rotate(radians, axis)));
    }
    /// @brief Return a MapBase::Ptr to a new map that is the result of accumulating
    /// translation on this map
    MapBase::Ptr translate(const Vec3d& t) const
    {
        return MapBase::Ptr(
            new NonlinearFrustumMap(mBBox, mTaper, mDepth, mSecondMap->translate(t)));
    }
    /// Return a MapBase::Ptr to a new map that is the result of accumulating scale on this map
    MapBase::Ptr scale(const Vec3d& s) const
    {
        return MapBase::Ptr(
            new NonlinearFrustumMap(mBBox, mTaper, mDepth, mSecondMap->scale(s)));
    }
    /// Return a MapBase::Ptr to a new map that is the result of accumulating shear on this map
    MapBase::Ptr shear(double shear, Axis axis0, Axis axis1) const
    {
        return MapBase::Ptr(new NonlinearFrustumMap(
            mBBox, mTaper, mDepth, mSecondMap->shear(shear, axis0, axis1)));
    }

private:
    void init() { 
        // set up as a frustum
        mLx = mBBox.extents().x()-1;
        mLy = mBBox.extents().y()-1;
        mLz = mBBox.extents().z()-1;
        mXo = 0.5* mLx;
        mYo = 0.5* mLy;

        // mDepth is non-dimensionalized on near
        mGamma = (1./mTaper - 1) / mDepth;
        //mGamma = mTaper -1;
        mDepthOnLz = mDepth/mLz;
        mDepthOnLzLxLx = mDepthOnLz/(mLx * mLx);

        /// test for shear and non-uniform scale
        mHasSimpleAffine = true;
        Vec3d tmp = mSecondMap->voxelDimensions();

        /// false if there is non-uniform scale
        if (!isApproxEqual(tmp(0), tmp(1))) { mHasSimpleAffine = false; return; }
        if (!isApproxEqual(tmp(0), tmp(2))) { mHasSimpleAffine = false; return; }

        Vec3d trans = mSecondMap->applyMap(Vec3d(0,0,0));
        /// look for shear
        Vec3d tmp1 = mSecondMap->applyMap(Vec3d(1,0,0)) - trans;
        Vec3d tmp2 = mSecondMap->applyMap(Vec3d(0,1,0)) - trans;
        Vec3d tmp3 = mSecondMap->applyMap(Vec3d(0,0,1)) - trans;

        /// false if there is shear
        if (!isApproxEqual(tmp1.dot(tmp2), 0., 1.e-7)) { mHasSimpleAffine  = false; return; }
        if (!isApproxEqual(tmp2.dot(tmp3), 0., 1.e-7)) { mHasSimpleAffine  = false; return; }
        if (!isApproxEqual(tmp3.dot(tmp1), 0., 1.e-7)) { mHasSimpleAffine  = false; return; }
    }

    Vec3d applyFrustumMap(const Vec3d& in) const
    {

        // move lower-left hand corder of x-face of
        // bbox to the orgin in index space
        Vec3d out(in);
        out = out - mBBox.min();
        out.x() -= mXo;
        out.y() -= mYo;

        // scale the z-direction on depth / K count
        out.z() *= mDepthOnLz;

        double scale = (mGamma * out.z() + 1.)/ mLx;

        // scale the x-y on the length I count and apply tapper
        out.x() *= scale ;
        out.y() *= scale ;

        return out;
    }

    Vec3d applyFrustumInverseMap(const Vec3d& in) const
    {
        // invert taper and resize:  scale = 1/( (z+1)/2 (mt-1) + 1)
        Vec3d out(in);
        double invScale = mLx / (mGamma * out.z() + 1.);
        out.x() *= invScale;
        out.y() *= invScale;

        out.x() += mXo;
        out.y() += mYo;

        out.z() /= mDepthOnLz;

        // move back
        out = out +  mBBox.min();
        return out;
    }

    // bounding box in index space used in Frustum transforms.
    CoordBBox   mBBox;

    // taper value used in constructing Frustums.
    double      mTaper;
    double      mDepth;

    // defines the second map
    MapBase::Ptr mSecondMap;

    // these are derived from the above.
    int mLx, mLy, mLz;
    double mXo, mYo, mGamma, mDepthOnLz, mDepthOnLzLxLx;
    // true: if the mSecondMap is linear and has no shear, and has no non-uniform scale
    bool mHasSimpleAffine;
}; // class NonlinearFrustumMap


////////////////////////////////////////


///  @brief Creates the composition of two maps, each of which could be a composition.
///  In the case that each component of the composition classified as linear an
///  acceleration AffineMap is stored.
template<typename FirstMapType, typename SecondMapType>
class CompoundMap
{
public:
    typedef CompoundMap<FirstMapType, SecondMapType>    MyType;

    typedef boost::shared_ptr<MyType>       Ptr;
    typedef boost::shared_ptr<const MyType> ConstPtr;


    CompoundMap() { updateAffineMatrix(); }

    CompoundMap(const FirstMapType& f, const SecondMapType& s): mFirstMap(f), mSecondMap(s)
    {
        updateAffineMatrix();
    }

    CompoundMap(const MyType& other):
        mFirstMap(other.mFirstMap),
        mSecondMap(other.mSecondMap),
        mAffineMap(other.mAffineMap)
    {}

    Name type() const { return mapType(); }
    static Name mapType()
    {
        return (FirstMapType::mapType() + Name(":") + SecondMapType::mapType());
    }

    bool operator==(const MyType& other) const
    {
        if (mFirstMap != other.mFirstMap)   return false;
        if (mSecondMap != other.mSecondMap) return false;
        if (mAffineMap != other.mAffineMap) return false;
        return true;
    }

    bool operator!=(const MyType& other) const { return !(*this == other); }

    MyType& operator=(const MyType& other)
    {
        mFirstMap = other.mFirstMap;
        mSecondMap = other.mSecondMap;
        mAffineMap = other.mAffineMap;
        return *this;
    }

    bool isIdentity() const {
        if (is_linear<MyType>::value) {
            return mAffineMap.isIdentity();
        } else {
            return mFirstMap.isIdentity()&&mSecondMap.isIdentity();
        }
    }

    bool isDiagonal() const {
        if (is_linear<MyType>::value) {
            return mAffineMap.isDiagonal();
        } else {
            return mFirstMap.isDiagonal()&&mSecondMap.isDiagonal();
        }
    }
    AffineMap::Ptr getAffineMap() const
    {
        if (is_linear<MyType>::value) {
            AffineMap::Ptr affine(new AffineMap(mAffineMap));
            return affine;
        } else {
            OPENVDB_THROW(ArithmeticError,
                "Constant affine matrix representation not possible for this nonlinear map");
        }
    }

    // direct decompotion
    const FirstMapType& firstMap() const { return mFirstMap; }
    const SecondMapType& secondMap() const {return mSecondMap; }

    void setFirstMap(const FirstMapType& first) { mFirstMap = first; updateAffineMatrix(); }
    void setSecondMap(const SecondMapType& second) { mSecondMap = second; updateAffineMatrix(); }

    void read(std::istream& is)
    {
        mAffineMap.read(is);
        mFirstMap.read(is);
        mSecondMap.read(is);
    }
    void write(std::ostream& os) const
    {
        mAffineMap.write(os);
        mFirstMap.write(os);
        mSecondMap.write(os);
    }

private:
    void updateAffineMatrix()
    {
        if (is_linear<MyType>::value) {
            // both maps need to be linear, these methods are only defined for linear maps
            AffineMap::Ptr first = mFirstMap.getAffineMap();
            AffineMap::Ptr second= mSecondMap.getAffineMap();
            mAffineMap = AffineMap(*first, *second);
        }
    }

    FirstMapType   mFirstMap;
    SecondMapType  mSecondMap;
    // used for acceleration
    AffineMap      mAffineMap;
}; // class CompoundMap


} // namespace math
} // namespace OPENVDB_VERSION_NAME
} // namespace openvdb

#endif // OPENVDB_MATH_MAPS_HAS_BEEN_INCLUDED

// Copyright (c) 2012 DreamWorks Animation LLC
// All rights reserved. This software is distributed under the
// Mozilla Public License 2.0 ( http://www.mozilla.org/MPL/2.0/ )
