///////////////////////////////////////////////////////////////////////////
//
// Copyright (c) 2012 DreamWorks Animation LLC
//
// All rights reserved. This software is distributed under the
// Mozilla Public License 2.0 ( http://www.mozilla.org/MPL/2.0/ )
//
// Redistributions of source code must retain the above copyright
// and license notice and the following restrictions and disclaimer.
//
// *     Neither the name of DreamWorks Animation nor the names of
// its contributors may be used to endorse or promote products derived
// from this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
// OWNER OR CONTRIBUTORS BE LIABLE FOR ANY INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
// IN NO EVENT SHALL THE COPYRIGHT HOLDERS' AND CONTRIBUTORS' AGGREGATE
// LIABILITY FOR ALL CLAIMS REGARDLESS OF THEIR BASIS EXCEED US$250.00.
//
///////////////////////////////////////////////////////////////////////////

#ifndef OPENVDB_IO_COMPRESSION_HAS_BEEN_INCLUDED
#define OPENVDB_IO_COMPRESSION_HAS_BEEN_INCLUDED

#include <openvdb/Types.h>
#include <algorithm>
#include <iostream>
#include <vector>


namespace openvdb {
OPENVDB_USE_VERSION_NAMESPACE
namespace OPENVDB_VERSION_NAME {
namespace io {

/// @brief RealToHalf and its specializations define a mapping from
/// floating-point data types to analogous half float types.
template<typename T>
struct RealToHalf {
    enum { isReal = false }; // unless otherwise specified, type T is not a floating-point type
    typedef T HalfT; // type T's half float analogue is T itself
};
template<> struct RealToHalf<float>  { enum { isReal = true }; typedef half  HalfT; };
template<> struct RealToHalf<double> { enum { isReal = true }; typedef half  HalfT; };
template<> struct RealToHalf<Vec2s>  { enum { isReal = true }; typedef Vec2H HalfT; };
template<> struct RealToHalf<Vec2d>  { enum { isReal = true }; typedef Vec2H HalfT; };
template<> struct RealToHalf<Vec3s>  { enum { isReal = true }; typedef Vec3H HalfT; };
template<> struct RealToHalf<Vec3d>  { enum { isReal = true }; typedef Vec3H HalfT; };


OPENVDB_API void zipToStream(std::ostream&, const char* data, size_t numBytes);
OPENVDB_API void unzipFromStream(std::istream&, char* data, size_t numBytes);

/// @brief Read data from a stream.
/// @param is          the input stream
/// @param data        the contiguous array of data to read in
/// @param count       the number of elements to read in
/// @param compressed  if @c true, assume the data is ZIP compressed and uncompress it
/// This default implementation is instantiated only for types whose size
/// can be determined by the sizeof() operator.
template<typename T>
inline void
readData(std::istream& is, T* data, Index count, bool compressed)
{
    if (compressed) {
        unzipFromStream(is, reinterpret_cast<char*>(data), sizeof(T) * count);
    } else {
        is.read(reinterpret_cast<char*>(data), sizeof(T) * count);
    }
}

/// Specialization for std::string input
template<>
inline void
readData<std::string>(std::istream& is, std::string* data, Index count, bool /*compressed*/)
{
    for (Index i = 0; i < count; ++i) {
        size_t len = 0;
        is >> len;
        //data[i].resize(len);
        //is.read(&(data[i][0]), len);

        std::string buffer(len+1, ' ');
        is.read(&buffer[0], len+1 );
        data[i].assign(buffer, 0, len);
    }
}

/// HalfReader wraps a static function, read(), that is analogous to readData(), above,
/// except that it is partially specialized for floating-point types in order to promote
/// 16-bit half float values to full float.  A wrapper class is required because
/// only classes, not functions, can be partially specialized.
template<bool IsReal, typename T> struct HalfReader;
/// Partial specialization for non-floating-point types (no half to float promotion)
template<typename T>
struct HalfReader</*IsReal=*/false, T> {
    static inline void read(std::istream& is, T* data, Index count, bool compressed) {
        readData(is, data, count, compressed);
    }
};
/// Partial specialization for floating-point types
template<typename T>
struct HalfReader</*IsReal=*/true, T> {
    typedef typename RealToHalf<T>::HalfT HalfT;
    static inline void read(std::istream& is, T* data, Index count, bool compressed) {
        std::vector<HalfT> halfData(count); // temp buffer into which to read half float values
        readData<HalfT>(is, reinterpret_cast<HalfT*>(&halfData[0]), count, compressed);
        // Copy half float values from the temporary buffer to the full float output array.
        std::copy(halfData.begin(), halfData.end(), data);
    }
};


/// Write data to a stream.
/// @param os        the output stream
/// @param data      the contiguous array of data to write
/// @param count     the number of elements to write out
/// @param compress  if @c true, apply ZIP compression to the data
/// This default implementation is instantiated only for types whose size
/// can be determined by the sizeof() operator.
template<typename T>
inline void
writeData(std::ostream &os, const T *data, Index count, bool compress)
{
    if (compress) {
        zipToStream(os, reinterpret_cast<const char*>(data), sizeof(T) * count);
    } else {
        os.write(reinterpret_cast<const char*>(data), sizeof(T) * count);
    }
}

/// Specialization for std::string output
/// @todo Add compression
template<>
inline void
writeData<std::string>(std::ostream& os, const std::string* data, Index count, bool /*compress*/)
{
    for (Index i = 0; i < count; ++i) {
        const size_t len = data[i].size();
        os << len;
        os.write(data[i].c_str(), len+1);
        //os.write(&(data[i][0]), len );
    }
}

/// HalfWriter wraps a static function, write(), that is analogous to writeData(), above,
/// except that it is partially specialized for floating-point types in order to quantize
/// floating-point values to 16-bit half float.  A wrapper class is required because
/// only classes, not functions, can be partially specialized.
template<bool IsReal, typename T> struct HalfWriter;
/// Partial specialization for non-floating-point types (no float to half quantization)
template<typename T>
struct HalfWriter</*IsReal=*/false, T> {
    static inline void write(std::ostream& os, const T* data, Index count, bool compress) {
        writeData(os, data, count, compress);
    }
};
/// Partial specialization for floating-point types
template<typename T>
struct HalfWriter</*IsReal=*/true, T> {
    typedef typename RealToHalf<T>::HalfT HalfT;
    static inline void write(std::ostream& os, const T* data, Index count, bool compress) {
        // Convert full float values to half float, then output the half float array.
        std::vector<HalfT> halfData(count);
        std::copy(data, data + count, halfData.begin());
        writeData<HalfT>(os, reinterpret_cast<const HalfT*>(&halfData[0]), count, compress);
    }
};

} // namespace io
} // namespace OPENVDB_VERSION_NAME
} // namespace openvdb

#endif // OPENVDB_IO_COMPRESSION_HAS_BEEN_INCLUDED

// Copyright (c) 2012 DreamWorks Animation LLC
// All rights reserved. This software is distributed under the
// Mozilla Public License 2.0 ( http://www.mozilla.org/MPL/2.0/ )
