#
# This script was written by George A. Theall, <theall@tifaware.com>.
#
# See the Nessus Scripts License for details.
#


# NB: I define the script description here so I can later modify
#     it with the version number and install directory.
  desc["english"] = "
This script detects whether the target is running Open WebMail and
extracts version numbers and locations of any instances found. 

Open WebMail is a webmail package written in Perl that provides access
to mail accounts via POP3 or IMAP.  See <http://www.openwebmail.org/>
for more information.";


if (description) {
  script_id(14221);
  script_version ("$Revision: 38 $"); 

  name["english"] = "Open WebMail Detection";
  script_name(english:name["english"]);
 
  script_description(english:desc["english"]);
 
  summary["english"] = "Checks for the presence of Open WebMail";
  script_summary(english:summary["english"]);
 
  script_category(ACT_GATHER_INFO);
  script_copyright(english:"This script is Copyright (C) 2004 George A. Theall");

  family["english"] = "General";
  script_family(english:family["english"]);

  script_dependencie("global_settings.nasl", "http_version.nasl", "no404.nasl");
  script_require_ports("Services/www", 80);
 script_exclude_keys("Settings/disable_cgi_scanning");

  exit(0);
}

include("global_settings.inc");
include("http_func.inc");
include("http_keepalive.inc");

host = get_host_name();
port = get_http_port(default:80);

if (!get_port_state(port)) exit(0);
if (get_kb_item("www/no404/" + port)) exit(0);
if (debug_level) display("debug: looking for Open WebMail on ", host, ":", port, ".\n");

# Search for Open WebMail in a couple of different locations.
#
# NB: Directories beyond cgi_dirs() come from a Google search - 
#     'inurl:openwebmail.pl userid' - and represent the more popular
#     installation paths currently. Still, cgi_dirs() should catch
#     the directory if its referenced elsewhere on the target.
dirs = make_list(cgi_dirs());
xtra_dirs = make_array(
  "/cgi-bin/openwebmail", 1,
  "/openwebmail-cgi", 1
);
foreach dir (dirs) {
  # Set value to zero if it's already in dirs.
  if (!isnull(xtra_dirs[dir])) xtra_dirs[dir] = 0;
}
foreach dir (keys(xtra_dirs)) {
  # Add it to dirs if the value is still set.
  if (xtra_dirs[dir]) dirs = make_list(dirs, dir);
}

installs = 0;
foreach dir (dirs) {
  url = string(dir, "/openwebmail.pl");
  if (debug_level) display("debug: retrieving ", url, "...\n");
  req = http_get(item:url, port:port);
  res = http_keepalive_send_recv(port:port, data:req);
  if (isnull(res)) exit(0);             # can't connect
  if (debug_level) display("debug: res =>>", res, "<<\n");

  # If the page refers to Open WebMail, try to get its version number.
  if (
    egrep(string:res, pattern:"^HTTP/.* 200 OK") &&
    egrep(string:res, pattern:"(http://openwebmail\.org|Open WebMail)")
  ) {
    # First see if version's included in the form. If it is, Open WebMail 
    # puts it on a line by itself, prefixed by the word "version".
    pat = "^version (.+)$";
    if (debug_level) display("debug: grepping results for =>>", pat, "<<\n");
    matches = egrep(pattern:pat, string:res);
    foreach match (split(matches)) {
      match = chomp(match);
      if (debug_level) display("debug: grepping >>", match, "<< for =>>", pat, "<<\n");
      ver = eregmatch(pattern:pat, string:match);
      if (!isnull(ver)) ver = ver[1];
      break;                            # nb: only worried about first match.
    }

    # If that didn't work, looking for it in doc/changes.txt,
    # under the Open WebMail data directory.
    if (isnull(ver)) {
      # Identify data directory from links to images or help files.
      pat = '([^\'"]*/openwebmail)/(images|help)/';
      if (debug_level) display("debug: grepping results for =>>", pat, "<<\n");
      matches = egrep(pattern:pat, string:res);
      foreach match (split(matches)) {
        match = chomp(match);
        if (debug_level) display("debug: grepping >>", match, "<< for =>>", pat, "<<\n");
        data_url = eregmatch(string:match, pattern:pat);
        if (!isnull(data_url)) data_url = data_url[1];
        break;                          # nb: only worried about first match.
      }
      # Try to get doc/changes.txt under data directory.
      if (!isnull(data_url)) {
        if (debug_level) display("debug: url for data files =>>", data_url, "<<\n");
        url = string(data_url, "/doc/changes.txt");
        if (debug_level) display("debug: retrieving ", url, "...\n");
        req = http_get(item:url, port:port);
        res = http_keepalive_send_recv(port:port, data:req);
        if (isnull(res)) exit(0);       # can't connect
        if (debug_level) display("debug: res =>>", res, "<<\n");

        # Try to get version number.
        #
        # nb: this won't identify intermediate releases, only full ones.
        if (egrep(string:res, pattern:"^HTTP/.* 200 OK")) {
          pat = "^[0-1][0-9]/[0-3][0-9]/20[0-9][0-9]( +.version .+)?";
          if (debug_level) display("debug: grepping results for =>>", pat, "<<\n");
          matches = egrep(pattern:pat, string:res);
          foreach match (split(matches)) {
            match = chomp(match);
            if (debug_level) display("debug: grepping >>", match, "<< for =>>", pat, "<<\n");
            ver = eregmatch(pattern:"version +(.+).$", string:match);
            if (isnull(ver)) {
              # nb: only first release date matters.
              if (isnull(rel)) {
                # Rearrange date: mm/dd/yyyy -> yyyyddmm.
                parts = split(match, sep:"/", keep:FALSE);
                rel = string(parts[2], parts[0], parts[1]);
              }
            }
            else {
              ver = ver[1];
              if (!isnull(rel)) ver = string(ver, " ", rel);
              break;                    # nb: only worried about first match.
            }
          }
        }
      }
    }

    # nb: in the event the version number is still unknown, I want 
    #     to record the fact that there's *some* version installed.
    if (isnull(ver)) {
      ver = "*** VERSION UNKNOWN ***";
      if (log_verbosity > 1) display("Can't determine version of Open WebMail installed under ", dir, " on ", host, ":", port, "!\n");
    }

    if (debug_level) display("debug: Open WebMail version =>>", ver, "<<\n");
    set_kb_item(
      name:string("www/", port, "/openwebmail"), 
      value:string(ver, " under ", dir)
    );
    installations[dir] = ver;
    ++installs;
  }
  # Scan for multiple installations only if "Thorough Tests" is checked.
  if (installs && !thorough_tests) break;
}

# Report any instances found unless Report verbosity is "Quiet".
if (installs && report_verbosity > 0) {
  if (installs == 1) {
    foreach dir (keys(installations)) {
      # empty - just need to set 'dir'.
    }
    info = string("Open WebMail ", ver, " was detected on the remote host under the path ", dir, ".");
  }
  else {
    info = string(
      "Multiple instances of Open WebMail were detected on the remote host:\n",
      "\n"
    );
    foreach dir (keys(installations)) {
      info = info + string("    ", installations[dir], ", installed under ", dir, "\n");
    }
    info = chomp(info);
  }

  desc = ereg_replace(
    string:desc["english"],
    pattern:"This script[^\.]+\.", 
    replace:info
  );
  security_note(port:port, data:desc);
}
