/****************************************************************************************/
/*											*/
/* This program is free software; you can redistribute it and/or modify it under the	*/
/* terms of the GNU General Public License as published by the Free Software		*/
/* Foundation; either version 2 of the License, or (at your option) any later version.	*/
/*											*/
/* This program is distributed in the hope that it will be useful, but WITHOUT ANY	*/
/* WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A	*/
/* PARTICULAR PURPOSE. See the GNU General Public License for more details.		*/
/*											*/
/* You should have received a copy of the GNU General Public License along with this	*/
/* program; (See "COPYING"). If not, If not, see <http://www.gnu.org/licenses/>.        */
/*											*/
/*--------------------------------------------------------------------------------------*/
/*											*/
/*  Copyright   Joerg Anders, TU Chemnitz, Fakultaet fuer Informatik, GERMANY           */
/*		ja@informatik.tu-chemnitz.de						*/
/*											*/
/*											*/
/****************************************************************************************/

#include <string.h>
#include "system.h"
#include "staff.h"
#include "page.h"
#include "chordorrest.h"
#include "resource.h"
#include "mainwindow.h"
#include "clipboard.h"
#include "commandlist.h"
#include "commandhistory.h"
#include "voice.h"
#include "movechordsandrestscommand.h"
#include "movechordsandrestsreversecommand.h"
#include "appendstaffcommand.h"
#include "mainwindow.h"
#include "musicxmlimport.h"

#define DEFAULT_SYSTEM_DIST (4 * LINE_DIST)
#define STAFF_START_LEFT_THICK 0.08

#define X_POS_PAGE_REL(p) ((getPage()->getContetXpos() + (p)) * zoom_factor - leftx)
#define Y_POS(p) ((p) * zoom_factor - topy)
#define Y_POS_INVERSE(p) (((p)  + topy) / zoom_factor)

#define X_PS_POS(p) ((DEFAULT_BORDER + LEFT_RIGHT_BORDER + (p)) * PS_ZOOM)
#define Y_PS_POS(p) ((height - (p)) * PS_ZOOM)


NedSystem::NedSystem(NedPage *page, double ypos, double width, int nr, unsigned int start_measure_number, bool start) :
m_system_start(CLEF_SPACE+ TIME_SIGNATURE_SPACE), m_isUntouched(TRUE), m_staffs(NULL), m_deleted_staffs(NULL), m_ypos(ypos), m_width(width), m_system_number(nr),  m_page(page), m_measure_count(0), m_only_whole_elements(FALSE) {
	int i;
	double staff_pos = 0;
	double indent;
	NedStaff *staff;
	GList *lptr;
	for (i = 0; i < MAX_MEASURES; m_measures[i++].setSystem(this));
	if (start) {
		for (i = 0; i < getMainWindow()->getStaffCount(); i++) {
			m_staffs = g_list_append(m_staffs,
				staff = new NedStaff(this,
			 	staff_pos, m_width, i, TRUE));
			staff_pos += staff->getHeight() + DEFAULT_SYSTEM_DIST;
		}
		do {
			for (lptr = g_list_first(m_staffs); lptr; lptr = g_list_next(lptr)) {
				((NedStaff *) lptr->data)->appendWholeRest();
			}
			//do_reposit();
			indent = (m_system_number == 0 && m_page->getPageNumber() == 0) ? getMainWindow()->getFirstSystemIndent() : 0.0;
			reposit(0, start_measure_number);
		}
		while (m_extra_space > START_EXTRA_SPACE);
	}
}
	
NedSystem::~NedSystem() {
	GList *lptr;

	for (lptr = g_list_first(m_staffs); lptr; lptr = g_list_next(lptr)) {
		delete ((NedStaff *) (lptr->data));
	}
	g_list_free(m_staffs);
	m_staffs = NULL;

	for (lptr = g_list_first(m_deleted_staffs); lptr; lptr = g_list_next(lptr)) {
		delete ((NedStaff *) (lptr->data));
	}
	g_list_free(m_deleted_staffs);
	m_deleted_staffs = NULL;
}

NedStaff *NedSystem::getStaff(int nr) {
	GList *lptr;

	if ((lptr = g_list_nth(m_staffs, nr)) == NULL) {
		return NULL;
	}
	return (NedStaff *) lptr->data;
}

void NedSystem::fill_up(double indent, NedCommandList *command_list) {
	GList *lptr;


	do {
		for (lptr = g_list_first(m_staffs); lptr; lptr = g_list_next(lptr)) {
			((NedStaff *) lptr->data)->appendWholeRest(command_list);
		}
		compute_extra_space(indent);
		//do_reposit();
	}
	while (m_extra_space > MINIMUM_EXTRA_SPACE);
	if (m_only_whole_elements) {
		m_isUntouched = TRUE;
	}
}

double NedSystem::getHeight() {
	NedStaff *staff;

	staff = (NedStaff *) g_list_last(m_staffs)->data;
	return staff->getBottomPos();
}

NedMainWindow *NedSystem::getMainWindow() {return m_page->getMainWindow();}

void NedSystem::draw(cairo_t *cr, bool first_page) {
	double leftx = getMainWindow()->getLeftX();
	double topy = getMainWindow()->getTopY();
	double zoom_factor = getMainWindow()->getCurrentZoomFactor();
	NedStaff *firststaff, *laststaff;
	NedStaff *staff;
	double brace_start_pos = -1.0;
	double bracket_start_pos = -1.0;
	bool inside_connected_staves = false;
	int i;
	double botp;
	double indent = first_page ? getMainWindow()->getFirstSystemIndent() : 0;

	GList *lptr1, *lptr2;
	lptr1 = g_list_first(m_staffs);
	lptr2 = g_list_next(lptr1);
	for (i = 0, lptr2 = g_list_next(lptr1); lptr1; lptr1 = lptr2, lptr2 = g_list_next(lptr2), i++) {
		if ((getMainWindow()->m_staff_contexts[i].m_flags & CON_BAR_START) != 0) {
			inside_connected_staves = true;
		}
		if ((getMainWindow()->m_staff_contexts[i].m_flags & CON_BAR_END) != 0) {
			inside_connected_staves = false;
		}
		if (inside_connected_staves) {
			if (lptr2 == NULL) {
				NedResource::Abort("NedSystem::draw");
			}
			botp = ((NedStaff *) lptr2->data)->getBottomPos();
		}
		else {
			botp = ((NedStaff *) lptr1->data)->getBottomPos();
		}

		((NedStaff *) lptr1->data)->draw(cr, botp, first_page, indent);
	}
	firststaff = (NedStaff *) g_list_first(m_staffs)->data;
	laststaff = (NedStaff *) g_list_last(m_staffs)->data;
	//drawBrace(cr, leftx, topy, zoom_factor, laststaff->getBottomPos() - firststaff->getTopPos(), firststaff->getTopPos());
	//drawBracket(cr, leftx, topy, zoom_factor, firststaff->getTopPos(), laststaff->getBottomPos());
	cairo_new_path(cr);
	cairo_set_source_rgb (cr, 0.0, 0.0, 0.0);
	cairo_set_line_width(cr, zoom_factor * STAFF_START_LEFT_THICK);
	cairo_move_to(cr, X_POS_PAGE_REL(indent),
		 Y_POS(m_ypos + firststaff->getTopPos() - LINE_THICK / 2.0));
	cairo_line_to(cr,  X_POS_PAGE_REL(indent),
		 Y_POS(m_ypos + laststaff->getBottomPos() + LINE_THICK / 2.0));
	cairo_move_to(cr, X_POS_PAGE_REL(m_width),
		 Y_POS(m_ypos + firststaff->getTopPos() - LINE_THICK / 2.0));
	cairo_line_to(cr,  X_POS_PAGE_REL(m_width),
		 Y_POS(m_ypos + laststaff->getBottomPos() + LINE_THICK / 2.0));
	cairo_stroke(cr);
	for (i = 0; i < getMainWindow()->getStaffCount(); i++) {
		staff = (NedStaff *) g_list_nth(m_staffs, i)->data;
		if ((getMainWindow()->m_staff_contexts[i].m_flags & BRACE_START) != 0) {
			brace_start_pos = staff->getTopPos();
		}
		if ((getMainWindow()->m_staff_contexts[i].m_flags & BRACE_END) != 0) {
			if (brace_start_pos >= 0.0) {
				drawBrace(cr, leftx, indent, topy, zoom_factor, staff->getBottomPos() - brace_start_pos, brace_start_pos);
				brace_start_pos = -1.0;
			}
		}
		if ((getMainWindow()->m_staff_contexts[i].m_flags & BRACKET_START) != 0) {
			bracket_start_pos = staff->getTopPos();
		}
		if ((getMainWindow()->m_staff_contexts[i].m_flags & BRACKET_END) != 0) {
			if (bracket_start_pos >= 0.0) {
				drawBracket(cr, leftx, indent, topy, zoom_factor, bracket_start_pos, staff->getBottomPos());
				bracket_start_pos = -1.0;
			}
		}
	}
};


void NedSystem::drawBrace (cairo_t *cr, double leftx, double indent, double topy, double zoom_factor, double height, double toppos) {
	int i;
	cairo_new_path(cr);
	cairo_set_line_width(cr, 1.0);
#define BRACE_OFFS -0.03
#define BRACE_X_FACTOR 0.004
#define BRACE_Y_FACTOR 0.00106
#define FAKX(a) (BRACE_X_FACTOR * zoom_factor * a)
#define FAKY(a) (BRACE_Y_FACTOR * zoom_factor * a * height)
	cairo_move_to(cr, X_POS_PAGE_REL(indent-0.05), Y_POS(m_ypos + toppos + BRACE_OFFS));
	for (i = 0; i < 16; i++) {
		cairo_rel_curve_to(cr, FAKX(NedResource::m_braceOutline[i][0]), FAKY(NedResource::m_braceOutline[i][1]),
				   FAKX(NedResource::m_braceOutline[i][2]), FAKY(NedResource::m_braceOutline[i][3]),
				   FAKX(NedResource::m_braceOutline[i][4]), FAKY(NedResource::m_braceOutline[i][5]));
	}
	cairo_fill(cr);
}

void NedSystem::drawBracket (cairo_t *cr, double leftx, double indent, double topy, double zoom_factor, double toppos, double botpos) {
	int i;
	cairo_new_path(cr);
	cairo_set_line_width(cr, 1.0);
#define BRACKET_X_OFFS -0.2
#define BRACKET_Y_OFFS 0.1
#define BRACKET_X_FACTOR 0.0008
#define BRACKET_Y_FACTOR 0.0008
#define BRACKET_FAKX(a) (BRACKET_X_FACTOR * zoom_factor * a)
#define BRACKET_FAKY(a) (BRACKET_Y_FACTOR * zoom_factor * a)
#define BRACKET_THICK 0.1
#define BRACKET_LINE_DIST 0.05
	cairo_move_to(cr, X_POS_PAGE_REL(indent+BRACKET_X_OFFS), Y_POS(m_ypos + botpos  + BRACKET_Y_OFFS));
	for (i = 0; i < 5; i++) {
		cairo_rel_curve_to(cr, BRACKET_FAKX(NedResource::m_bracketEndOutline[i][0]), BRACKET_FAKY(NedResource::m_bracketEndOutline[i][1]),
				   BRACKET_FAKX(NedResource::m_bracketEndOutline[i][2]), BRACKET_FAKY(NedResource::m_bracketEndOutline[i][3]),
				   BRACKET_FAKX(NedResource::m_bracketEndOutline[i][4]), BRACKET_FAKY(NedResource::m_bracketEndOutline[i][5]));
	}
	cairo_close_path(cr);
	cairo_fill(cr);
	cairo_new_path(cr);
	cairo_move_to(cr, X_POS_PAGE_REL(indent+BRACKET_X_OFFS), Y_POS(m_ypos + toppos  - BRACKET_Y_OFFS));
	for (i = 0; i < 5; i++) {
		cairo_rel_curve_to(cr, BRACKET_FAKX(NedResource::m_bracketEndOutline[i][0]), BRACKET_FAKY(-NedResource::m_bracketEndOutline[i][1]),
				   BRACKET_FAKX(NedResource::m_bracketEndOutline[i][2]), BRACKET_FAKY(-NedResource::m_bracketEndOutline[i][3]),
				   BRACKET_FAKX(NedResource::m_bracketEndOutline[i][4]), BRACKET_FAKY(-NedResource::m_bracketEndOutline[i][5]));
	}
	cairo_close_path(cr);
	cairo_fill(cr);
	cairo_new_path(cr);
	cairo_set_line_width(cr, zoom_factor * BRACKET_THICK);
	cairo_move_to(cr, X_POS_PAGE_REL(indent+BRACKET_X_OFFS - BRACKET_LINE_DIST), Y_POS(m_ypos + toppos - BRACKET_Y_OFFS));
	cairo_line_to(cr, X_POS_PAGE_REL(indent+BRACKET_X_OFFS - BRACKET_LINE_DIST), Y_POS(m_ypos + botpos + BRACKET_Y_OFFS));
	cairo_stroke(cr);

}

void NedSystem::empty() {
	GList *lptr;
	for (lptr = g_list_first(m_staffs); lptr; lptr = g_list_next(lptr)) {
		((NedStaff *) lptr->data)->empty();
	}
}


bool NedSystem::trySelect(double x, double y) {
	GList *lptr;
	for (lptr = g_list_first(m_staffs); lptr; lptr = g_list_next(lptr)) {
		if (((NedStaff *) lptr->data)->trySelect(x, y)) {
			return TRUE;
		}
	}
	return FALSE;
}

bool NedSystem::tryChangeLength(NedChordOrRest *chord_or_rest) {
	GList *lptr;
	for (lptr = g_list_first(m_staffs); lptr; lptr = g_list_next(lptr)) {
		if (((NedStaff *) lptr->data)->tryChangeLength(chord_or_rest)) {
			return TRUE;
		}
	}
	return FALSE;
}

double NedSystem::computeMidDist(double y) {
	double tp, bt;
	double topy = getMainWindow()->getTopY();
	double zoom_factor = getMainWindow()->getCurrentZoomFactor();

	tp = Y_POS(m_ypos + ((NedStaff *) g_list_first(m_staffs)->data)->getTopPos());
	bt = Y_POS(m_ypos + ((NedStaff *) g_list_last(m_staffs)->data)->getBottomPos());


	if (y < tp) {
		return tp - y;
	}
	if (y > bt) {
		return y - bt;
	}
	return 0.0;
}

bool NedSystem::testLineOverflow(double indent) {
	compute_extra_space(indent);
	return m_extra_space < MINIMUM_EXTRA_SPACE;
}

void NedSystem::removeLastImported() {
	GList *lptr;

	for (lptr = g_list_first(m_staffs); lptr; lptr = g_list_next(lptr)) {
		((NedStaff *) lptr->data)->removeLastImported();
	}
}

bool NedSystem::truncateAtStart(NedCommandList *command_list, unsigned long long midi_time) {
	GList *lptr;

	for (lptr = g_list_first(m_staffs); lptr; lptr = g_list_next(lptr)) {
		if (!((NedStaff *) lptr->data)->truncateAtStart(command_list, midi_time)) return false;
	}
	return true;
}

bool NedSystem::handleImportedTuplets() {
	GList *lptr;

	for (lptr = g_list_first(m_staffs); lptr; lptr = g_list_next(lptr)) {
		if (!((NedStaff *) lptr->data)->handleImportedTuplets()) return false;
	}
	return true;
}

void NedSystem::prepareForImport() {
	GList *lptr;

	for (lptr = g_list_first(m_staffs); lptr; lptr = g_list_next(lptr)) {
		((NedStaff *) lptr->data)->prepareForImport();
	}
}

void NedSystem::appendElementsOfMeasureLength(part *parts, unsigned int meas_duration) {
	NedStaff *staff;
	part *part_ptr;
	int i;
	GList *lptr;

	for (i = 0, lptr = g_list_first(m_staffs); lptr; lptr = g_list_next(lptr), i++) {
		staff = (NedStaff *) lptr->data;
		part_ptr = &(parts[i]);
		staff->appendElementsOfMeasureLength(part_ptr, meas_duration);
	}
	m_isUntouched = FALSE;
}

void NedSystem::handleStaffElements() {
	GList *lptr;

	for (lptr = g_list_first(m_staffs); lptr; lptr = g_list_next(lptr)) {
		((NedStaff *) lptr->data)->handleStaffElements();
	}
}

void NedSystem::removeUnneededAccidentals() {
	GList *lptr;

	for (lptr = g_list_first(m_staffs); lptr; lptr = g_list_next(lptr)) {
		((NedStaff *) lptr->data)->removeUnneededAccidentals();
	}
}


bool NedSystem::tryInsertOrErease(double x, double y) {
	GList *lptr;
	double mindist, d;
	lptr = g_list_first(m_staffs);
	NedStaff *nearest_staff = (NedStaff *) lptr->data;
	mindist = nearest_staff->computeMidDist(y);
	for (lptr = g_list_next(lptr); lptr; lptr = g_list_next(lptr)) {
		if ((d = ((NedStaff *) lptr->data)->computeMidDist(y)) < mindist) {
			nearest_staff = (NedStaff *) lptr->data;
			mindist = d;
		}
	}
	if (nearest_staff->tryInsertOrErease(x, y)) {
		return TRUE;
	}
	return FALSE;
}


bool NedSystem::findLine(double x, double y, int *ypos, int *line, int *bottom) {
	GList *lptr;
	double mindist, d;
	lptr = g_list_first(m_staffs);
	NedStaff *nearest_staff = (NedStaff *) lptr->data;
	mindist = nearest_staff->computeMidDist(y);
	for (lptr = g_list_next(lptr); lptr; lptr = g_list_next(lptr)) {
		if ((d = ((NedStaff *) lptr->data)->computeMidDist(y)) < mindist) {
			nearest_staff = (NedStaff *) lptr->data;
			mindist = d;
		}
	}
	if (nearest_staff->findLine(x, y, ypos, line, bottom)) {
		return TRUE;
	}
	return FALSE;
}

NedStaff *NedSystem::findStaff(double x, double y, NedMeasure **measure) {
	double zoom_factor = getMainWindow()->getCurrentZoomFactor();
	double topy = getMainWindow()->getTopY();
	double yl = Y_POS_INVERSE(y) - m_ypos;
	int i;
	GList *lptr;

	x -= getPage()->getContetXpos();
	for (lptr = g_list_first(m_staffs); lptr; lptr = g_list_next(lptr)) {
		if (((NedStaff *) lptr->data)->getTopPos() < yl && ((NedStaff *) lptr->data)->getBottomPos() > yl) {
			for (i = 0; i <= m_measure_count; i++) {
				if (m_measures[i].isNearStart(x)) {
					*measure = &(m_measures[i]);
				}
			}
			return (NedStaff *) lptr->data;
		}
	}
	return NULL;
}
void NedSystem::appendStaff(NedCommandList *command_list, int p_staff_nr /* = -1 */) {
	NedStaff *staff;
	int staff_nr;

	if (p_staff_nr < 0) {
		staff_nr = g_list_length(m_staffs);
	}
	else {
		staff_nr = p_staff_nr;
	}
	staff = new NedStaff(this, 10.0 /* dummy */, m_width, staff_nr, TRUE);
	staff->appendAppropriateWholes();

	command_list->addCommand(new NedAppendStaffCommand(this, staff));
}

void NedSystem::do_append_staff(NedStaff *staff) {
	m_staffs = g_list_append(m_staffs, staff);
}

void NedSystem::do_remove_staff(NedStaff *staff) {
	GList *lptr;

	if ((lptr = g_list_find(m_staffs, staff)) == NULL) {
		NedResource::Abort("NedSystem::do_remove_staff");
	}
	m_staffs = g_list_delete_link(m_staffs, lptr);
}

void NedSystem::removeLastStaff() {
	GList *lptr;
	NedStaff *staff;


	if (g_list_length(m_staffs) < 2) {
		NedResource::Abort("NedSystem::removeLastStaff");
	}
	lptr = g_list_last(m_staffs);
	staff = (NedStaff *) lptr->data;
	delete staff;
	m_staffs = g_list_delete_link(m_staffs, lptr);
}

void NedSystem::deleteStaff(int staff_number) {
	GList *lptr;
	NedStaff *staff;
	int i;

	if ((lptr = g_list_nth(m_staffs, staff_number)) == NULL) {
		NedResource::Abort("NedSystem::deleteStaff");
	}
	staff = (NedStaff *) lptr->data;
	m_staffs = g_list_delete_link(m_staffs, lptr);
	m_deleted_staffs = g_list_append(m_deleted_staffs, staff);
	for (i = 0, lptr = g_list_first(m_staffs); lptr; i++, lptr = g_list_next(lptr)) {
		((NedStaff *) lptr->data)->setStaffNumber(i);
	}
}

void NedSystem::restoreStaff(int staff_number) {
	GList *lptr;
	NedStaff *staff;
	int pos, i;

	if ((lptr = g_list_last(m_deleted_staffs)) == NULL) {
		NedResource::Abort("NedSystem::restoreStaff");
	}
	staff = (NedStaff *) lptr->data;
	m_deleted_staffs = g_list_delete_link(m_deleted_staffs, lptr);
	pos = staff->getStaffNumber();
	if (pos != staff_number) {
		NedResource::Abort("NedSystem::restoreStaff: pos != staff_number");
	}
	m_staffs = g_list_insert(m_staffs, staff, pos);
	for (i = 0, lptr = g_list_first(m_staffs); lptr; i++, lptr = g_list_next(lptr)) {
		((NedStaff *) lptr->data)->setStaffNumber(i);
	}
}


void NedSystem::shiftStaff(int staff_number, unsigned int position) {
	GList *lptr;
	NedStaff *staff;
	int i;

	if (position > g_list_length(m_staffs)) {
		NedResource::Abort("NedSystem::shiftStaff(1)");
	}
	if ((lptr = g_list_nth(m_staffs, staff_number)) == NULL) {
		NedResource::Abort("NedSystem::shiftStaff(2)");
	}
	staff = (NedStaff *) lptr->data;
	m_staffs = g_list_delete_link(m_staffs, lptr);
	m_staffs = g_list_insert(m_staffs, staff, position);
	for (i = 0, lptr = g_list_first(m_staffs); lptr; i++, lptr = g_list_next(lptr)) {
		((NedStaff *) lptr->data)->setStaffNumber(i);
	}
}


bool NedSystem::reposit(int pass, int measure_number, NedCommandList *command_list /* = NULL */, NedSystem **next_system /* = NULL */) {
	GList *lptr;
	bool changes = FALSE;
	NedSystem *other_system;
	int newnum, i;
	unsigned long long midi_time = 0;
	unsigned int meas_duration;
	bool use_upbeat;
	double indent = (use_upbeat = (m_system_number == 0 && m_page->getPageNumber() == 0)) ? getMainWindow()->getFirstSystemIndent() : 0.0;

	if (next_system != NULL) *next_system = NULL;
	if (measure_number < 0) {
		measure_number = m_measures[0].getMeasureNumber();
	}
	for (i = 0; i < MAX_MEASURES; i++) {
		m_measures[i].midi_start = midi_time;
		meas_duration = getMainWindow()->getMeasureDuration(measure_number);
		m_measures[i].setMeasureNumber(measure_number++, getMainWindow()->m_special_measures /* friend !! */, true);
		midi_time += meas_duration;
		m_measures[i].midi_end = midi_time;
	}
		
	if (command_list == NULL) {
		do_reposit(use_upbeat, indent);
		return FALSE;
	}
	compute_extra_space(indent);
	//do_reposit(indent);
	m_check_line_compression = TRUE;
	if (pass == 0 && m_extra_space < MINIMUM_EXTRA_SPACE && m_measure_count > 1) {
		changes = TRUE;
		m_check_line_compression = FALSE;
		NedClipBoard *board = new NedClipBoard();
		for (lptr = g_list_first(m_staffs); lptr; lptr = g_list_next(lptr)) {
			board->setRefTimeAndDuration(m_measures[m_measure_count-1].midi_start, m_measures[m_measure_count-1].midi_end - m_measures[m_measure_count-1].midi_start);
			((NedStaff *) lptr->data)->collectChordsAndRests(board, m_measures[m_measure_count-1].midi_start,  m_measures[m_measure_count-1].midi_end);
		}
		other_system = m_page->getNextSystem(this, command_list);
		if (board->onlyWholeRests() && getMainWindow()->isLastTouchedSystem(this)) {
			board->setDeleteOnly();
		}
		else {
			other_system->m_isUntouched = FALSE;
			if (next_system != NULL)  *next_system = other_system;
		}
		newnum = m_measures[m_measure_count-1].getMeasureNumber();
		m_check_line_compression = FALSE;
		other_system->collectDestinationVoices(board);
		NedMoveChordsAndRestsCommand *command = new NedMoveChordsAndRestsCommand(board);
		command_list->addCommand(command);
		command_list->addSystemForRepositioning(other_system);
		command->execute();
		compute_extra_space(indent);
		//do_reposit(indent);
		newnum = m_measures[m_measure_count-1].getMeasureNumber() + 1;
		if (other_system != NULL && getMainWindow()->m_start_measure_number_for_renumbering > newnum) {
			getMainWindow()->m_start_system_for_renumbering = other_system;
			getMainWindow()->m_start_measure_number_for_renumbering = newnum;
		}
	}
	if (pass == 1 && m_check_line_compression && m_extra_space > MAXIMUM_EXTRA_SPACE) {
		if ((other_system = m_page->getNextSystem(this)) != NULL) {
			NedClipBoard *board = new NedClipBoard();
			if (other_system->collectFirstMeasure(board)) {
				collectDestinationVoices(board);
				if (board->onlyWholeRests() && getMainWindow()->isLastTouchedSystem(this)) {
					board->setInsertOnly();
				}
				board->execute_reverse();
				compute_extra_space(indent);
				//do_reposit(indent);
				if (m_extra_space < MINIMUM_EXTRA_SPACE) {
					board->unexecute_reverse();
					compute_extra_space(indent);
					//do_reposit(indent);
					other_system->do_reposit(false, 0.0); // the ping-pong elements have bad x position
					m_check_line_compression = FALSE;
				}
				else {
					NedMoveChordsAndRestsReverseCommand *command = new NedMoveChordsAndRestsReverseCommand(board);
					for (lptr = g_list_first(other_system->m_staffs); lptr; lptr = g_list_next(lptr)) {
						((NedStaff *) lptr->data)->resetPositionPointers();
					}
					command_list->addCommand(command);
					command_list->addSystemForRepositioning(other_system);
					if (board->onlyWholeRests() && getMainWindow()->isLastTouchedSystem(this)) {
						//board->setInsertOnly();
					}
					else if (other_system != NULL) {
						if (next_system != NULL) *next_system = other_system;
						if (other_system != NULL) {
							other_system->m_isUntouched = FALSE;
						}
						else {
						}
					}
					compute_extra_space(indent);
					//do_reposit(indent);
					newnum = m_measures[0].getMeasureNumber();
					if (other_system != NULL && getMainWindow()->m_start_measure_number_for_renumbering > newnum) {
						getMainWindow()->m_start_system_for_renumbering = this;
						getMainWindow()->m_start_measure_number_for_renumbering = newnum;
					}
					changes = TRUE;
				}
			}
			else {
				m_check_line_compression = FALSE;
			}
		}
		else {
			m_check_line_compression = FALSE;
			if (command_list != NULL) {
				fill_up(indent, command_list);
			}
		}
	}
	/*
	if (!changes) {
	*/
		do_reposit(use_upbeat, indent);
		/*
	}
	*/
	return changes;
}

bool NedSystem::findTimeOfMeasure(int meas_num, unsigned long long *meas_time) {
	int i;

	for (i = 0; i < m_measure_count; i++) {
		if (m_measures[i].getMeasureNumber() == meas_num) {
			*meas_time = m_measures[i].midi_start;
			return true;
		}
	}
	return false;
}

void NedSystem::renumberMeasures(int *measure_number, GList *special_measures, bool force) {
	int i;

	for (i = 0; i <= m_measure_count; i++) {
		m_measures[i].setMeasureNumber((*measure_number)++, special_measures, force);
	}
	(*measure_number)--;
}

int NedSystem::getNumberOfFirstMeasure() {
	return m_measures[0].getMeasureNumber();
}

int NedSystem::getNumberOfLastMeasure() {
	return m_measures[m_measure_count - 1].getMeasureNumber();
}

unsigned long long NedSystem::getSystemEndTime() {
	return m_measures[m_measure_count - 1].midi_end;
}


double NedSystem::placeStaffs(double staffpos) {
#define SYSTEM_MIN_DIST 0.08
	GList *lptr;
	m_ypos = staffpos;

	double yy = 0.0;
	for (lptr = g_list_first(m_staffs); lptr; lptr = g_list_next(lptr)) {
		yy = ((NedStaff *) lptr->data)->placeStaff(yy);
	}
	return yy + m_ypos + SYSTEM_MIN_DIST;
}

bool NedSystem::collectFirstMeasure(NedClipBoard *board) {
	GList *lptr;
	if (m_measure_count < 2) return FALSE;
	board->setRefTimeAndDuration(m_measures[0].midi_start, m_measures[0].midi_end);
	for (lptr = g_list_first(m_staffs); lptr; lptr = g_list_next(lptr)) {
		((NedStaff *) lptr->data)->collectChordsAndRests(board, m_measures[0].midi_start,  m_measures[0].midi_end);
	}
	return TRUE;
}

void NedSystem::collectSelectionRectangleElements(double xp, NedBbox *sel_rect, GList **sel_group,
		NedSystem *first_selected_system, NedSystem *last_selected_system,
		bool is_first_selected_page, bool is_last_selected_page) {
	GList *lptr;
	NedStaff *firststaff, *laststaff;

	firststaff = (NedStaff *) g_list_first(m_staffs)->data;
	laststaff = (NedStaff *) g_list_last(m_staffs)->data;
	double yp = firststaff->getTopPos() + m_ypos;
	if (is_last_selected_page && sel_rect->y + sel_rect->height < yp) return;
	if (is_first_selected_page && yp + laststaff->getBottomPos() - firststaff->getTopPos() < sel_rect->y) return;
	yp = sel_rect->y - (firststaff->getTopPos() + m_ypos);
	for (lptr = g_list_first(m_staffs); lptr; lptr = g_list_next(lptr)) {
		((NedStaff *) lptr->data)->collectSelectionRectangleElements(xp, yp, sel_rect, sel_group,
		is_first_selected_page && (first_selected_system == this), is_last_selected_page && (last_selected_system == this));
	}
}

void NedSystem::findSelectedSystems(NedBbox *sel_rect, int *number_of_first_selected_staff, 
		int *number_of_last_selected_staff, NedSystem **first_selected_system, NedSystem **last_selected_system) {
	GList *lptr;
	NedStaff *firststaff, *laststaff;

	firststaff = (NedStaff *) g_list_first(m_staffs)->data;
	laststaff = (NedStaff *) g_list_last(m_staffs)->data;
	double yp = firststaff->getTopPos() + m_ypos;
	if (sel_rect->y + sel_rect->height < yp || yp + laststaff->getBottomPos() - firststaff->getTopPos() < sel_rect->y) return;
	if (*first_selected_system == NULL) {
		*first_selected_system = *last_selected_system = this;
	}
	else {
		*last_selected_system = this;
	}
	if (first_selected_system == last_selected_system) {
		*number_of_first_selected_staff = 0;
		*number_of_last_selected_staff = g_list_length(m_staffs) - 1;
	}
	else {
		double yp = sel_rect->y - (firststaff->getTopPos() + m_ypos);
		for (lptr = g_list_first(m_staffs); lptr; lptr = g_list_next(lptr)) {
			((NedStaff *) lptr->data)->findSelectedStaffs(yp, sel_rect, number_of_first_selected_staff, number_of_last_selected_staff);
		}
	}
}

void NedSystem::setAndUpdateClefTypeAndKeySig(int *clef_and_key_array) {
	GList *lptr;
	m_system_start = CLEF_SPACE+ TIME_SIGNATURE_SPACE;
	for (lptr = g_list_first(m_staffs); lptr; lptr = g_list_next(lptr)) {
		((NedStaff *) lptr->data)->setAndUpdateClefTypeAndKeySig(clef_and_key_array);
	}
}

GList *NedSystem::getFirstChordOrRest(int staff_nr, int voice_nr, int lyrics_line, bool lyrics_required) {
	GList *lptr;

	if ((lptr = g_list_nth(m_staffs, staff_nr)) == NULL) {
		NedResource::Abort("getFirstChordOrRest");
	}
	return ((NedStaff *) lptr->data)->getFirstChordOrRest(voice_nr, lyrics_line, lyrics_required);
}

GList *NedSystem::getLastChordOrRest(int staff_nr, int voice_nr, int lyrics_line, bool lyrics_required) {
	GList *lptr;

	if ((lptr = g_list_nth(m_staffs, staff_nr)) == NULL) {
		NedResource::Abort("getFirstChordOrRest");
	}
	return ((NedStaff *) lptr->data)->getLastChordOrRest(voice_nr, lyrics_line, lyrics_required);
}

bool NedSystem::hasTupletConflict(unsigned int meas_duration, GList **elements, int from_staff, int to_staff, unsigned long long start_midi_time) {
	GList *lptr;
	if ((lptr = g_list_nth(m_staffs, to_staff)) == NULL) {
		NedResource::Abort("NedSystem::hasTupletConflict");
	}
	return ((NedStaff *) lptr->data)->hasTupletConflict(meas_duration, elements, from_staff, start_midi_time);
}


void NedSystem::pasteElements(NedCommandList *command_list, GList **elements, int from_staff, int to_staff, unsigned long long start_midi_time) {
	GList *lptr;
	if ((lptr = g_list_nth(m_staffs, to_staff)) == NULL) {
		NedResource::Abort("NedSystem::pasteElements");
	}
	((NedStaff *) lptr->data)->pasteElements(command_list, elements, from_staff, start_midi_time);
}

bool NedSystem::findFromTo(GList *clipboard, NedSystem **min_sys, NedSystem **max_sys) {
	GList *lptr;
	bool found = FALSE;

	for (lptr = g_list_first(m_staffs); lptr; lptr = g_list_next(lptr)) {
		if (((NedStaff *) lptr->data)->findFromTo(clipboard)) {
			found = TRUE;
			if (*min_sys == NULL) {
				*min_sys = *max_sys = this;
			}
			else {
				*max_sys = this;
			}
		}
	}
	return found;
}
bool NedSystem::findStartMeasureLimits(GList *clipboard, unsigned long long *start_midi) {
	GList *lptr;
	bool found = FALSE, fnd;

	for (lptr = g_list_first(m_staffs); lptr; lptr = g_list_next(lptr)) {
		fnd = ((NedStaff *) lptr->data)->findStartMeasureLimits(clipboard, start_midi);
		found = found || fnd;
	}
	return found;
}

bool NedSystem::findEndMeasureLimits(GList *clipboard, unsigned long long *end_midi) {
	GList *lptr;
	bool found = FALSE, fnd;

	for (lptr = g_list_first(m_staffs); lptr; lptr = g_list_next(lptr)) {
		fnd = ((NedStaff *) lptr->data)->findEndMeasureLimits(clipboard, end_midi);
		found = found || fnd;
	}
	return found;
}

void NedSystem::deleteItemsFromTo(NedCommandList *command_list, bool is_first, bool is_last, unsigned long long start_midi, unsigned long long end_midi) {
	GList *lptr;

	for (lptr = g_list_first(m_staffs); lptr; lptr = g_list_next(lptr)) {
		((NedStaff *) lptr->data)->deleteItemsFromTo(command_list, is_first, is_last, start_midi, end_midi);
	}
}

void NedSystem::removeNotesFromTo(NedCommandList *command_list, GList *items, bool is_first, bool is_last) {
	GList *lptr;

	for (lptr = g_list_first(m_staffs); lptr; lptr = g_list_next(lptr)) {
		((NedStaff *) lptr->data)->removeNotesFromTo(command_list, items, is_first, is_last);
	}
}

void NedSystem::insertBlocks(NedCommandList *command_list, int blockcount, unsigned long long midi_time) {
	GList *lptr;

	for (lptr = g_list_first(m_staffs); lptr; lptr = g_list_next(lptr)) {
		((NedStaff *) lptr->data)->insertBlocks(command_list, blockcount, midi_time);
	}
}

void NedSystem::checkForElementsToSplit(NedCommandList *command_list) {
	GList *lptr;

	for (lptr = g_list_first(m_staffs); lptr; lptr = g_list_next(lptr)) {
		((NedStaff *) lptr->data)->checkForElementsToSplit(command_list);
	}
}

void NedSystem::testForPageBackwardTies(NedCommandList *command_list) {
	GList *lptr;

	for (lptr = g_list_first(m_staffs); lptr; lptr = g_list_next(lptr)) {
		((NedStaff *) lptr->data)->testForPageBackwardTies(command_list);
	}
}

void NedSystem::collectDestinationVoices(NedClipBoard *board) {
	GList *lptr;

	board->element_counter = g_list_first(board->m_elements);
	for (lptr = g_list_first(m_staffs); lptr; lptr = g_list_next(lptr)) {
		((NedStaff *) lptr->data)->collectDestinationVoices(board);
	}

}

void NedSystem::resetPositionPointers() {
	GList *lptr;
	for (lptr = g_list_first(m_staffs); lptr; lptr = g_list_next(lptr)) {
		((NedStaff *) lptr->data)->resetPositionPointers();
	}
}


void NedSystem::do_reposit(bool use_upbeat, double indent) {
	GList *lptr;
	unsigned long long midi_time = 0;
	VoiceInfoContainer elements[COUNT_OF_PARALLEL_ELEMENTS], *elptr;
	VoiceInfoContainer positioned_elements[80][VOICE_COUNT];
	int num_elements;
	int i, k;
	double current_position, max_xoffs, max_width;
	NedBbox *bbox;
	int position_counter = 0, pc;
	double shift;
	double sh;
	unsigned long long min_midi_time = (1 >> 30);
	double extra_width_at_end;
	int new_measure_count;
	int number_of_graces = 0;
	int grace_positions[MAX_POSITIONS];
	bool staff_elems_available, grace_available;
#ifdef WITH_WEIGHTS
	unsigned int weight_total = 0, weight;
	unsigned int position_weights[1024];
#endif
	new_measure_count = 0;
	current_position = m_system_start + indent;
	if (m_measures[0].getSpecial() == REPEAT_OPEN || m_measures[0].getSpecial() == REPEAT_OPEN_CLOSE) {
		current_position +=  m_measures[0].getNeededSpaceAfter();
	}

	for (lptr = g_list_first(m_staffs); lptr; lptr = g_list_next(lptr)) {
		((NedStaff *) lptr->data)->resetPositionPointers();
	}
	memset(grace_positions, 0, sizeof(grace_positions));
	do {
		num_elements = 0;
		memset(positioned_elements, 0, sizeof(positioned_elements));
		for (lptr = g_list_first(m_staffs); lptr; lptr = g_list_next(lptr)) {
			((NedStaff *) lptr->data)->collectElmentsAfter(midi_time, elements, &num_elements);
		}
		if (num_elements < 1) break;
		min_midi_time = (1 << 30);
		min_midi_time *= (1 << 30);
		for (i = 0, elptr = elements; i < num_elements; i++, elptr++) {
			if (elptr->element->getMidiTime() < min_midi_time) {
				min_midi_time = elptr->element->getMidiTime();
			}
		}
		midi_time = min_midi_time;
		max_xoffs = -1000.0; max_width = -10000.0;
		m_only_whole_elements = TRUE;
		staff_elems_available = false;
		grace_available = false;
		for (i = 0, elptr = elements; i < num_elements; i++, elptr++) {
			if (elptr->element->getMidiTime() != min_midi_time) continue;
			if (elptr->voice == NULL) {
				staff_elems_available = true; // assuming the staff elements have the widest bbox
			}
			if (elptr->element->getType() == TYPE_GRACE) {
				number_of_graces++;
				grace_available = true;
			}
			bbox = elptr->element->getBBox();
			if (-bbox->x > max_xoffs) max_xoffs = -bbox->x;
			if (elptr->element->getNeededSpace() > max_width) max_width = elptr->element->getNeededSpace(); // returns 0 for KEYSIGS at start
		}
		if (!staff_elems_available) {
			if (midi_time >= m_measures[new_measure_count].midi_start) {
				if (midi_time > 0) {
					current_position +=  m_measures[new_measure_count].getNeededSpaceBefore();
				}
				if (new_measure_count >= MAX_MEASURES) {
					printf("sys = %d, position_counter = %d, midi_time = %llu, new_measure_count = %d\n",
					m_system_number, position_counter, midi_time, new_measure_count);
					NedResource::Abort("NedSystem::do_reposit: new_measure_count >= MAX_MEASURES");
				}
				m_measures[new_measure_count].start = current_position;
#ifdef WITH_WEIGHTS
				position_weights[position_counter] = mylog(MEASURE_WEIGHT);
				weight_total += position_weights[position_counter];
#endif
				m_measures[new_measure_count].position = position_counter++;
				if (midi_time > 0) {
					current_position += m_measures[new_measure_count].getNeededSpaceAfter();
				}
				new_measure_count++;
			}
			for (i = 0, elptr = elements; i < num_elements; i++, elptr++) {
				if (elptr->element->getMidiTime() != min_midi_time) continue;
				if (elptr->element->getLength() != WHOLE_NOTE) m_only_whole_elements = FALSE;
				bbox = elptr->element->getBBox();
				if (-bbox->x > max_xoffs) max_xoffs = -bbox->x;
				if (elptr->element->getNeededSpace() > max_width) max_width = elptr->element->getNeededSpace(); // returns 0 for KEYSIGS at start
			}
			if (m_only_whole_elements) {
				if (max_xoffs < WHOLE_EXTRA_SPACE) max_xoffs = WHOLE_EXTRA_SPACE;
				if (max_width < 2 * WHOLE_EXTRA_SPACE) max_width = 2 * WHOLE_EXTRA_SPACE;
			}
		}
#ifdef WITH_WEIGHTS
		weight = 0;
#endif
		for (i = 0, elptr = elements; i < num_elements; i++, elptr++) {
			if (elptr->element->getMidiTime() != min_midi_time) continue;
			if (elptr->voice == NULL) { // staff element
				elptr->staff->placeCurrentElementAt(current_position + max_xoffs);
				elptr->element->m_position = position_counter;
			}
			else if (elptr->element->getType() == TYPE_GRACE) {
				if (staff_elems_available) continue;
				elptr->voice->placeCurrentElementAt(current_position + max_xoffs);
				elptr->element->m_position = position_counter;
				grace_positions[position_counter] = 1;
				positioned_elements[elptr->staff->getStaffNumber()][elptr->voice->getVoiceNumber()] = *elptr;
			}
			else {
				if (staff_elems_available || grace_available) continue;
				elptr->voice->placeCurrentElementAt(current_position + max_xoffs);
				elptr->element->m_position = position_counter;
				positioned_elements[elptr->staff->getStaffNumber()][elptr->voice->getVoiceNumber()] = *elptr;
			}
#ifdef WITH_WEIGHTS
			if (elptr->element->isRest()) {
				if (!elptr->element->isHidden()) {
					weight += NOTE_4;
				}
			}
			else {
				weight += elptr->element->getDuration();
			}
#endif
		}
#ifdef WITH_WEIGHTS
		position_weights[position_counter] = mylog(weight);
		weight_total += position_weights[position_counter];
#endif
		for (lptr = g_list_first(m_staffs); lptr; lptr = g_list_next(lptr)) {
			if (((NedStaff *) lptr->data)->hasReallyMultipleVoices()) {
				for (i = 0; i < VOICE_COUNT - 1; i++) {
					if (positioned_elements[((NedStaff *) lptr->data)->getStaffNumber()][i].voice == NULL) continue;
					if (positioned_elements[((NedStaff *) lptr->data)->getStaffNumber()][i].element->isRest()) continue;
					for (k = i + 1; k < VOICE_COUNT; k++) {
						if (positioned_elements[((NedStaff *) lptr->data)->getStaffNumber()][k].voice == NULL) continue;
						if (positioned_elements[((NedStaff *) lptr->data)->getStaffNumber()][k].element->isRest()) continue;
						if ((sh = positioned_elements[((NedStaff *) lptr->data)->getStaffNumber()][i].element->compute_microshift(
							positioned_elements[((NedStaff *)lptr->data)->getStaffNumber()][k].element)) > 0) {
							positioned_elements[((NedStaff *)lptr->data)->getStaffNumber()][k].element->shiftX(sh);
							max_width += sh;
						}
					}
				}
				
			}
		}
		position_counter++;
		current_position += max_width;
		if (!staff_elems_available && !grace_available) midi_time++;
	}
	while (num_elements > 0);
	m_measure_count = new_measure_count;
	if (m_measure_count < 1) {
		m_extra_space = m_width - current_position - RIGHT_DIST;
	}
	else {
		m_measures[m_measure_count-1].end = m_width;
		m_measures[m_measure_count].start = m_width;
		if (((m_measures[m_measure_count].getSpecial() & REP_TYPE_MASK) == REPEAT_CLOSE) || ((m_measures[m_measure_count].getSpecial() & REP_TYPE_MASK) == REPEAT_OPEN_CLOSE)) {
			extra_width_at_end = m_measures[m_measure_count-1].getNeededSpaceBefore();
		}
		else {
			extra_width_at_end = 0.0;
		}
		m_extra_space = m_width - current_position - RIGHT_DIST - extra_width_at_end;
	}
#ifndef WITH_WEIGHTS
	m_extra_space /= (position_counter - number_of_graces);
#endif
	if (m_extra_space > 0.0) {
		for (lptr = g_list_first(m_staffs); lptr; lptr = g_list_next(lptr)) {
			((NedStaff *) lptr->data)->resetPositionPointers(FALSE);
		}
		shift = 0.0;
		for (pc = 0; pc < position_counter; pc++) {
			for (lptr = g_list_first(m_staffs); lptr; lptr = g_list_next(lptr)) {
				((NedStaff *) lptr->data)->shiftX(pc, shift);
			}
			for (i = 0; i < m_measure_count; i++) {
				if (pc > 0 && m_measures[i].position == pc) {
					m_measures[i].start += shift;
					if (i > 0) {
						m_measures[i-1].end = m_measures[i].start;
					}
				}
			}
			if (grace_positions[pc]) continue;
#ifdef WITH_WEIGHTS
			shift += (double) position_weights[pc] * m_extra_space / ((double) weight_total);
#else
			shift += m_extra_space;
#endif
		}
	}
	m_measures[m_measure_count].end = m_width - RIGHT_DIST;
	bool stem_force = FALSE;
	for (lptr = g_list_first(m_staffs); lptr; lptr = g_list_next(lptr)) {
		if (((NedStaff *) lptr->data)->hasReallyMultipleVoices()) {
			stem_force = TRUE;
			((NedStaff *) lptr->data)->stemDirDecicion(position_counter);

		}
	}
	for (lptr = g_list_first(m_staffs); lptr; lptr = g_list_next(lptr)) {
		((NedStaff *) lptr->data)->changeDirs();
	}
	if (stem_force) {
		for (lptr = g_list_first(m_staffs); lptr; lptr = g_list_next(lptr)) {
			if (((NedStaff *) lptr->data)->hasReallyMultipleVoices()) {
				((NedStaff *) lptr->data)->placeRests(position_counter);
	
			}
		}
	}
	for (lptr = g_list_first(m_staffs); lptr; lptr = g_list_next(lptr)) {
		((NedStaff *) lptr->data)->searchForBeamGroups(use_upbeat ? getMainWindow()->getUpBeatInverse() : 0);
		((NedStaff *) lptr->data)->computeBeams();
		((NedStaff *) lptr->data)->computeTuplets();
	}
}

void NedSystem::compute_extra_space(double indent) {
	GList *lptr;
	unsigned long long midi_time = 0;
	VoiceInfoContainer elements[COUNT_OF_PARALLEL_ELEMENTS], *elptr;
	VoiceInfoContainer positioned_elements[80][VOICE_COUNT];
	int num_elements;
	int i, k;
	double current_position, max_xoffs, max_width;
	NedBbox *bbox;
	int position_counter = 0;
	double sh;
	unsigned long long min_midi_time = (1 >> 30);
	double extra_width_at_end;
	int new_measure_count;
	int number_of_graces = 0;
	int grace_positions[MAX_POSITIONS];
	bool staff_elems_available, grace_available;
#ifdef WITH_WEIGHTS
	unsigned int weight_total = 0, weight;
	unsigned int position_weights[1024];
#endif
	new_measure_count = 0;
	current_position = m_system_start + indent;
	if (m_measures[0].getSpecial() == REPEAT_OPEN || m_measures[0].getSpecial() == REPEAT_OPEN_CLOSE) {
		current_position +=  m_measures[0].getNeededSpaceAfter();
	}

	for (lptr = g_list_first(m_staffs); lptr; lptr = g_list_next(lptr)) {
		((NedStaff *) lptr->data)->resetPositionPointers();
	}
	memset(grace_positions, 0, sizeof(grace_positions));
	do {
		num_elements = 0;
		memset(positioned_elements, 0, sizeof(positioned_elements));
		for (lptr = g_list_first(m_staffs); lptr; lptr = g_list_next(lptr)) {
			((NedStaff *) lptr->data)->collectElmentsAfter(midi_time, elements, &num_elements);
		}
		if (num_elements < 1) break;
		min_midi_time = (1 << 30);
		min_midi_time *= (1 << 30);
		for (i = 0, elptr = elements; i < num_elements; i++, elptr++) {
			if (elptr->element->getMidiTime() < min_midi_time) {
				min_midi_time = elptr->element->getMidiTime();
			}
		}
		midi_time = min_midi_time;
		max_xoffs = -1000.0; max_width = -10000.0;
		m_only_whole_elements = TRUE;
		staff_elems_available = false;
		grace_available = false;
		for (i = 0, elptr = elements; i < num_elements; i++, elptr++) {
			if (elptr->element->getMidiTime() != min_midi_time) continue;
			if (elptr->voice == NULL) {
				staff_elems_available = true; // assuming the staff elements have the widest bbox
			}
			if (elptr->element->getType() == TYPE_GRACE) {
				grace_available = true;
				grace_available = true;
			}
			bbox = elptr->element->getBBox();
			if (-bbox->x > max_xoffs) max_xoffs = -bbox->x;
			if (elptr->element->getNeededSpace() > max_width) max_width = elptr->element->getNeededSpace(); // returns 0 for KEYSIGS at start
		}
		if (!staff_elems_available) {
			if (midi_time >= m_measures[new_measure_count].midi_start) {
				if (midi_time > 0) {
					current_position +=  m_measures[new_measure_count].getNeededSpaceBefore();
				}
				if (new_measure_count >= MAX_MEASURES) {
					printf("sys = %d, position_counter = %d, midi_time = %llu, new_measure_count = %d\n",
					m_system_number, position_counter, midi_time, new_measure_count);
					NedResource::Abort("NedSystem::compute_extra_space: new_measure_count >= MAX_MEASURES");
				}
				m_measures[new_measure_count].start = current_position;
#ifdef WITH_WEIGHTS
				position_weights[position_counter] = mylog(MEASURE_WEIGHT);
				weight_total += position_weights[position_counter];
#endif
				m_measures[new_measure_count].position = position_counter++;
				if (midi_time > 0) {
					current_position += m_measures[new_measure_count].getNeededSpaceAfter();
				}
				new_measure_count++;
			}
			for (i = 0, elptr = elements; i < num_elements; i++, elptr++) {
				if (elptr->element->getMidiTime() != min_midi_time) continue;
				if (elptr->element->getLength() != WHOLE_NOTE) m_only_whole_elements = FALSE;
				bbox = elptr->element->getBBox();
				if (-bbox->x > max_xoffs) max_xoffs = -bbox->x;
				if (elptr->element->getNeededSpace() > max_width) max_width = elptr->element->getNeededSpace(); // returns 0 for KEYSIGS at start
			}
			if (m_only_whole_elements) {
				if (max_xoffs < WHOLE_EXTRA_SPACE) max_xoffs = WHOLE_EXTRA_SPACE;
				if (max_width < 2 * WHOLE_EXTRA_SPACE) max_width = 2 * WHOLE_EXTRA_SPACE;
			}
		}
#ifdef WITH_WEIGHTS
		weight = 0;
#endif
		for (i = 0, elptr = elements; i < num_elements; i++, elptr++) {
			if (elptr->element->getMidiTime() != min_midi_time) continue;
			if (elptr->voice == NULL) { // staff element
				elptr->staff->incrElementPointer();
				elptr->element->m_position = position_counter;
			}
			else if (elptr->element->getType() == TYPE_GRACE) {
				if (staff_elems_available) continue;
				elptr->voice->incrElementPointer();
				elptr->element->m_position = position_counter;
				grace_positions[position_counter] = 1;
				positioned_elements[elptr->staff->getStaffNumber()][elptr->voice->getVoiceNumber()] = *elptr;
			}
			else {
				if (staff_elems_available || grace_available) continue;
				elptr->voice->incrElementPointer();
				elptr->element->m_position = position_counter;
				positioned_elements[elptr->staff->getStaffNumber()][elptr->voice->getVoiceNumber()] = *elptr;
			}
#ifdef WITH_WEIGHTS
			if (elptr->element->isRest()) {
				if (!elptr->element->isHidden()) {
					weight += NOTE_4;
				}
			}
			else {
				weight += elptr->element->getDuration();
			}
#endif
		}
#ifdef WITH_WEIGHTS
		position_weights[position_counter] = mylog(weight);
		weight_total += position_weights[position_counter];
#endif
		for (lptr = g_list_first(m_staffs); lptr; lptr = g_list_next(lptr)) {
			if (((NedStaff *) lptr->data)->hasReallyMultipleVoices()) {
				for (i = 0; i < VOICE_COUNT - 1; i++) {
					if (positioned_elements[((NedStaff *) lptr->data)->getStaffNumber()][i].voice == NULL) continue;
					if (positioned_elements[((NedStaff *) lptr->data)->getStaffNumber()][i].element->isRest()) continue;
					for (k = i + 1; k < VOICE_COUNT; k++) {
						if (positioned_elements[((NedStaff *) lptr->data)->getStaffNumber()][k].voice == NULL) continue;
						if (positioned_elements[((NedStaff *) lptr->data)->getStaffNumber()][k].element->isRest()) continue;
						if ((sh = positioned_elements[((NedStaff *) lptr->data)->getStaffNumber()][i].element->compute_microshift(
							positioned_elements[((NedStaff *)lptr->data)->getStaffNumber()][k].element)) > 0) {
							positioned_elements[((NedStaff *)lptr->data)->getStaffNumber()][k].element->shiftX(sh);
							max_width += sh;
						}
					}
				}
				
			}
		}
		position_counter++;
		current_position += max_width;
		if (!staff_elems_available && !grace_available) midi_time++;
	}
	while (num_elements > 0);
	m_measure_count = new_measure_count;
	if (m_measure_count < 1) {
		m_extra_space = m_width - current_position - RIGHT_DIST;
	}
	else {
		m_measures[m_measure_count-1].end = m_width;
		m_measures[m_measure_count].start = m_width;
		if (((m_measures[m_measure_count].getSpecial() & REP_TYPE_MASK) == REPEAT_CLOSE) || ((m_measures[m_measure_count].getSpecial() & REP_TYPE_MASK) == REPEAT_OPEN_CLOSE)) {
			extra_width_at_end = m_measures[m_measure_count-1].getNeededSpaceBefore();
		}
		else {
			extra_width_at_end = 0.0;
		}
		m_extra_space = m_width - current_position - RIGHT_DIST - extra_width_at_end;
	}
#ifndef WITH_WEIGHTS
	m_extra_space /= (position_counter - number_of_graces);
#endif
}

unsigned int NedSystem::getWholeMidiLength() {
	return m_measure_count * getMainWindow()->getNumerator() * WHOLE_NOTE / getMainWindow()->getDenominator();
}

#ifdef WITH_WEIGHTS
unsigned int NedSystem::mylog(unsigned int x) {
	x /= FACTOR;
	int k = 0;
	while (x) {
		x >>= 1;
		k++;
	}
	return k;
}
#endif


NedMeasure *NedSystem::getMeasure(unsigned int midi_time) {
	int i;
	for (i = 0; i < m_measure_count; i++) {
		if (m_measures[i].midi_start <= midi_time && m_measures[i].midi_end > midi_time) {
			return (&(m_measures[i]));
		}
	}
#ifdef AAA
	printf("pag: %d, sys. %d, midi_time = %u , m_measures[m_measure_count(%d)].midi_start = %llu, m_measures[m_measure_count(%d)].midi_end = %llu, NOTE_4 = %u\n",
	m_page->getPageNumber(), m_system_number,
	midi_time, m_measure_count, m_measures[m_measure_count].midi_start, m_measure_count, m_measures[m_measure_count].midi_end, NOTE_4); fflush(stdout);
	if (m_measure_count >= 0) {
		printf("m_measures[m_measure_count - 1(%d)].midi_start = %llu,  m_measures[m_measure_count - 1(%d)].midi_end = %llu\n",
		m_measure_count-1, m_measures[m_measure_count-1].midi_start, m_measure_count-1, m_measures[m_measure_count-1].midi_end); fflush(stdout);
	}
	NedResource::Abort("NedSystem::getMeasure");
#endif
	return NULL;
}

int NedSystem::getNumberOfStaffs() {
	return g_list_length(m_staffs);
}


void NedSystem::prepareReplay() {
	GList *lptr;

	for (lptr = g_list_first(m_staffs); lptr; lptr = g_list_next(lptr)) {
		((NedStaff *) lptr->data)->prepareReplay();
	}
}

void NedSystem::setWidth(double w) {
	GList *lptr;

	m_width = w;

	for (lptr = g_list_first(m_staffs); lptr; lptr = g_list_next(lptr)) {
		((NedStaff *) lptr->data)->setWidth(m_width);
	}
}

void NedSystem::saveSystem(FILE *fp) {
	int i;
	GList *lptr;
	fprintf(fp, "SYSTEM %d: %d MEASURES\n", m_system_number, m_measure_count);
	for (i = 0; i < m_measure_count; i++) {
		fprintf(fp, "MEASURE: %d: %llu %llu\n", i, m_measures[i].midi_start, m_measures[i].midi_end);
	}
	for (lptr = g_list_first(m_staffs); lptr; lptr = g_list_next(lptr)) {
		((NedStaff *) lptr->data)->saveStaff(fp);
	}
}

void NedSystem::restoreSystem(FILE *fp) {
	char buffer[128];
	int i, measnum, staffnum;
	NedStaff *staff;
	double staff_pos = 0;

	if (!NedResource::readInt(fp, &m_measure_count)) {
		NedResource::m_error_message = "Measure count expected";
		return;
	}
	if (m_measure_count < 0 || m_measure_count >= MAX_MEASURES) {
		NedResource::m_error_message = "Bad measure count";
	}

	if (!NedResource::readWord(fp, buffer) || strcmp(buffer, "MEASURES")) {
		NedResource::m_error_message = "MEASURES expected";
		return;
	}
	for (i = 0; NedResource::m_error_message == NULL && i < m_measure_count; i++) {
		if (!NedResource::readWord(fp, buffer) || strcmp(buffer, "MEASURE")) {
			NedResource::m_error_message = "MEASURE expected";
			return;
		}
		if (!NedResource::readWord(fp, buffer) || strcmp(buffer, ":")) {
			NedResource::m_error_message = ": expected";
			return;
		}
		if (!NedResource::readInt(fp, &measnum)) {
			NedResource::m_error_message = "MeasureNumber ected";
			return;
		}
		if (i != measnum) {
			NedResource::m_error_message = "Bad measure number";
			return;
		}
		if (!NedResource::readWord(fp, buffer) || strcmp(buffer, ":")) {
			NedResource::m_error_message = ": expected";
			return;
		}
		if (!NedResource::readLong(fp, (unsigned long long *) &(m_measures[i].midi_start))) {
			NedResource::m_error_message = "Midi start value expected";
			return;
		}
		if (!NedResource::readLong(fp, (unsigned long long *) &(m_measures[i].midi_end))) {
			NedResource::m_error_message = "Midi end value expected";
			return;
		}
	}
	for (i = 0; NedResource::m_error_message == NULL && i < getMainWindow()->getPreStaffCount(); i++) {
		if (!NedResource::readWord(fp, buffer) || strcmp(buffer, "STAFF")) {
			NedResource::m_error_message = "STAFF expected(2)";
			return;
		}
		if (!NedResource::readInt(fp, &staffnum)) {
			NedResource::m_error_message = "STAFF number expected";
			return;
		}
		if (staffnum != i) {
			NedResource::m_error_message = "Bad STAFF number";
			return;
		}
		staff = new NedStaff(this, staff_pos, m_width, i, FALSE);

		staff->restoreStaff(fp);
		m_staffs = g_list_append(m_staffs, staff);

		staff_pos += staff->getHeight() + DEFAULT_SYSTEM_DIST;
	}
}


