//=============================================================================
//  MusE Score
//  Linux Music Score Editor
//  $Id: slur.h,v 1.27 2006/03/02 17:08:43 wschweer Exp $
//
//  Copyright (C) 2002-2007 Werner Schweer and others
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License version 2.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//=============================================================================

#ifndef __SLUR_H__
#define __SLUR_H__

#include "globals.h"
#include "element.h"

class Note;
class System;
class SlurTie;
class Score;

struct UP {
      QPointF p;            // layout position relative to pos()
      QPointF off;          // user offset in spatium units
      QPointF pos() const { return p + off * _spatium; }
      };

//---------------------------------------------------------
//   SlurSegment
//---------------------------------------------------------

class SlurSegment : public Element {
      Q_DECLARE_TR_FUNCTIONS(SlurSegment)

      struct UP ups[4];
      QPainterPath path;
      qreal bow;
      SegmentType _segmentType;
      System* _system;

      virtual QRectF bbox() const;
      void updatePath();

   public:
      SlurSegment(Score*);
      SlurSegment(const SlurSegment&);
      virtual SlurSegment* clone() const { return new SlurSegment(*this); }
      virtual ElementType type() const { return SLUR_SEGMENT; }
      virtual bool isMovable() const { return true; }

      void layout(ScoreLayout*, const QPointF& p1, const QPointF& p2, qreal bow);
      virtual QPainterPath shape() const;
      virtual void draw(QPainter&) const;

      virtual bool startEdit(const QPointF&);
      virtual void editDrag(int, const QPointF&, const QPointF&);
      virtual bool edit(int, QKeyEvent*);
      virtual void updateGrips(int*, QRectF*) const;
      virtual QPointF gripAnchor(int grip) const;

      virtual void move(qreal xd, qreal yd) { move(QPointF(xd, yd)); }
      virtual void move(const QPointF& s);

      virtual bool genPropertyMenu(QMenu*) const;

      SlurTie* slurTie() const     { return (SlurTie*)parent(); }

      void write(Xml& xml, int no) const;
      void read(QDomElement);
      void dump() const;
      void setSegmentType(SegmentType s)  { _segmentType = s;  }
      void setSystem(System* s)           { _system = s;       }
      };

//---------------------------------------------------------
//   SlurTie
//---------------------------------------------------------

class SlurTie : public Element {
   protected:
      bool up;
      QList<SlurSegment*> segments;
      Direction _slurDirection;

   public:
      SlurTie(Score*);
      SlurTie(const SlurTie&);
      ~SlurTie();

      virtual ElementType type() const = 0;
      bool isUp() const                  { return up; }
      void setUp(bool val)               { up = val;  }
      Direction slurDirection() const    { return _slurDirection; }
      void setSlurDirection(Direction d) { _slurDirection = d; }

      virtual void layout2(ScoreLayout*, const QPointF, int, struct UP&)  {}
      virtual void setSelected(bool f);
      virtual bool contains(const QPointF&) const { return false; }  // not selectable

      QList<SlurSegment*>* elements()             { return &segments;      }
      virtual void add(Element* s);
      virtual void remove(Element* s);
      virtual void change(Element* o, Element* n);

      void writeProperties(Xml& xml) const;
      bool readProperties(QDomElement);
      QPointF slurPos(int tick, int track, System*& s);
      virtual void collectElements(QList<const Element*>& el) const;
      };

//---------------------------------------------------------
//   Slur
//---------------------------------------------------------

class Slur : public SlurTie {
      int _track2;
      int _tick2;

   public:
      Slur(Score*);
      ~Slur();
      virtual Slur* clone() const      { return new Slur(*this); }
      virtual ElementType type() const { return SLUR; }
      virtual void write(Xml& xml) const;
      virtual void read(QDomElement);
      virtual void layout(ScoreLayout*);
      virtual QRectF bbox() const;

      int tick2() const       { return _tick2; }
      void setTick2(int val);
      int track2() const      { return _track2; }
      void setTrack2(int val) { _track2 = val; }

      void setStart(int t, int track);
      void setEnd(int t,   int track);
      bool startsAt(int t, int track);
      bool endsAt(int t,   int track);
      };

//---------------------------------------------------------
//   Tie
//---------------------------------------------------------

class Tie : public SlurTie {
      Note* _startNote; // parent
      Note* _endNote;

   public:
      Tie(Score*);
      virtual Tie* clone() const    { return new Tie(*this); }
      virtual ElementType type() const { return TIE; }
      void setStartNote(Note* note);
      void setEndNote(Note* note)   { _endNote = note; }
      Note* startNote() const       { return _startNote; }
      Note* endNote() const         { return _endNote; }
      virtual void write(Xml& xml) const;
      virtual void read(QDomElement);
      virtual void layout(ScoreLayout*);
      };

#endif
