#ifndef DICOMFILE_H_
#define DICOMFILE_H_

#include <fstream>
#include <vector>
#include <map>
#include <sstream>
#include <bitset>

#include <wx/defs.h>
#include <wx/log.h>

namespace jcs {

struct DicomTag;
struct DicomElement;
class Dictionary;

class DicomFile {

private:
	typedef std::string string;

public :

	// Constructor/destructor
	DicomFile(const char* pFileName);

	~DicomFile();

	// Export file
	void DataDump(const char *pFilename);
	std::vector <DicomElement> DataDump();

	int FindInSequence(const std::string& seq, DicomElement& d, 
		std::vector<std::string>& v);

	int FindInSequence(const std::string& seq, const char* str, 
		std::vector<std::string>& v);

	// Find dicom element, convert to value of type T
	template<class T> int
	Find(DicomElement d, T& rValue);

	// Find element from dicom dict. using string id
	template<class T> int
	Find(const string& s, T& rValue);

	// Retrieve data_length bytes of pixel data
	// It's up to the caller to send correct type and data_length!
	template<class T> int
	PixelData(std::vector<T>& data, std::streamsize data_length, bool rescale = false, int frame = 0);

	// Or read directly into buffer...user must alloc/free mem
	template<class T> int
	PixelData(T* data, std::streamsize data_length);

	// These functions are for Syngo files only
	bool HasSiemensMrHeader();

	int GetCSAImageHeader(string& header);
	int GetCSASeriesHeader(string& header);

	int	ReadCSAImageHeader(const string& tag, string& value);
	int	ReadCSAImageHeader(const string& tag, std::vector<string>& value);

	int	ReadCSASeriesHeader(const string& tag, string& value);
	int	ReadCSASeriesHeader(const string& tag, std::vector<string>& value);

	int ReadCSAHeader(const string& header, const string& tag, string& value);
	int ReadCSAHeader(const string& header, const string& tag, std::vector<string>& value);

	bool IsBigEndian() const { return (mByteOrder == dBIG_ENDIAN); }
	bool IsOk(); 

	int GetHeaderSize();
	int GetFileSize();

	const char* ErrorMessage() const;

private :
	
	std::ifstream mInputFile;
	std::streampos mBeginning;

	enum ErrorCodes {
		NO_BYTES,
		NO_DICM,
		METAFILE_ERROR,
		TRANSFER_SYNTAX,
		NO_UID,
		NO_SERIES_UID,
		NO_INSTANCE,
		NO_IMAGE,
		NUM_CODES
	};

	std::bitset<NUM_CODES> mErrors;

	Dictionary* mDicom;
	Dictionary* mSpecial;
	
	const string mFileName;

	enum dicomEndianType { dLITTLE_ENDIAN, dBIG_ENDIAN } mByteOrder;
	enum VrSyntaxType { IMPLICIT_VR,  EXPLICIT_VR } mVrSyntax;

	void mCheckOk();
	int mSkipPreamble();
	int mReadMetafile();	

	void mSetTransferSyntax(VrSyntaxType v, dicomEndianType b);

	int mFindElement(const DicomTag& rTarget, DicomElement& rMatch, 
		std::istream& input);

	std::vector<DicomElement> mReadNextElementAll();
	void mReadNextElement(DicomElement& e, std::istream& input);
	void mReadElementTag(DicomElement& e, std::istream& input);
	void mReadValueRepresentation(DicomElement& e, std::istream& input);
	void mReadValueLength(DicomElement& e, std::istream& input);

	template<class T> int
		mReadValue(DicomElement d, T& rValue, std::istream& input);

	// Specialization for strings
	int mReadValue(DicomElement d, string& rValue, std::istream& input);

	typedef std::vector<string> (*read_ptr)(std::istream&,	std::streamsize);

	std::map<string, read_ptr> ReadMap;

	void mInitializeMap();

        template<class T> T ByteSwap(T value_to_swap);
        wxUint16 ByteSwap(wxUint16 value_to_swap);
        wxInt16 ByteSwap(wxInt16 value_to_swap);
        wxUint32 ByteSwap(wxUint32 value_to_swap);
        wxInt32 ByteSwap(wxInt32 value_to_swap);

	template<class T> int
	FindInStream(std::istream& input, DicomElement d, T& rValue);

	int FindElement(DicomElement& d, std::istream& input);

	std::istream::pos_type GetPixelDataStart();
	std::istream::pos_type pixelDataStart;
	bool foundPixelDataStart;

};

}

#include "DicomFile.txx"

#endif
