#include <wx/filename.h>

#include <string>
#include <sstream>
#include <iostream>
#include <algorithm>

#include "StringConvert.h"
#include "DicomFile.h"
#include "SeriesHandler.h"
#include "HandlerFactory.h"
#include "OutputFactory.h"
#include "Converter.h"

// for new stuff
#include "OutputterBase.h"

using namespace std;
using namespace jcs;

template <class T>
struct GetFirst : std::unary_function<T, typename T::first_type> {
	typename T::first_type operator()(const T& t) const {
	return t.first;
	}
};

static void
DeleteHandler (std::pair<string, SeriesHandler*> pair)
{
	delete pair.second;
}
	
Converter::Converter(int type)
{
	mpOutputter = OutputFactory::CreateNewOutputter(type);
}

Converter::~Converter()
{
	std::for_each(mHandlerMap.begin(), mHandlerMap.end(),
		DeleteHandler);

	delete mpOutputter;
}

void
Converter::SetOutput(int type)
{
	delete mpOutputter;
	mpOutputter = OutputFactory::CreateNewOutputter(type);
	UpdateAllOutput();
}

int
Converter::AddFile(const wxString& path, const wxString& match)
{

	Message fileMessage;
	fileMessage.push_back(path);
	
	wxString message;
	if (mAddFile(path, match, message))
		fileMessage.push_back(_T("Added"));
	else {
		fileMessage.push_back(_T("Not added"));
		fileMessage.push_back(message);
	}

	messages.push_back(fileMessage);
	
	return 1;

}

bool
Converter::mAddFile(const wxString& path, const wxString& match, wxString& message)
{
	wxString name;
	wxString ext;
	wxFileName::SplitPath(path, NULL, &name, &ext);
	if (name == _T("DICOMDIR")) {
		message = _T("We don't convert DICOMDIR files.");
		return false;
	}

	DicomFile input(path.mb_str(wxConvLocal));
	if (!input.IsOk()) {
		message = wxString(input.ErrorMessage(), wxConvLocal);
		return false;
	}

	if (!match.empty()) {
		string series_description;
		input.Find("SeriesDescription", series_description);
		if (series_description.find(match.mb_str(wxConvLocal)) == string::npos) {
			message = _T("Series description doesn't match ");
			message.append(match);
			return false;
		}
	}



	string series_uid;
	input.Find("SeriesInstanceUid", series_uid);

	// Fix problems with termination
	string temp(series_uid.c_str());
	series_uid.clear();
	series_uid = temp;

	mSeriesToUpdate.insert(series_uid);
		
	if (!mHandlerMap.count(series_uid))
		mHandlerMap[series_uid] = HandlerFactory::CreateHandler(path.mb_str(wxConvLocal));

	if (mHandlerMap[series_uid]->AddFile(path.mb_str(wxConvLocal)))
		return true;
	else {
		message = _T("File already in list.");
		return false;
	}
}
/*
void
Converter::ResetMessageBox()
{
	mMessages.clear();
}

void
Converter::ShowMessageBox()
{
	// don't bother if no messages
	if (mMessages.empty()) return;

	wxWindowDisabler disableAll;
	MessageListDlg lmd("Finished adding files");
	Message m;
	m.push_back("File name");
	m.push_back("Status");
	m.push_back("Information");
	lmd.SetColumns(m);
	lmd.AddMessages(mMessages);
	lmd.ShowModal();
	ResetMessageBox();
}
*/
void
Converter::UpdateOutput()
{
	set<string>::iterator it = mSeriesToUpdate.begin();
	set<string>::iterator it_end = mSeriesToUpdate.end();
	while (it != it_end) {
		mpOutputter->UpdateOutputForSeries(mHandlerMap[*it]);
		++it;
	}
	mSeriesToUpdate.clear();

}

void
Converter::UpdateAllOutput()
{
	HandlerMap::iterator it = mHandlerMap.begin();
	HandlerMap::iterator it_end = mHandlerMap.end();
	while (it != it_end) {
		mpOutputter->UpdateOutputForSeries(it->second); 
		++it;
	}
}

void
Converter::UpdateOutput(const string& seriesUid)
{
	mpOutputter->UpdateOutputForSeries(mHandlerMap[seriesUid]); 
}

void 
Converter::RemoveSeries(const std::string& seriesUid)
{
	delete mHandlerMap[seriesUid];
	mHandlerMap.erase(seriesUid);
	mpOutputter->RemoveSeries(seriesUid);

	
}


const vector<string> 
Converter::GetStringInfo(const string& s) const
{
	return (*mHandlerMap.find(s)).second->GetStringInfo();
}
/*
const DicomSeries&
Converter::GetSeries(const std::string& seriesUid) const
{
	return (*mHandlerMap.find(seriesUid)).second->GetSeries();
}
*/
/*
const SeriesHandler* 
Converter::GetHandler(const string& uid) const
{
	return (*mHandlerMap.find(uid)).second;
}

bool
Converter::HandlerExists(const string& uid) const
{
	return (mHandlerMap.count(uid) > 0) ? 1 : 0;
}
*/

const vector<string>
Converter::GetSeriesList() const
{
	vector<string> v;
	std::transform(
		mHandlerMap.begin(), mHandlerMap.end(),
		std::back_inserter(v),
		GetFirst<HandlerMap::value_type>()
	);

	return v;
}


void
Converter::ConvertAll()
{
	mCancel = false;

	HandlerMap::iterator it = mHandlerMap.begin();
	HandlerMap::iterator it_end = mHandlerMap.end();
	while (it != it_end) {
		if (mCancel) break;
		mpOutputter->ConvertSeries(it->second);
		++it;
	}

/*	mpOutputter->Overwrite(false);
	map<string, string>::iterator it = dirMap.begin();
	map<string, string>::iterator map_end = dirMap.end();
	while (it != map_end) {
		if (mCancel) break;
		mpOutputter->ConvertSeries(*it);
		++it;
	}
	*/
}


/*
int
Converter::ShowOptionsDlg(int type)
{
	int rebuild = >ShowOptionsDlg(type, mpOutputter);

	if (rebuild == 1) UpdateAllOutput();

	return rebuild;
}
*/
