#ifndef SERIES_HANDLER_TXX
#define SERIES_HANDLER_TXX

#include <string>
#include <vector>
#include <wx/log.h>
#include <wx/utils.h>
#include <wx/app.h>

#include "Dictionary.h"
#include "Volume.h"
#include "StringConvert.h"
#include "SeriesHandler.h"

namespace jcs {

  ///
  /**
   */
  template <class T> void
  SeriesHandler::mGetVolumes(std::map<VolId, Volume<T> >& v)
  { 
    v.clear();
    string uid = GetSeriesUid();
    wxString message = _("Reading series ") + wxString(uid.c_str(), wxConvLocal);

    if (verbose) wxLogStatus(message);

    int image_columns, image_rows;
    Find("ImageColumns", image_columns);
    Find("ImageRows", image_rows);

    int file_number = 0;
    FileMapType::const_iterator it = mFileMap.begin();
    FileMapType::const_iterator end = mFileMap.end();

    std::map<VolId, double> minDistMap;
    std::map<VolId, string> ippMap;

    int samples_per_pixel;
    Find("SamplesPerPixel", samples_per_pixel);

    // For each file
    while (it != end) {
      std::vector<T> data;
      DicomFile d_file((*it).first.c_str());

      // For each frame/slice in file
      VolListType::const_iterator vit = it->second.begin();
      for (unsigned int frame = 0; frame < it->second.size(); ++frame) {
	if (v.find(*vit) == v.end()) {
	  v[*vit] = Volume<T> (image_rows, image_columns);
	}
	std::vector<double> rotation = GetRotationMatrix(*vit);
		  
	string ipp;
	ipp = GetImagePositionPatient(d_file, frame);
		  
	double dist = 0;
	for (int i = 0; i < 3; ++i) {
	  dist += rotation[i+6] * stof(ipp, i);
	}

	double slope = 1;
	double intercept = 0;
	// If rescale, retrieved slope and intercept will cause PixelData to rescale.
	// Otherwise, slope and intercept values cause PixelData to avoid rescale.
	if (rescale) GetRescaleSlopeAndIntercept(d_file, slope, intercept, frame);
 
	d_file.PixelData<T>(data,
			    image_rows * image_columns * sizeof(T) * samples_per_pixel,
			    slope,
			    intercept,
			    frame);
	v[*vit].AddSlice(dist, data, (*it).first);

	if (minDistMap.find(*vit) == minDistMap.end()) {
	  minDistMap[*vit] = dist;
	  ippMap[*vit] = ipp;
	}

	if (dist < minDistMap[*vit]) {
	  minDistMap[*vit] = dist;
	  ippMap[*vit] = ipp;
	}
	++vit;
	wxTheApp->Yield();
      }

      ++it;
    }

    std::map<VolId, string>::iterator ippIt = ippMap.begin();
    while (ippIt != ippMap.end()) {
      std::vector<double> ipp;
      for (int i = 0; i < 3; ++i)
	ipp.push_back(stof(ippIt->second, i));
      position[ippIt->first] = ipp;
      ++ippIt;
    }

  }

 
  ///
  /**
   */
  template <class T> void
  NumarisMosaicHandler::mGetVolumes(std::map<VolId, Volume<T> >& v)
  {
    std::string s = GetSeriesUid();
    wxString message = _("Reading series ") + wxString(s.c_str(), wxConvLocal);
    if (verbose) wxLogStatus(message);

    v.clear();

    std::vector<double> slice_order = GetSliceOrder();

    FileMapType::const_iterator it = mFileMap.begin();
    FileMapType::const_iterator end = mFileMap.end();

    int image_rows, image_columns;
    Find("ImageColumns", image_columns);
    Find("ImageRows", image_rows);
    int n_slices = GetNumberOfSlices();
    int mosaic_rows = GetRows();
    int mosaic_columns = GetColumns();

    for (;it != end; it++) {
      v[(*it).second.front()] = Volume<T>(mosaic_rows, mosaic_columns);
      std::vector<T> data;
      DicomFile d_file((*it).first.c_str());

      double slope = 1;
      double intercept = 0;
      if (rescale) GetRescaleSlopeAndIntercept(d_file, slope, intercept);

      d_file.PixelData<T>(data, image_rows * image_columns * sizeof(T), slope, intercept);

      int number_of_columns = image_columns/mosaic_columns;

      for(int tile = 0; tile < n_slices; ++tile) {

	int tile_column = tile % number_of_columns;
	int tile_row = tile / number_of_columns;
	typename std::vector<T>::iterator tile_begin = data.begin() + 
	  tile_column * mosaic_columns + 
	  tile_row * mosaic_rows * image_columns;

	assert(tile_begin < data.end());
	std::vector<T> slice;
	slice.reserve(mosaic_rows * mosaic_columns);
		
	for (int row_no = 0; row_no < mosaic_rows; ++row_no) {
		
	  typename std::vector<T>::iterator row_begin = tile_begin + row_no * image_columns;
	  typename std::vector<T>::iterator row_end = row_begin + mosaic_columns;
	  assert(row_end <= data.end());
	  slice.insert(slice.end(), row_begin, row_end);

	}
	v[(*it).second.front()].AddSlice(slice_order[tile], slice, (*it).first);
      }
      assert(v[(*it).second.front()].size() == n_slices);
    }
  }


  ///
  /**
   */
  template <class T> void
  SyngoMosaicHandler::mGetVolumes(std::map<VolId, Volume<T> >& v)
  { 

    v.clear();

    string uid = GetSeriesUid();
    wxString message = _("Reading series ") + wxString(uid.c_str(), wxConvLocal);
    if (verbose) wxLogStatus(message);

    FileMapType::const_iterator it = mFileMap.begin();
    FileMapType::const_iterator end = mFileMap.end();

    int image_rows, image_columns;
    Find("ImageColumns", image_columns);
    Find("ImageRows", image_rows);
    int n_slices = GetNumberOfSlices();
    int mosaic_rows = GetRows();
    int mosaic_columns = GetColumns();

    std::vector<double> voxel_size = GetVoxelSize();
    std::vector<double> offset(2, 0);
    offset[0] = -0.5*GetColumns()*voxel_size[0];
    offset[1] = -0.5*GetRows()*voxel_size[1];

    std::map<VolId, double> minDistMap;

    while (it != end) {
      v[(*it).second.front()] = Volume<T>(mosaic_rows, mosaic_columns);
      std::vector<T> data;
      DicomFile d_file((*it).first.c_str());

      double slope = 1;
      double intercept = 0;
      if (rescale) GetRescaleSlopeAndIntercept(d_file, slope, intercept);

      d_file.PixelData<T>(data, image_rows* image_columns * sizeof(T), slope, intercept);

      std::vector<double> rotation = GetRotationMatrix(it->second.front());

      int number_of_columns = image_columns/mosaic_columns;

      std::string str;
      bool BottomUpMosaic = true;	  
      if (ReadCSAImageHeader("ProtocolSliceNumber", str))
	if (stoi(str) != 0) BottomUpMosaic = false; // we'll assume top down and not some weird order
	  
      std::string protocol;
      int err = ReadCSASeriesHeader("MrProtocol", protocol);
      if (err == -1) err = ReadCSASeriesHeader("MrPhoenixProtocol", protocol);
	
      if (err <= 0) return; 

      for(int tile = 0; tile < n_slices; ++tile) {

		int tile_column = tile % number_of_columns;
		int tile_row = tile / number_of_columns;
		typename std::vector<T>::iterator tile_begin = data.begin() + 
		tile_column * mosaic_columns + 
		tile_row * mosaic_rows * image_columns;

		assert(tile_begin < data.end());

		std::vector<T> slice;
		slice.reserve(mosaic_rows * mosaic_columns);
		
		for (int row_no = 0; row_no < mosaic_rows; ++row_no) {
		
			typename std::vector<T>::iterator row_begin = tile_begin + row_no * image_columns;
			typename std::vector<T>::iterator row_end = row_begin + mosaic_columns;
			assert(row_end <= data.end());
			slice.insert(slice.end(), row_begin, row_end);

		}
			
		// get slice distance
		int slice_number = (BottomUpMosaic) ? tile : (n_slices - tile - 1);
		std::vector<double> center(3,0);
		
		double value;
		wxString searchStr = wxString::Format(_T("sSliceArray.asSlice[%d].sPosition.dSag"), slice_number);
		std::string::size_type pos = protocol.find(searchStr.mb_str(wxConvLocal));
		if (pos != string::npos) {
			std::stringstream ss(protocol.substr(pos, 256));
			ss.ignore(256, '=');
			ss >> value;
			center[0] = value;
		}
		searchStr = wxString::Format(_T("sSliceArray.asSlice[%d].sPosition.dCor"), slice_number);
		pos = protocol.find(searchStr.mb_str(wxConvLocal));
		if (pos != string::npos) {
			std::stringstream ss(protocol.substr(pos, 256));
			ss.ignore(256, '=');
			ss >> value;
			center[1] = value;
		}
		searchStr = wxString::Format(_T("sSliceArray.asSlice[%d].sPosition.dTra"), slice_number);
		pos = protocol.find(searchStr.mb_str(wxConvLocal));
		if (pos != string::npos) {
			std::stringstream ss(protocol.substr(pos, 256));
			ss.ignore(256, '=');
			ss >> value;
			center[2] = value;
		}

		double dist = 0;
		for (int i = 0; i < 3; ++i) dist += rotation[i+6]*center.at(i);

		v[it->second.front()].AddSlice(dist, slice, it->first);
		
		if (minDistMap.find(it->second.front()) == minDistMap.end()) {

			// Convert center to ipp
			std::vector<double> ipp(3,0);

			ipp[0] = center[0] + rotation[0] * offset[0] + rotation[3] * offset[1];
			ipp[1] = center[1] + rotation[1] * offset[0] + rotation[4] * offset[1];
			ipp[2] = center[2] + rotation[2] * offset[0] + rotation[5] * offset[1];

			minDistMap[it->second.front()] = dist;
			position[it->second.front()] = ipp;
		}

		if (dist < minDistMap[it->second.front()]) {

			// Convert center to ipp
			std::vector<double> ipp(3,0);

			ipp[0] = center[0] + rotation[0] * offset[0] + rotation[3] * offset[1];
			ipp[1] = center[1] + rotation[1] * offset[0] + rotation[4] * offset[1];
			ipp[2] = center[2] + rotation[2] * offset[0] + rotation[5] * offset[1];

			minDistMap[it->second.front()] = dist;
			position[it->second.front()] = ipp;
		}

      }
      assert(v[(*it).second.front()].size() == n_slices);
      ++it;			
    }

  }


}


#endif

