/**
   Converter.cpp
**/

#include <wx/filename.h>

#include <string>
#include <sstream>
#include <iostream>
#include <algorithm>

#include "StringConvert.h"
#include "DicomFile.h"
#include "SeriesHandler.h"
#include "HandlerFactory.h"
#include "OutputFactory.h"
#include "Converter.h"

// for new stuff
#include "OutputterBase.h"

using namespace jcs;

template <class T>
struct GetFirst : unary_function<T, typename T::first_type> {
	typename T::first_type operator()(const T& t) const {
	return t.first;
	}
};

static void
DeleteHandler (pair<string, SeriesHandler*> pair)
{
	delete pair.second;
}
	
Converter::Converter(int type)
{
	mpOutputter = OutputFactory::CreateNewOutputter(type);
}

Converter::~Converter()
{
	HandlerMap::iterator it = mHandlerMap.begin();
	HandlerMap::iterator it_end = mHandlerMap.end();
	for (;it != it_end; it++)
	  delete it->second;
	//	for_each(mHandlerMap.begin(), mHandlerMap.end(),
	//	DeleteHandler);

	delete mpOutputter;
}


///
/**
 */
void
Converter::SetOutput(int type)
{
	delete mpOutputter;
	mpOutputter = OutputFactory::CreateNewOutputter(type);
	UpdateAllOutput();
}


///
/** Adds a file to the list of files to be processed.
\param path Path to file to add.
\param match 
*/
int
Converter::AddFile(const wxString& path, const wxString& match)
{
	Message fileMessage;
	wxString message;

	// Turn the bloody message dialog into a bearer of bad news.
	if (!mAddFile(path, match, message)) {
	  fileMessage.push_back(path);
	  fileMessage.push_back(_T("Not added"));
	  fileMessage.push_back(message);
	}

	if (fileMessage.size() > 0) 
	  messages.push_back(fileMessage);
	
	return 1;
}


///
/** Adds a file to the set to be processed. Creates a handler for each series.
    \param path Path to file.
    \param match String to match within a file's SeriesDescription.
    \param message Holds messages about this process.
    \return True if file successfully added, false otherwise.
*/
bool
Converter::mAddFile(const wxString& path, const wxString& match, wxString& message)
{
	wxString name;
	wxString ext;
	wxFileName::SplitPath(path, NULL, &name, &ext);
	if (name == _T("DICOMDIR")) {
		message = _T("We don't convert DICOMDIR files.");
		return false;
	}

	DicomFile input(path.mb_str(wxConvLocal));
	if (!input.IsOk()) {
		message = wxString(input.ErrorMessage(), wxConvLocal);
		return false;
	}

	if (!match.empty()) {
		string series_description;
		input.Find("SeriesDescription", series_description);
		if (series_description.find(match.mb_str(wxConvLocal)) == string::npos) {
			message = _T("Series description doesn't match ");
			message.append(match);
			return false;
		}
	}

	string series_uid;
	input.Find("SeriesInstanceUid", series_uid);

	// Fix problems with termination
	string temp(series_uid.c_str());
	series_uid.clear();
	series_uid = temp;

	mSeriesToUpdate.insert(series_uid);
		
	if (!mHandlerMap.count(series_uid))
		mHandlerMap[series_uid] = HandlerFactory::CreateHandler(path.mb_str(wxConvLocal));

	if (!mHandlerMap[series_uid]->AddFile(path.mb_str(wxConvLocal))) {
		message = _T("File already in list.");
		return false;
	}

	return true;
}


///
/**
 */
void
Converter::UpdateOutput()
{
	set<string>::iterator it = mSeriesToUpdate.begin();
	set<string>::iterator it_end = mSeriesToUpdate.end();
	for (;it != it_end; it++)
		mpOutputter->UpdateOutputForSeries(mHandlerMap[*it]);
	mSeriesToUpdate.clear();
}


///
/**
 */
void
Converter::UpdateAllOutput()
{
	HandlerMap::iterator it = mHandlerMap.begin();
	HandlerMap::iterator it_end = mHandlerMap.end();
	for (;it != it_end; it++)
		mpOutputter->UpdateOutputForSeries(it->second); 
}


///
/**
 */
void
Converter::UpdateOutput(const string& seriesUid)
{
	mpOutputter->UpdateOutputForSeries(mHandlerMap[seriesUid]); 
}


///
/**
 */
void 
Converter::RemoveSeries(const string& seriesUid)
{
	delete mHandlerMap[seriesUid];
	mHandlerMap.erase(seriesUid);
	mpOutputter->RemoveSeries(seriesUid);
}


///
/**
 */
const vector<string> 
Converter::GetStringInfo(const string& s) const
{
	return (*mHandlerMap.find(s)).second->GetStringInfo();
}


///
/** Returns list of series names (SeriesUID)
    \return An immutable vector of strings.
 */
const vector<string>
Converter::GetSeriesList()
{
  std::vector<string> v;
  HandlerMap::iterator it = mHandlerMap.begin();
  HandlerMap::iterator it_end = mHandlerMap.end();
  for (; it != it_end; it++)
    v.push_back(it->first);
  //  std::transform(mHandlerMap.begin(), mHandlerMap.end(), back_inserter(v),
  //		 GetFirst<HandlerMap::value_type>());

  return v;
}


///
/**
 */
void
Converter::ConvertAll()
{
	mCancel = false;

	HandlerMap::iterator it = mHandlerMap.begin();
	HandlerMap::iterator it_end = mHandlerMap.end();
	for (; it != it_end; it++) {
		if (mCancel) break;
		mpOutputter->ConvertSeries(it->second);
	}
}
