/* libc */
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/socket.h>
/* other libs */
#include <apr.h>
#include <apr_pools.h>
/* local */
#include <libbttracker.h>

int run = 1;
int stopped = 0;
sighandler_t old_int;

const struct sockaddr_in real_address = { AF_INET, 0, { 0 } };

void sigINT(int signum)
{
 run = 0;
 stopped = 1;
 signal(signum, old_int);
 return;
}

int main(int argc, const char* const*argv)
{
 int master = 0;
 apr_pool_t* pool;
 btt_tracker* tracker;
 int i;
 char* rv = NULL;
 int len = 0;
 int code = 0;
 apr_pool_t* subpool;

 if(argc < 3)
 {
  fprintf(stderr, "Usage: %s homedir [-master] urlargs urlargs urlargs\n", argv[0]);
  exit(1);
 }
 
 if(!strcmp("-master", argv[2]))
 {
  master = 1;
 }
 
 if(apr_app_initialize(&argc, &argv, NULL) != APR_SUCCESS)
 {
  fprintf(stderr, "apr_app_initialize() failed!\n");
  fflush(stderr);
  exit(20);
 }
 
 atexit(apr_terminate);
 

 if(apr_pool_initialize() != APR_SUCCESS)
 {
  fprintf(stderr, "apr_pool_initialize() failed!\n");
  fflush(stderr);
  exit(2);
 }
 
 if(apr_pool_create(&pool, NULL) != APR_SUCCESS)
 {
  fprintf(stderr, "apr_pool_create() failed!\n");
  fflush(stderr);
  exit(3);
 }

 if(!bt_random_seed())
 {
  fprintf(stderr, "bt_random_seed() failed!\n");
  fflush(stderr);
  exit(4);
 }

 if(!(tracker = btt_tracker_alloc(pool, argv[1], master)))
 {
  fprintf(stderr, "bt_tracker_alloc() failed!\n");
  fflush(stderr);
  exit(5);
 }
 
 if(!btt_tracker_connect(tracker, master))
 {
  fprintf(stderr, "bt_tracker_connect() failed!\n");
  fflush(stderr);
  exit(6);
 }
 
 if(master)
  tracker->s->start_t = time(NULL);

 if((old_int = signal(SIGINT, sigINT)) == SIG_ERR)
 {
  fprintf(stderr, "signal() failed!\n");
  exit(7);
 }
 
 printf("Fake tracker started %ld second(s) ago. (Master = %d)\nSend SIGINT (^C) to shutdown.\n", time(NULL) - tracker->s->start_t, master);

 for(i = 2 + master; i < argc; i++)
 {
  apr_pool_create(&subpool, pool);
  code = btt_cxn_announce(tracker, subpool, NULL, argv[i], "TestClient/0.01", real_address, &rv, &len);
  printf("%s\n%u (%u bytes)\n", argv[i], code, len);
  if(rv)
  {
   fwrite(rv, len, 1, stdout);
   printf("\n");
  }
  printf("\n");
  apr_pool_destroy(subpool);
 }

 if(stopped)
  printf("Caught SIGINT, shutting down.\n");
 else if(!tracker->s->master_pid)
  printf("Master tracker stopped, shutting down.\n");

 if(master)
  tracker->s->master_pid = 0;
 
 if(!btt_tracker_disconnect(tracker))
 {
  fprintf(stderr, "bt_tracker_disconnect() failed!\n");
  exit(8);
 }
 
 if(!btt_tracker_free(&tracker, master))
 {
  fprintf(stderr, "bt_tracker_free() failed!\n");
  exit(9);
 }
 
 printf("Tracker instance shut down successfully.\n");
 exit(0);
}
