# -*- coding: utf-8 -*-
#
# Author: Natalia Bidart <nataliabidart@gmail.com>
#
# Copyright 2010-2011 Chicharreros
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Tests for the Operations widget."""

import gtk

from magicicada.gui.gtk.tests import BaseTestCase, FakedSyncdaemon
from magicicada.gui.gtk import operations

# Instance of 'A' has no 'y' member
# pylint: disable=E1101

# Instance of 'A' has no 'y' member (but some types could not be inferred)
# pylint: disable=E1103

# Invalid name Node
# pylint: disable=C0103

# Example of info sent from backend is:
#[('Root',
#  {u'Ubuntu One':
#   <Node Ubuntu One 'Dir' last_modified=None done=None operations=[]
#    children={u'1.txt':
#              <Node 1.txt 'File' last_modified=1313966619.14 done=False
#               operations=[(dbus.String(u'99537584'),
#                            dbus.String(u'MakeFile'), {})]
#               children={}>}>})]

Node = operations.queue_content.Node
DONE_OPS = {operations.queue_content.DONE: True}
UNDONE_OPS = {operations.queue_content.DONE: False}


class NodeOpsTestCase(BaseTestCase):
    """UI test cases for the node ops tree views."""

    ui_class = operations.Operations
    sd = FakedSyncdaemon()
    kwargs = {'syncdaemon_instance': sd}

    def assert_store_correct(self, items, store=None):
        """Test that 'store' has 'items' as content."""
        store = self.ui.ops_store
        super(NodeOpsTestCase, self).assert_store_correct(items, store)

    def test_itself_is_packed(self):
        """The main widget is packed."""
        self.assertIs(self.ui.itself, self.ui.get_child())

    def test_syncdaemon_instance(self):
        """The syncadaemon instance is stored."""
        self.assertIs(self.ui.sd, self.sd)

    def test_callback_is_connected(self):
        """The on_node_ops_changed callback is connected."""
        self.assertEqual(self.sd.on_node_ops_changed_callback,
                         self.ui.on_node_ops_changed)

    def test_is_visible(self):
        """Is visible at startup."""
        self.assertTrue(self.ui.get_visible())

    def test_model_is_bind(self):
        """The store is bind to the view."""
        self.assertEqual(self.ui.ops_store, self.ui.ops_view.get_model())

    def test_clear_button_is_disabled(self):
        """Can not clear listing at startup."""
        self.assertFalse(self.ui.clear_button.is_sensitive(),
                         'Clear button must be disabled.')

    def test_on_node_ops_changed_roots_only(self):
        """On on_node_ops_changed the view is updated."""
        sd_items = [(operations.queue_content.ROOT_HOME, {}),
                    ('Yadda', {}), ('Yodda', {})]
        self.sd.on_node_ops_changed_callback(sd_items)

        root = ('', '', None, operations.HOME_ICON_NAME,
                gtk.ICON_SIZE_LARGE_TOOLBAR)
        yada = ('', '', None, operations.REMOTE_ICON_NAME,
                gtk.ICON_SIZE_LARGE_TOOLBAR)
        yoda = ('', '', None, operations.REMOTE_ICON_NAME,
                gtk.ICON_SIZE_LARGE_TOOLBAR)

        self.assert_store_correct([(root, []), (yada, []), (yoda, [])])
        self.assertFalse(self.ui.clear_button.is_sensitive(),
                         'Clear button must be disabled.')

    def test_on_node_ops_changed_root_info(self):
        """On on_node_ops_changed the view is updated."""
        node = Node(name='some node', parent=None,
                    kind=operations.queue_content.KIND_DIR)
        sd_items = [('Yadda', {node.name: node})]
        self.sd.on_node_ops_changed_callback(sd_items)

        yada = ('', '', None, operations.REMOTE_ICON_NAME,
                gtk.ICON_SIZE_LARGE_TOOLBAR)
        node_row = (node.name, '', None,
                    operations.FOLDER_ICON_NAME, gtk.ICON_SIZE_SMALL_TOOLBAR)
        expected = [(yada, [(node_row, [])])]

        self.assert_store_correct(expected)
        self.assertFalse(self.ui.clear_button.is_sensitive(),
                         'Clear button must be disabled.')

    def test_on_node_ops_changed_dir_and_file(self):
        """On on_node_ops_changed the view is updated."""
        file_node = Node(name='a_file.txt', parent=None,
                    kind=operations.queue_content.KIND_FILE)
        file_node.operations = [(object(), 'foo', UNDONE_OPS),
                                (object(), 'bar', DONE_OPS),
                                (object(), 'baz', UNDONE_OPS)]
        dir_node = Node(name='a_dir', parent=None,
                    kind=operations.queue_content.KIND_DIR)
        dir_node.operations = [(object(), 'doo', UNDONE_OPS),
                                (object(), 'bar', UNDONE_OPS)]
        node = Node(name='some node', parent=None,
                    kind=operations.queue_content.KIND_DIR)
        node.children = {file_node.name: file_node, dir_node.name: dir_node}
        sd_items = [('Yadda', {node.name: node})]
        self.sd.on_node_ops_changed_callback(sd_items)

        yada = ('', '', None, operations.REMOTE_ICON_NAME,
                gtk.ICON_SIZE_LARGE_TOOLBAR)
        node_row = (node.name, '', None,
                    operations.FOLDER_ICON_NAME, gtk.ICON_SIZE_SMALL_TOOLBAR)
        file_row = (file_node.name, operations.OPS_MARKUP % 'foo, baz', None,
                    operations.FILE_ICON_NAME, gtk.ICON_SIZE_SMALL_TOOLBAR)
        dir_row = (dir_node.name, operations.OPS_MARKUP % 'doo, bar', None,
                   operations.FOLDER_ICON_NAME, gtk.ICON_SIZE_SMALL_TOOLBAR)
        expected = [(yada, [(node_row, [(file_row, []), (dir_row, [])])])]

        self.assert_store_correct(expected)
        self.assertFalse(self.ui.clear_button.is_sensitive(),
                         'Clear button must be disabled.')

    def test_on_node_ops_changed_all_done(self):
        """On on_node_ops_changed the view is updated."""
        fixed_time = 12345678
        self.patch(operations.time, 'time', lambda: fixed_time)
        delta = fixed_time - (60 * 60 * 5)
        file_node = Node(name='a_file.txt', parent=None, last_modified=delta,
                    kind=operations.queue_content.KIND_FILE)
        long_op_name = 'x' * (operations.MAX_OP_LEN + 1)
        file_node.operations = [(object(), 'foo', DONE_OPS),
                                (object(), 'bar', DONE_OPS),
                                (object(), long_op_name, DONE_OPS)]
        sd_items = [('Yadda', {file_node.name: file_node})]
        self.sd.on_node_ops_changed_callback(sd_items)

        yada = ('', '', None, operations.REMOTE_ICON_NAME,
                gtk.ICON_SIZE_LARGE_TOOLBAR)
        op_name = ('foo, bar, ' + long_op_name)
        op_name = op_name[:operations.MAX_OP_LEN - len(operations.ELLIPSIS)]
        op_name += operations.ELLIPSIS
        ago = '5 hours'
        op_name = operations.OPS_COMPLETED % (ago, op_name)
        file_row = (file_node.name, operations.OPS_MARKUP % op_name, None,
                    operations.FILE_ICON_NAME, gtk.ICON_SIZE_SMALL_TOOLBAR)
        expected = [(yada, [(file_row, [])])]

        self.assert_store_correct(expected)
        self.assertTrue(self.ui.clear_button.is_sensitive(),
                        'Clear button must be enabled.')

        # can not clear any longer
        self.sd.on_node_ops_changed_callback([])
        self.assertFalse(self.ui.clear_button.is_sensitive(),
                         'Clear button must be disabled.')

    def test_on_node_ops_changed_handles_none(self):
        """On on_node_ops_changed handles None as items."""
        self.sd.on_node_ops_changed_callback(None)
        self.assert_store_correct([])

    def test_model_is_cleared_before_updating(self):
        """The model is cleared before upadting with a new set of data."""
        sd_items = [(operations.queue_content.ROOT_HOME, {}),
                    ('Yadda', {}), ('Yodda', {})]
        self.sd.on_node_ops_changed_callback(sd_items)

        self.sd.on_node_ops_changed_callback([])
        self.assertEqual(len(self.ui.ops_store), 0)

    def test_load(self):
        """Calling load will query the backend."""
        expected = object()
        self.patch(self.ui.sd.queue_content, 'node_ops', expected)
        self.patch(self.ui, 'on_node_ops_changed', self._set_called)

        self.ui.load()

        self.assertEqual(self._called, ((expected,), {}))

    def test_clear_button_clicked(self):
        """When the clear button was clicked, the store is cleared."""
        self.patch(self.ui, 'load', self._set_called)
        sd_items = [(operations.queue_content.ROOT_HOME, {}),
                    ('Yadda', {}), ('Yodda', {})]
        self.sd.on_node_ops_changed_callback(sd_items)

        self.ui.clear_button.clicked()

        self.assertEqual(len(self.ui.sd.queue_content), 0)
        self.assertEqual(self._called, ((), {}))


class NumberedFolderTestCase(BaseTestCase):
    """Tets case for the widget that representsa folder with a number in it."""

    ui_class = operations.NumberedFolder.new_from_amount
    kwargs = {'amount': 5}

    def test_is_pixbuf(self):
        """Is a gtk.gdk.Pixbuf."""
        self.assertIsInstance(self.ui, gtk.gdk.Pixbuf)
