# __init__.py
#
# Author: Natalia Bidart <natalia.bidart@gmail.com>
#
# Copyright 2010 Chicharreros
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Magicicada GTK UI."""

import logging
import os

import gettext
_ = gettext.gettext
gettext.textdomain('magicicada')

import gtk

# optional Launchpad integration
# this shouldn't crash if not found as it is simply used for bug reporting
try:
    import LaunchpadIntegration
    LAUNCHPAD_AVAILABLE = True
except ImportError:
    LAUNCHPAD_AVAILABLE = False

from twisted.internet import gtk2reactor  # for gtk-2.0
gtk2reactor.install()

from magicicada import syncdaemon, logger as logger_helper
from magicicada.dbusiface import NOT_SYNCHED_PATH
from magicicada.helpers import humanize_bytes, get_data_file, get_builder, \
    log, NO_OP

CONTENT_QUEUE = 'content'
META_QUEUE = 'meta'
UBUNTU_ONE_ROOT = os.path.expanduser('~/Ubuntu One')

# set up the logging for all the project
# Instance of 'RootLogger' has no 'set_up' member
# pylint: disable=E1103
logger_helper.set_up()
logger = logging.getLogger('magicicada.ui')

# Instance of 'A' has no 'y' member
# pylint: disable=E1101

# Unused argument, we need them for GTK callbacks
# pylint: disable=W0613


class MagicicadaUI(object):
    """Magicicada GUI main class."""

    CURRENT_ROW = '<b><span foreground="#000099">%s</span></b>'
    STATUS_JOINER = ' - '
    STATUS = {
        'initial': _('Service is not started, click Start to continue.'),
    }

    _u1_root = UBUNTU_ONE_ROOT

    def __init__(self, on_destroy=NO_OP,
                 syncdaemon_class=syncdaemon.SyncDaemon):
        """Init."""
        self.builder = get_builder('gui.glade')
        self.builder.connect_signals(self)

        if LAUNCHPAD_AVAILABLE:
            # for more information about LaunchpadIntegration:
            # wiki.ubuntu.com/UbuntuDevelopment/Internationalisation/Coding
            helpmenu = self.builder.get_object('helpMenu')
            if helpmenu:
                LaunchpadIntegration.set_sourcepackagename('magicicada')
                LaunchpadIntegration.add_items(helpmenu, 0, False, True)

        self.on_destroy = on_destroy

        animation_filename = get_data_file('media', 'loader-ball.gif')
        self.loading_animation = gtk.gdk.PixbufAnimation(animation_filename)
        active_filename = get_data_file('media', 'active-016.png')
        self.active_indicator = gtk.gdk.pixbuf_new_from_file(active_filename)

        widgets = (
            'start', 'stop', 'connect', 'disconnect',  # toolbar buttons
            'folders', 'folders_dialog',  # folders
            'folders_view', 'folders_store', 'folders_close',
            'shares_to_me', 'shares_to_me_dialog',  # shares_to_me
            'shares_to_me_view', 'shares_to_me_store', 'shares_to_me_close',
            'shares_to_others', 'shares_to_others_dialog',  # shares_to_others
            'shares_to_others_view', 'shares_to_others_store',
            'shares_to_others_close', 'metadata',  # metadata
            'public_files', 'public_files_dialog',  # public_files
            'public_files_view', 'public_files_store', 'public_files_close',
            'is_started', 'is_connected', 'is_online',  # status bar images
            'status_label', 'status_icon',  # status label and systray icon
            'metaq_view', 'contentq_view',  # queues tree views
            'metaq_store', 'contentq_store',  # queues list stores
            'metaq_label', 'contentq_label',  # queues labels
            'file_chooser', 'file_chooser_open', 'file_chooser_cancel',
            'about_dialog', 'main_window')

        for widget in widgets:
            obj = self.builder.get_object(widget)
            setattr(self, widget, obj)
            assert obj is not None, '%s must not be None' % widget

        self._sorting_order = {}
        self._make_view_sortable('folders')
        self._make_view_sortable('shares_to_me')
        self._make_view_sortable('shares_to_others')
        self._make_view_sortable('public_files')

        self.metadata_dialogs = {}
        self.volumes = (self.folders, self.shares_to_me, self.shares_to_others)

        self._icons = {}
        for size in (16, 32, 48, 64, 128):
            icon_filename = get_data_file('media', 'logo-%.3i.png' % size)
            self._icons[size] = gtk.gdk.pixbuf_new_from_file(icon_filename)
        self.status_icon.set_from_pixbuf(self._icons[16])
        self.main_window.set_icon_list(*self._icons.values())
        gtk.window_set_default_icon_list(*self._icons.values())

        about_fname = get_data_file('media', 'logo-128.png')
        self.about_dialog.set_logo(gtk.gdk.pixbuf_new_from_file(about_fname))

        self.sd = syncdaemon_class()
        self.sd.on_started_callback = self.on_started
        self.sd.on_stopped_callback = self.on_stopped
        self.sd.on_connected_callback = self.on_connected
        self.sd.on_disconnected_callback = self.on_disconnected
        self.sd.on_online_callback = self.on_online
        self.sd.on_offline_callback = self.on_offline
        self.sd.status_changed_callback = self.on_status_changed
        self.sd.content_queue_changed_callback = self.on_content_queue_changed
        self.sd.meta_queue_changed_callback = self.on_meta_queue_changed
        self.sd.on_metadata_ready_callback = self.on_metadata_ready
        self.sd.on_initial_data_ready_callback = self.on_initial_data_ready
        self.sd.on_initial_online_data_ready_callback = \
            self.on_initial_online_data_ready

        self.widget_is_visible = lambda w: w.get_property('visible')
        self.widget_enabled = lambda w: self.widget_is_visible(w) and \
                                        w.is_sensitive()
        self.update()

    def _make_view_sortable(self, view_name):
        """Set up view so columns are sortable."""
        store = getattr(self, '%s_store' % view_name)
        view = getattr(self, '%s_view' % view_name)
        self._sorting_order[store] = {}
        # this enforces that the order that columns will be shown and sorted
        # matches the order in the underlying model
        for i, col in enumerate(view.get_columns()):
            col.set_clickable(True)
            col.connect('clicked', self.on_store_sort_column_changed, i, store)
            self._sorting_order[store][i] = gtk.SORT_ASCENDING

    def _new_metadata_dialog(self, path):
        """Return a new metadata dialog."""
        dialog = gtk.Dialog(title='Metadata for %s' % path,
                            parent=self.main_window,
                            flags=gtk.DIALOG_NO_SEPARATOR,
                            buttons=(gtk.STOCK_CLOSE, gtk.RESPONSE_CLOSE))
        dialog.set_size_request(600, 300)
        dialog.set_border_width(10)
        # gtk.WIN_POS_CENTER makes dialogs overlap
        dialog.set_position(gtk.WIN_POS_MOUSE)

        close_button = dialog.action_area.get_children()[-1]
        close_button.connect('clicked', self.on_metadata_close_clicked, path)
        close_button.connect('activate', self.on_metadata_close_clicked, path)
        dialog.close = close_button

        spinner = gtk.Spinner()
        spinner.start()
        dialog.spinner = spinner

        dialog.get_child().add(spinner)

        text_view = gtk.TextView()
        text_view.set_editable(False)
        text_view.set_wrap_mode(gtk.WRAP_WORD)
        dialog.get_child().add(text_view)
        dialog.view = text_view

        self.metadata_dialogs[path] = dialog
        return dialog

    # GTK callbacks

    def on_main_window_destroy(self, widget, data=None):
        """Called when the MagicicadaWindow is closed."""
        # Clean up code for saving application state should be added here.
        self.sd.shutdown()
        self.on_destroy()

    def on_quit_activate(self, widget, data=None):
        """Signal handler for closing the program."""
        self.on_main_window_destroy(self.main_window)

    def on_about_activate(self, widget, data=None):
        """Display the about box."""
        self.about_dialog.run()
        self.about_dialog.hide()

    def on_start_clicked(self, widget, data=None):
        """Start syncdaemon."""
        self.sd.start()
        self.start.set_sensitive(False)
        self._start_loading(self.is_started)

    def on_stop_clicked(self, widget, data=None):
        """Stop syncdaemon."""
        for volume in self.volumes:
            volume.set_sensitive(False)
        self.metadata.set_sensitive(False)
        self.public_files.set_sensitive(False)

        if self.widget_enabled(self.disconnect):
            self.on_disconnect_clicked(self.disconnect)
        self.connect.set_sensitive(False)

        self.stop.set_sensitive(False)
        self.sd.quit()

    def on_connect_clicked(self, widget, data=None):
        """Connect syncdaemon."""
        self.sd.connect()
        self.connect.set_sensitive(False)
        self._start_loading(self.is_connected)

    def on_disconnect_clicked(self, widget, data=None):
        """Disconnect syncdaemon."""
        self.disconnect.set_sensitive(False)
        self.sd.disconnect()

    def on_folders_close_clicked(self, widget, data=None):
        """Close the folders dialog."""
        self.folders_dialog.response(gtk.RESPONSE_CLOSE)

    def on_folders_clicked(self, widget, data=None):
        """List user folders."""
        items = self.sd.folders
        if items is None:
            items = []

        self.folders_store.clear()
        for item in items:
            row = (item.path, item.suggested_path, item.subscribed,
                   item.node, item.volume)
            self.folders_store.append(row)

        self.folders_dialog.run()
        self.folders_dialog.hide()

    def on_shares_to_me_close_clicked(self, widget, data=None):
        """Close the shares_to_me dialog."""
        self.shares_to_me_dialog.response(gtk.RESPONSE_CLOSE)

    @log(logger)
    def _on_shares_clicked(self, items, store, dialog):
        """List shares to the user or to others."""
        if items is None:
            items = []

        store.clear()
        for item in items:
            free_bytes = item.free_bytes
            try:
                free_bytes = humanize_bytes(int(free_bytes), precision=2)
            except (ValueError, TypeError):
                logger.exception('Error while humanizing bytes')
            row = (item.name, item.other_visible_name, item.accepted,
                   item.access_level, free_bytes, item.path,
                   item.other_username, item.node_id, item.volume_id)
            store.append(row)

        dialog.run()
        dialog.hide()

    def on_shares_to_me_clicked(self, widget, data=None):
        """List shares to the user."""
        self._on_shares_clicked(self.sd.shares_to_me,
                                self.shares_to_me_store,
                                self.shares_to_me_dialog)

    def on_shares_to_others_close_clicked(self, widget, data=None):
        """Close the shares_to_others dialog."""
        self.shares_to_others_dialog.response(gtk.RESPONSE_CLOSE)

    def on_shares_to_others_clicked(self, widget, data=None):
        """List user shares to others."""
        self._on_shares_clicked(self.sd.shares_to_others,
                                self.shares_to_others_store,
                                self.shares_to_others_dialog)

    def on_file_chooser_open_clicked(self, widget, data=None):
        """Close the file_chooser dialog."""
        self.file_chooser.response(gtk.FILE_CHOOSER_ACTION_OPEN)

    def on_file_chooser_show(self, widget, data=None):
        """Close the file_chooser dialog."""
        self.file_chooser.set_current_folder(self._u1_root)

    def on_metadata_close_clicked(self, widget, path):
        """Close the metadata dialog."""
        self.metadata_dialogs[path].destroy()

    def on_metadata_clicked(self, widget, data=None):
        """Show metadata for a path choosen by the user."""
        res = self.file_chooser.run()
        self.file_chooser.hide()
        if res != gtk.FILE_CHOOSER_ACTION_OPEN:
            return

        path = self.file_chooser.get_filename()
        assert path is not None
        dialog = self._new_metadata_dialog(path)
        self.sd.get_metadata(path)
        dialog.view.hide()
        dialog.spinner.show()
        dialog.show()

    def on_public_files_close_clicked(self, widget, data=None):
        """Close the public_files dialog."""
        self.public_files_dialog.response(gtk.RESPONSE_CLOSE)

    def on_public_files_clicked(self, widget, data=None):
        """Show information about public files."""
        items = self.sd.public_files
        if items is None:
            items = {}

        self.public_files_store.clear()
        for item in items.itervalues():
            row = (item.path, item.public_url, item.volume, item.node)
            self.public_files_store.append(row)

        self.public_files_dialog.run()
        self.public_files_dialog.hide()

    def on_status_icon_activate(self, widget, data=None):
        """Systray icon was clicked."""
        if self.widget_is_visible(self.main_window):
            self.main_window.hide()
        else:
            self.main_window.show()

    def on_store_sort_column_changed(self, column, col_index, store):
        """Store sort requested."""
        order = self._sorting_order[store][col_index]
        last_col = self._sorting_order[store].get('last_col')
        if last_col is not None:
            last_col.set_sort_indicator(False)
        logger.debug('Sorting col index %s, named %s, with order %s',
                     col_index, column.get_name(), order)

        store.set_sort_column_id(col_index, order)
        column.set_sort_indicator(True)
        column.set_sort_order(order)
        self._sorting_order[store]['last_col'] = column

        # change order
        if order == gtk.SORT_ASCENDING:
            order = gtk.SORT_DESCENDING
        else:
            order = gtk.SORT_ASCENDING
        self._sorting_order[store][col_index] = order

    # SyncDaemon callbacks

    @log(logger)
    def on_started(self, *args, **kwargs):
        """Callback'ed when syncadaemon is started."""
        self.start.hide()
        self.stop.show()
        self.stop.set_sensitive(True)
        self._activate_indicator(self.is_started)
        self.connect.set_sensitive(True)

        self._update_queues_and_status(self.sd.current_state)

    @log(logger)
    def on_stopped(self, *args, **kwargs):
        """Callback'ed when syncadaemon is stopped."""
        if self.widget_enabled(self.stop):
            self.on_stop_clicked(self.stop)

        self.stop.hide()
        self.start.show()
        self.start.set_sensitive(True)
        self.connect.set_sensitive(False)

        self._activate_indicator(self.is_started, sensitive=False)
        self._activate_indicator(self.is_connected, sensitive=False)
        self._activate_indicator(self.is_online, sensitive=False)

        self._update_queues_and_status(self.sd.current_state)

    @log(logger)
    def on_connected(self, *args, **kwargs):
        """Callback'ed when syncadaemon is connected."""
        self.connect.hide()
        self.disconnect.show()
        self.disconnect.set_sensitive(True)
        self._activate_indicator(self.is_connected)
        self._start_loading(self.is_online)
        self._start_loading(self.is_online)

        self._update_queues_and_status(self.sd.current_state)

    @log(logger)
    def on_disconnected(self, *args, **kwargs):
        """Callback'ed when syncadaemon is disconnected."""
        self.disconnect.hide()
        self.connect.show()
        self.connect.set_sensitive(True)

        self._activate_indicator(self.is_connected, sensitive=False)
        self._activate_indicator(self.is_online, sensitive=False)

        self._update_queues_and_status(self.sd.current_state)

    @log(logger)
    def on_online(self, *args, **kwargs):
        """Callback'ed when syncadaemon is online."""
        self.is_online.set_sensitive(True)
        self._activate_indicator(self.is_online)
        self._update_queues_and_status(self.sd.current_state)

    @log(logger)
    def on_offline(self, *args, **kwargs):
        """Callback'ed when syncadaemon is offline."""
        self._activate_indicator(self.is_online, sensitive=False)
        self._update_queues_and_status(self.sd.current_state)

    @log(logger)
    def on_status_changed(self, *args, **kwargs):
        """Callback'ed when the SD status changed."""
        self.update(*args, **kwargs)

    @log(logger)
    def _on_queue_changed(self, queue_name, items, must_highlight):
        """Callback'ed when a queue changed."""
        if items is None:
            items = []

        markup = lambda value: self.CURRENT_ROW % value \
                 if must_highlight and value is not None else value
        queue_label = getattr(self, '%sq_label' % queue_name)
        queue_view = getattr(self, '%sq_view' % queue_name)
        queue_store = getattr(self, '%sq_store' % queue_name)
        queue_store.clear()
        for i, item in enumerate(items):
            row = (item.operation, item.path, item.share, item.node)
            if i == 0:
                row = map(markup, row)
            queue_store.append(row)

        items_len = len(items)
        label = '%s Queue (%i)' % (queue_name.capitalize(), items_len)
        queue_label.set_text(label)

        if not queue_view.is_sensitive() and len(items) > 0:
            queue_view.set_sensitive(True)

    def on_content_queue_changed(self, items, *args, **kwargs):
        """Callback'ed when syncadaemon's content queue changed."""
        state = self.sd.current_state
        highlight = state.processing_content and state.is_online
        self._on_queue_changed(CONTENT_QUEUE, items, highlight)

    def on_meta_queue_changed(self, items, *args, **kwargs):
        """Callback'ed when syncadaemon's meta queue changed."""
        state = self.sd.current_state
        highlight = state.processing_meta and state.is_online
        self._on_queue_changed(META_QUEUE, items, highlight)

    @log(logger)
    def on_metadata_ready(self, path, metadata):
        """Lower layer has the requested metadata for 'path'."""
        if path not in self.metadata_dialogs:
            logger.info('on_metadata_ready: path %r not in stored paths!')
            return

        dialog = self.metadata_dialogs[path]
        dialog.spinner.hide()
        dialog.view.show()
        if metadata == NOT_SYNCHED_PATH:
            # Metadata path doesn't exsit for syncdaemon
            text = NOT_SYNCHED_PATH
        else:
            text = '\n'.join('%s: %s' % i for i in metadata.iteritems())
        dialog.view.get_buffer().set_text(text)

    @log(logger, level=logging.INFO)
    def on_initial_data_ready(self):
        """Initial data is now available in syncdaemon."""
        for volume in self.volumes:
            volume.set_sensitive(True)
        self.metadata.set_sensitive(True)

    @log(logger, level=logging.INFO)
    def on_initial_online_data_ready(self):
        """Online initial data is now available in syncdaemon."""
        self.public_files.set_sensitive(True)

    # custom

    def _start_loading(self, what):
        """Set a loader animation on 'what'."""
        what.set_sensitive(True)
        what.set_from_animation(self.loading_animation)

    def _activate_indicator(self, what, sensitive=True):
        """Set ready pixbuf on 'what' and make it 'sensitive'."""
        what.set_sensitive(sensitive)
        what.set_from_pixbuf(self.active_indicator)

    def _update_status_label(self, state):
        """Update the status label based on SD state."""
        values = (v for v in (state.name, state.description,
                              state.queues, state.connection) if v)
        text = self.STATUS_JOINER.join(values)
        if not (text or state.is_started):
            text = self.STATUS['initial']
        logger.debug('setting status label to %r', text)
        self.status_label.set_text(text)

    def _update_queues_and_status(self, state):
        """Update UI based on SD current state."""
        self.on_meta_queue_changed(self.sd.meta_queue)
        self.on_content_queue_changed(self.sd.content_queue)
        self._update_status_label(state)

    def update(self, *args, **kwargs):
        """Update UI based on SD current state."""
        current_state = self.sd.current_state
        logger.debug('updating UI with state %r', current_state)

        self._activate_indicator(self.is_online,
                                 sensitive=current_state.is_online)

        if current_state.is_started:
            self.on_started()
            if current_state.is_connected:
                self.on_connected()
                if current_state.is_online:
                    self.on_online()
            else:
                self.on_disconnected()
        else:
            self.on_disconnected()
            self.on_stopped()

        self._update_queues_and_status(current_state)
