#define PHIDGETS_INTERNAL

#include <phidgets/rfid.h>
#include <debug.h>
#include <assert.h>

/* System includes */
#include <time.h>

PhidgetRFID* phidget_new_PhidgetRFID() 
{
	PhidgetRFID *ret;
		  
	TRACE("creating a new PhidgetRFID instance...");

	ret = (PhidgetRFID*)malloc(sizeof(PhidgetRFID));
	if(!ret) {
		ERROR("could not allocate memory for PhidgetRFID instance.");
		return 0;
	}

	ret->phidget = phidget_new_Phidget();

	phidget_reset_PhidgetRFID(ret);
	return ret;
}

void phidget_delete_PhidgetRFID(PhidgetRFID** const qsc)
{
	if (!qsc || !*qsc) {
		ERROR("cannot delete NULL PhidgetRFID.");
		return;
	}

	free(*qsc);
	*qsc = 0;
}

void phidget_reset_PhidgetRFID(PhidgetRFID* const qsc)
{
	if(!qsc) {
		ERROR("cannot reset NULL PhidgetRFID.");
	}	

	phidget_reset_Phidget(qsc->phidget);

	qsc->onboard_led = false;
	qsc->plus_five = false;
	qsc->external_led = false;
	qsc->enabled = false;
}

phidget_return phidget_rfid_open(PhidgetRFID* const qsc,
	unsigned int serial, unsigned short retries)
{
	HIDInterfaceMatcher matcher;
	phidget_return ret;
		  
	if(!phidget_is_initialised()) {
		ERROR("cannot open PhidgetRFID when Phidgets library has not been initialised.");
		return PHIDGET_RET_NOT_INITIALISED;
	}

	if(!qsc) {
		ERROR("cannot open NULL PhidgetRFID.");
		return PHIDGET_RET_INVALID_PARAMETER;
	}

	if(phidget_rfid_is_opened(qsc)) {
		ERROR("cannot open already opened PhidgetRFID.");
		return PHIDGET_RET_DEVICE_ALREADY_OPENED;
	}

	TRACE("opening PhidgetRFID with serial number %d...", serial);

	matcher.vendor_id = PHIDGETS_USB_VENDORID;
	matcher.product_id = PHIDGETS_USB_PRODUCTID_RFID;

	ret = phidget_open(qsc->phidget, 0, &matcher, serial, retries);
	if(ret != PHIDGET_RET_SUCCESS) return ret;

	NOTICE("successfully opened PhidgetRFID %s...", qsc->phidget->id);
	return PHIDGET_RET_SUCCESS;
}

phidget_return phidget_rfid_close(PhidgetRFID* const qsc)
{
	if(!qsc) {
		ERROR("cannot close NULL PhidgetRFID.");
		return PHIDGET_RET_INVALID_PARAMETER;
	}

	if(phidget_rfid_is_opened(qsc)) {
		phidget_return ret;
		TRACE("closing PhidgetRFID %s...", qsc->phidget->id);
		ret = phidget_close(qsc->phidget);
		if (ret != PHIDGET_RET_SUCCESS) return ret;
	}
	else WARNING("attempt to close unopened PhidgetRFID.");

	NOTICE("successfully closed PhidgetRFID %s.", qsc->phidget->id);
	return PHIDGET_RET_SUCCESS;
}

bool phidget_rfid_is_opened(PhidgetRFID const* const qsc)
{
	if(!qsc) WARNING("attempting to query open status of NULL PhidgetRFID.");
	return qsc && phidget_is_opened(qsc->phidget);
}

phidget_return phidget_rfid_set_toggle(PhidgetRFID* const qsc,
	phidget_rfid_toggle which, bool state) 
{
	bool enabled = qsc->enabled,
		  onboard_led = qsc->onboard_led,
		  external_led = qsc->external_led,
		  plus_five = qsc->plus_five;
		  
	switch(which) {
		case PHIDGET_RFID_PLUS_FIVE: plus_five = state; break;
		case PHIDGET_RFID_EXTERNAL_LED: external_led = state; break;
		case PHIDGET_RFID_ONBOARD_LED: onboard_led = state; break;
		case PHIDGET_RFID_ENABLE: enabled = state; break;
		default:
			return PHIDGET_RET_INVALID_PARAMETER;
	}

	return phidget_rfid_set_state(qsc, enabled, onboard_led, external_led,
		plus_five);
}

phidget_return phidget_rfid_set_state(PhidgetRFID* const qsc,
	bool enable, bool onboard_led, bool external_led, bool plus_five) 
{
	char packet[PHIDGETS_RFID_SET_PACKETLEN];
	int const PATH[] = 
		{ PHIDGETS_HID_PATH_1, PHIDGETS_HID_PATH_2, PHIDGETS_HID_PATH_RFID_SET };
	hid_return ret;
	
	if(!phidget_rfid_is_opened(qsc)) {
		ERROR("cannot toggle led of unopen PhidgetRFID.");
		return PHIDGET_RET_DEVICE_NOT_OPENED;
	}
		  
	TRACE("setting state: enabled: %i; onboard led: %i; external led: %i; +5: %i.", enable, onboard_led, external_led, plus_five);

	packet[0] = packet[1] = packet[2] = packet[3] = 0;
	if (enable) 		packet[0] |= PHIDGETS_RFID_ENABLE;
	if (onboard_led)  packet[0] |= PHIDGETS_RFID_ONBOARD_LED;
	if (external_led) packet[0] |= PHIDGETS_RFID_EXTERNAL_LED;
	if (plus_five) 	packet[0] |= PHIDGETS_RFID_PLUS_FIVE;
	
	TRACE("sending packet 0x[%02x %02x %02x %02x] to PhidgetRFID "
		 "%s.", packet[0], packet[1], packet[2], packet[3], qsc->phidget->id);


	ret = hid_set_output_report(qsc->phidget->hid_iface,
		PATH, PHIDGETS_HID_PATH_DEPTH, packet, 4);
	if(ret != HID_RET_SUCCESS) {
		WARNING("failed to send packet to PhidgetRFID %s.", qsc->phidget->id);
		return PHIDGET_RET_HID_ERROR;
	}
	
	qsc->enabled = enable;
	qsc->onboard_led = onboard_led;
	qsc->external_led = external_led;
	qsc->plus_five = plus_five;
	
	return PHIDGET_RET_SUCCESS;
}

phidget_return phidget_rfid_get_tag(PhidgetRFID *const qsc, const int timeout) {
	int i, ret;
	char packet[PHIDGETS_RFID_GET_PACKETLEN];
	
	if(!phidget_rfid_is_opened(qsc)) {
		ERROR("cannot get tag of unopened PhidgetRFID.");
		return PHIDGET_RET_DEVICE_NOT_OPENED;
	}

   TRACE("getting tag off of PhidgetRFID %s.", qsc->phidget->id);

	/* Read the initial card
	 * 0 means we got garbage .. -110 means we timed out */
	ret = 0;
	while(ret == 0) {
		ret = usb_interrupt_read(qsc->phidget->hid_iface->dev_handle,
			USB_ENDPOINT_IN + 1, packet, PHIDGETS_RFID_GET_PACKETLEN, timeout);
	}
	
	/* Did we timeout, or actually get an entire packet, or something else? */
	if (ret != PHIDGETS_RFID_GET_PACKETLEN) {
		NOTICE("Read Error: %i", ret);
		if (ret == -110)
			return PHIDGET_RET_TIMEOUT;
		return PHIDGET_RET_HID_ERROR;
	}
	
	/* Convert tag to a long long */
	qsc->l_tag = 0;
	for(i = 1; i <= 5; i++) {
		qsc->l_tag <<= 8;
		qsc->l_tag |= (0xFF & packet[i]);
	}
	
	/* Set timestamp */
	qsc->time = time(NULL);
	
	TRACE("receieved tag: [ %02hhx %02hhx %02hhx %02hhx %02hhx ]",
		packet[1], packet[2], packet[3], packet[4], packet[5]);
	
	return PHIDGET_RET_SUCCESS;
}
