/* main.h,v 1.12 2009/11/27 01:12:37 lacos Exp */

#ifndef MAIN_H
#  define MAIN_H

#  include <limits.h>  /* CHAR_BIT */
#  include <stddef.h>  /* size_t */
#  include <pthread.h> /* pthread_mutex_t */

#  if 8 != CHAR_BIT
#    error "Environments where 8 != CHAR_BIT are not supported."
#  endif


/* Utilities that can be called/accessed from multiple threads. */

/*
  (I) The treatment for fatal errors.

  bailout() doesn't need to be extern. If called from the main thread, remove
  any current output file and bail out. If called (indirectly) from any other
  thread, send SIGUSR1 to the process, in order to signal the fatal error in
  the sub-thread, then terminate the thread.
*/


/* (II) Logging utilities. */

/*
  Name of the executable, for logging purposes. Set up by the main thread
  before starting any threads.
*/
extern const char *pname;

/*
  Return a static string corresponding to the error number. strerror_r() is not
  SUSv2.
*/
const char *
err2str(int err);

/*
  Format a message to standard error. If the underlying vfprintf() fails, call
  bailout().
*/
void
log_info(const char *fmt, ...)
#ifdef __GNUC__
__attribute__((format(printf, 1, 2)))
#endif
;

/* Same as log_info(), but always call bailout(). */
void
log_fatal(const char *fmt, ...)
#ifdef __GNUC__
__attribute__((format(printf, 1, 2)))
#endif
;


/* (III) Memory allocation. */

/* Allocate memory. If unsuccessful, call log_fatal(). */
void *
xalloc(size_t size);

/*
  Allocation function pointers, set up by the main thread before starting any
  threads. They may call log_info().
*/
extern void (*freef)(void *ptr),
    *(*lbzallocf)(void *ignored, int n, int m),
    (*lbzfreef)(void *ignored, void *ptr);


/*
  (IV) Threading utilities. If they fail, they call log_fatal().
*/

struct cond
{
  pthread_mutex_t lock; /* Lock this to protect shared resource. */
  pthread_cond_t cond;  /* Trigger this if predicate becomes true. */
  long unsigned ccount, /* Increment this when checking predicate. */
      wcount;           /* Increment this when waiting is necessary. */
};

void
xinit(struct cond *cond);

void
xdestroy(struct cond *cond);

void
xlock(struct cond *cond);

void
xlock_pred(struct cond *cond);

void
xunlock(struct cond *cond);

void
xwait(struct cond *cond);

void
xsignal(struct cond *cond);

void
xbroadcast(struct cond *cond);

void
xcreate(pthread_t *thread, void *(*routine)(void *), void *arg);

void
xjoin(pthread_t thread);

void
xraise(int sig);

#endif
