// Copyright(C) 2005,2006,2007 Stefan Siegl <stesie@brokenpipe.de>
// kopete_silc - silc plugin for kopete messenger
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include <iostream>

//#undef KDE_NO_COMPAT
#include <kxmlguiclient.h>
#include <kgenericfactory.h>

#include <kopeteonlinestatus.h>
#include <kopeteonlinestatusmanager.h>
#include <kopeteaccountmanager.h>
#include <kopetecommandhandler.h>
#include <kopetechatsession.h>
#include <kopetemessage.h>

#include "silcprotocol.h"
#include "silcaddcontactpage.h"
#include "silceditaccountwidget.h"
#include "silcchannelcontact.h"
#include "silcbuddycontact.h"
#include "silcaccount.h"

typedef KGenericFactory<SilcProtocol> SilcProtocolFactory;
K_EXPORT_COMPONENT_FACTORY(kopete_silc, SilcProtocolFactory("kopete_silc"));

SilcProtocol *SilcProtocol::_protocol = NULL;

SilcProtocol::SilcProtocol(QObject *parent, const char *name,
			   const QStringList & /* args */)
  : Kopete::Protocol(SilcProtocolFactory::instance(), parent, name),

    // register online statuses ...
    statusOnline(Kopete::OnlineStatus::Online, Online, this,
		 Online, QString::null, i18n("Online"),
		 i18n("Online"), Kopete::OnlineStatusManager::Online),
    statusOnlineMute(Kopete::OnlineStatus::Online, OnlineMute, this,
		     OnlineMute, "silc_mute", i18n("Mute"),
		     i18n("Online, Mute"),
		     Kopete::OnlineStatusManager::Online,
		     Kopete::OnlineStatusManager::HideFromMenu),
    statusOnlineOpMute(Kopete::OnlineStatus::Online, OnlineOpMute, this,
		       OnlineOpMute,
		       QStringList::split(' ', "silc_op silc_mute"),
		       i18n("Op/Mute"), i18n("Online, Op, Mute"),
		       Kopete::OnlineStatusManager::Online,
		       Kopete::OnlineStatusManager::HideFromMenu),
    statusOnlineOp(Kopete::OnlineStatus::Online, OnlineOp, this,
		   OnlineOp, "silc_op", i18n("Op"), i18n("Online, Op"),
		   Kopete::OnlineStatusManager::Online,
		   Kopete::OnlineStatusManager::HideFromMenu),

    statusOnlineChannel(Kopete::OnlineStatus::Online, OnlineChannel, this,
			OnlineChannel, "silc_channel", i18n("Online"),
			i18n("Online"), Kopete::OnlineStatusManager::Online,
			Kopete::OnlineStatusManager::HideFromMenu),

    statusConnecting(Kopete::OnlineStatus::Connecting, Connecting, this,
		     Connecting, "silc_channel", i18n("Connecting"),
		     i18n("Connecting"),Kopete::OnlineStatusManager::Offline,
		     Kopete::OnlineStatusManager::HideFromMenu),
			

    statusGone(Kopete::OnlineStatus::Away, Gone, this,
	       Gone, "silc_away", 
	       i18n("Gone"), i18n("Gone"), Kopete::OnlineStatusManager::Away),
    statusGoneOp(Kopete::OnlineStatus::Away, GoneOp, this, GoneOp,
		 QStringList::split(' ', "silc_op silc_away"), 
		 i18n("Gone, Op"), i18n("Gone, Op"),
		 Kopete::OnlineStatusManager::Away,
		 Kopete::OnlineStatusManager::HideFromMenu),

    statusDetached(Kopete::OnlineStatus::Away, Detached, this,
		   Gone, "silc_detached", 
		   i18n("Detached"), i18n("Detached"),
		   Kopete::OnlineStatusManager::Away,
		   Kopete::OnlineStatusManager::HideFromMenu),

    statusIndisposed(Kopete::OnlineStatus::Away, Indisposed, this,
		     Indisposed, "silc_indisposed", 
		     i18n("Indisposed"), i18n("Indisposed"),
		     Kopete::OnlineStatusManager::Away),
    statusIndisposedOp(Kopete::OnlineStatus::Away,IndisposedOp, this,
		       IndisposedOp, 
		       QStringList::split(' ', "silc_indisposed silc_op"),
		       i18n("Indisposed, Op"), i18n("Indisposed, Op"),
		       Kopete::OnlineStatusManager::Away,
		       Kopete::OnlineStatusManager::HideFromMenu),

    statusBusy(Kopete::OnlineStatus::Away, Busy, this, Busy, "silc_busy", 
	       i18n("Busy"), i18n("Busy"), Kopete::OnlineStatusManager::Away),
    statusBusyOp(Kopete::OnlineStatus::Away, BusyOp, this, BusyOp,
		 QStringList::split(' ', "silc_op silc_busy"), 
		 i18n("Busy, Op"), i18n("Busy, Op"),
		 Kopete::OnlineStatusManager::Away,
		 Kopete::OnlineStatusManager::HideFromMenu),

    statusHyper(Kopete::OnlineStatus::Online, Hyper, this, Hyper,
		"silc_hyper", i18n("Hyper Active"), i18n("Hyper Active"),
		Kopete::OnlineStatusManager::Online),
    statusHyperMute(Kopete::OnlineStatus::Online, HyperMute, this, HyperMute,
		    QStringList::split(' ', "silc_hyper silc_mute"), 
		    i18n("Hyper Active, Mute"), i18n("Hyper Active, Mute"),
		    Kopete::OnlineStatusManager::Online,
		    Kopete::OnlineStatusManager::HideFromMenu),
    statusHyperOpMute(Kopete::OnlineStatus::Online, HyperOpMute, this,
		      HyperOpMute,
		      QStringList::split(' ', "silc_hyper silc_op silc_mute"), 
		      i18n("Hyper Active, Op, Mute"),
		      i18n("Hyper Active, Op, Mute"),
		      Kopete::OnlineStatusManager::Online,
		      Kopete::OnlineStatusManager::HideFromMenu),
    statusHyperOp(Kopete::OnlineStatus::Online, HyperOp, this, HyperOp,
		  QStringList::split(' ', "silc_hyper silc_op"), 
		  i18n("Hyper Active, Op"), i18n("Hyper Active, Op"),
		  Kopete::OnlineStatusManager::Online,
		  Kopete::OnlineStatusManager::HideFromMenu),

    statusOffline(Kopete::OnlineStatus::Offline, 0, this,
		  Offline, QString::null,
		  i18n("Offline"), i18n("Offline"),
		  Kopete::OnlineStatusManager::Offline)
{
  _protocol = this;
  setCapabilities(FullRTF);

  // add our i18n catalogue ...
  KGlobal::locale()->insertCatalogue("kopete_silc");

  // register silc-protocol's commands (at least these we do support) ...
  Kopete::CommandHandler::commandHandler()->registerCommand
    (this, QString("whois"),
     SLOT(slotWhoisCommand(const QString&, Kopete::ChatSession *)),
     i18n("USAGE: /whois <nickname> - request whois information."), 1);

  Kopete::CommandHandler::commandHandler()->registerCommand
    (this, QString("watch"),
     SLOT(slotWatchCommand(const QString&, Kopete::ChatSession *)),
     i18n("USAGE: /watch -add|-del <nickname> - watch a nickname."), 1);
  
  Kopete::CommandHandler::commandHandler()->registerCommand
    (this, QString("notice"),
     SLOT(slotNoticeCommand(const QString &, Kopete::ChatSession *)),
     i18n("USAGE: /notice <message> - send a notice message."), 1);

  Kopete::CommandHandler::commandHandler()->registerCommand
    (this, QString("me"),
     SLOT(slotAction(const QString &, Kopete::ChatSession *)),
     i18n("USAGE: /me <action> - send an action message."), 1);
  
  Kopete::CommandHandler::commandHandler()->registerCommand
    (this, QString("getkey"),
     SLOT(slotGetkeyCommand(const QString&, Kopete::ChatSession *)),
     i18n("USAGE: /getkey <nickname> - fetch remote client's public key"), 1);
  Kopete::CommandHandler::commandHandler()->registerCommand
    (this, QString("killme"),
    SLOT(slotKillmeCommand(const QString&, Kopete::ChatSession *)),
    i18n("USAGE: /killme"), 0);
}



SilcProtocol::~SilcProtocol()
{
  _protocol = NULL;
}



Kopete::Account *
SilcProtocol::createNewAccount(const QString &accountId)
{
  return new SilcAccount(this, accountId);
}



AddContactPage *
SilcProtocol::createAddContactWidget(QWidget *parent, Kopete::Account *account)
{
  return new SilcAddContactPage(parent, static_cast<SilcAccount *>(account));
}




KopeteEditAccountWidget *
SilcProtocol::createEditAccountWidget(Kopete::Account *account,
				      QWidget *parent)
{
  return new SilcEditAccountWidget(this, static_cast<SilcAccount *>(account),
				   parent);
}


SilcProtocol *
SilcProtocol::protocol(void) 
{
  return _protocol;
}

Kopete::Contact *
SilcProtocol::deserializeContact(Kopete::MetaContact *meta,
				 const QMap<QString, QString> &serializedData, 
				 const QMap<QString, QString> &)
{
  QString contactId = serializedData[ "contactId" ];

  QDict<Kopete::Account> accounts = 
    Kopete::AccountManager::self()->accounts(this);

  if(accounts.isEmpty()) {
    std::cerr << "No accounts loaded, configuration invalid." << std::endl;
    return NULL;
  }

  Kopete::Account *a = accounts[serializedData["accountId"]];
  if(! a) {
    std::cerr << "Account " << serializedData["accountId"]
	      << "used to be available, but isn't anymore" << std::endl;
    return NULL;
  }

  if(! a->addContact(contactId, meta))
    return NULL; 

  Kopete::Contact *contact = a->contacts()[contactId];
  if(! strcmp(contact->className(), "SilcBuddyContact")) {
    SilcBuddyContact *buddy = static_cast<SilcBuddyContact *>(contact);
    buddy->setFpTrusted(serializedData["fpTrusted"].compare("yes") == 0);
    buddy->setAllowRichText(serializedData["allowRichText"]
			    .compare("yes") == 0);
  } 
  else if(! strcmp(contact->className(), "SilcChannelContact")) {
    SilcChannelContact *channel = static_cast<SilcChannelContact *>(contact);
    channel->setAllowRichText(serializedData["allowRichText"]
			      .compare("yes") == 0);
  }

  return contact;
}

/**
 * sends a WHOIS request to the network for the given nick
 *
 * @todo if possible the public key should be used.
 */

void
SilcProtocol::slotWhoisCommand(const QString &nick, Kopete::ChatSession *cs)
{
  SilcAccount *account = static_cast<SilcAccount *>(cs->account());
  account->sendSilcCommand(QString("WHOIS %1").arg(nick));
}

void
SilcProtocol::slotKillmeCommand(const QString &, Kopete::ChatSession *cs)
{
  SilcAccount *account = static_cast<SilcAccount *>(cs->account());
  account->sendSilcCommand(QString("KILL %1 -pubkey").arg(account->nickName()));
}

void
SilcProtocol::slotWatchCommand(const QString &cmd, Kopete::ChatSession *cs)
{    
  SilcAccount *account = static_cast<SilcAccount *>(cs->account());
  if(cmd.contains("-add ") || cmd.contains("-del ") || cmd.contains("-pubkey "))
    account->sendSilcCommand(QString("WATCH %1").arg(cmd));
  // @todo emit error message in case of error?
}

void
SilcProtocol::slotNoticeCommand(const QString &notice, Kopete::ChatSession *cs)
{
  SilcAccount *account = static_cast<SilcAccount *>(cs->account());
  Kopete::ContactPtrList members = cs->members();
  Kopete::Contact *contact = members.first();

  QCString text = notice.utf8();
  SilcTK::SilcMessageFlags flags =
    SILC_MESSAGE_FLAG_NOTICE | SILC_MESSAGE_FLAG_UTF8;

  if(! strcmp(contact->className(), "SilcChannelContact")) {
    // send action to channel ...
    SilcChannelContact *channel = static_cast<SilcChannelContact *>(contact);

    // @fixme: add signChannelNotices() method
    //if(account->signChannelActions())
    //  flags |= SILC_MESSAGE_FLAG_SIGNED;

    SilcTK::silc_client_send_channel_message
      (account->client(), account->conn(), channel->channelEntry(), NULL,
        flags, account->sha1hash, (unsigned char *) (const char *) text, 
        text.length());

  }
  else if(! strcmp(contact->className(), "SilcBuddyContact")) {
    // send action to buddy ...
    SilcBuddyContact *buddy = static_cast<SilcBuddyContact *>(contact);

    // @fixme add signPrivateNotices() method
    //if(account->signPrivateActions())
    //  flags |= SILC_MESSAGE_FLAG_SIGNED;

    SilcTK::silc_client_send_private_message
      (account->client(), account->conn(), buddy->clientEntry(), flags,
       account->sha1hash, (unsigned char *) (const char *) text, 
       text.length());     
  }
  else 
    return; // ignore for SilcServerContact ...

  Kopete::Message msg(account->myself(), cs->members(),
		      notice, 
		      Kopete::Message::Internal, 
		      Kopete::Message::PlainText, QString::null,
		      Kopete::Message::TypeAction);
  SilcContact::prettyPrintMessage(msg, flags);

  cs->appendMessage(msg);
}

void
SilcProtocol::slotGetkeyCommand(const QString &nick, Kopete::ChatSession *cs)
{
  SilcAccount *account = static_cast<SilcAccount *>(cs->account());
  account->sendSilcCommand(QString("GETKEY %1").arg(nick));
}

void
SilcProtocol::slotAction(const QString &action, Kopete::ChatSession *cs)
{
  SilcAccount *account = static_cast<SilcAccount *>(cs->account());
  Kopete::ContactPtrList members = cs->members();
  Kopete::Contact *contact = members.first();

  QCString text = action.utf8();
  SilcTK::SilcMessageFlags flags =
    SILC_MESSAGE_FLAG_ACTION | SILC_MESSAGE_FLAG_UTF8;

  if(! strcmp(contact->className(), "SilcChannelContact")) {
    // send action to channel ...
    SilcChannelContact *channel = static_cast<SilcChannelContact *>(contact);

    if(account->signChannelActions())
      flags |= SILC_MESSAGE_FLAG_SIGNED;

    SilcTK::silc_client_send_channel_message
      (account->client(), account->conn(), channel->channelEntry(), NULL,
       flags, account->sha1hash, (unsigned char *) (const char *) text, 
       text.length());


  }
  else if(! strcmp(contact->className(), "SilcBuddyContact")) {
    // send action to buddy ...
    SilcBuddyContact *buddy = static_cast<SilcBuddyContact *>(contact);

    if(account->signPrivateActions())
      flags |= SILC_MESSAGE_FLAG_SIGNED;

    SilcTK::silc_client_send_private_message
      (account->client(), account->conn(), buddy->clientEntry(), flags,
       account->sha1hash, (unsigned char *) (const char *) text, 
       text.length());     
  }
  else 
    return; // ignore for SilcServerContact ...

  Kopete::Message msg(account->myself(), cs->members(), action, 
		      Kopete::Message::Outbound, 
		      Kopete::Message::PlainText, QString::null,
		      Kopete::Message::TypeAction);
  SilcContact::prettyPrintMessage(msg, flags);

  cs->appendMessage(msg);

}

#include "silcprotocol.moc"
