/***************************************************************************
                          igdcontrolpoint.cpp -  description
                             -------------------
    begin                : Mon Jul 25 2005
    copyright            : (C) 2005 by Diederik van der Boor
    email                : vdboor --at-- codingdomain.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "igdcontrolpoint.h"

#include "rootservice.h"
#include "layer3forwardingservice.h"
#include "wanconnectionservice.h"

#include <kdebug.h>
#include "../../kmessdebug.h"

#ifdef KMESSDEBUG_UPNP
  #define KMESSDEBUG_UPNP_GENERAL
#endif


namespace UPnP
{

#define InternetGatewayDeviceType "urn:schemas-upnp-org:device:InternetGatewayDevice:1"
#define Layer3ForwardingType      "urn:schemas-upnp-org:service:Layer3Forwarding:1"


// The constructor
IgdControlPoint::IgdControlPoint(const QString &hostname, int port, const QString &rootUrl)
  : QObject()
  , forwardingService_(0)
  , gatewayAvailable_(false)
  , igdPort_(0)
  , rootService_(0)
  , wanConnectionService_(0)
{
#ifdef KMESSDEBUG_UPNP_GENERAL
  kdDebug() << "CREATED UPnP::IgdControlPoint: Created control point"
            << " url='" << hostname << ":" << port << "/" << rootUrl << "'." << endl;
  kdDebug() << "UPnP::IgdControlPoint: querying services..." << endl;
#endif

  // Store device url
  igdHostname_ = hostname;
  igdPort_     = port;

  // Query the device for it's services
  rootService_ = new RootService(igdHostname_, igdPort_, rootUrl);
  connect(rootService_, SIGNAL(queryFinished(bool)), this, SLOT(slotDeviceQueried(bool)));
}



// The destructor
IgdControlPoint::~IgdControlPoint()
{
  delete rootService_;
  delete forwardingService_;
  delete wanConnectionService_;

#ifdef KMESSDEBUG_UPNP_GENERAL
  kdDebug() << "DESTROYED UPnP::IgdControlPoint [host=" << igdHostname_ << ", port=" << igdPort_ << "]" << endl;
#endif
}



// Return the external IP address
QString IgdControlPoint::getExternalIpAddress() const
{
  // Do not expose  wanConnectionService_;
  if(wanConnectionService_ != 0)
  {
    return wanConnectionService_->getExternalIpAddress();
  }
  else
  {
    return QString::null;
  }
}



// Initialize the control point
void IgdControlPoint::initialize()
{
#ifdef KMESSTEST
  ASSERT( rootService_ != 0 );
  ASSERT( rootService_->getDeviceType().isNull() );
#endif

  rootService_->queryDevice();
}



// Return true if a controlable gateway is available
bool IgdControlPoint::isGatewayAvailable()
{
  return gatewayAvailable_;
}



// The IGD was queried for it's services
void IgdControlPoint::slotDeviceQueried(bool error)
{
#ifdef KMESSTEST
  ASSERT( rootService_          != 0 );
  ASSERT( forwardingService_    == 0 );
#endif

  if(! error)
  {
#ifdef KMESSTEST
    ASSERT( rootService_->getDeviceType() == InternetGatewayDeviceType );
#endif

    // Get the Layer3ForwardingService from the retrieved service list
    ServiceParameters params = rootService_->getServiceByType(Layer3ForwardingType);

    if(! params.controlUrl.isNull())
    {
#ifdef KMESSDEBUG_UPNP_GENERAL
      kdDebug() << "UPnP::IgdControlPoint: Services found, "
                << "querying service '" << params.serviceId << "' for port mapping service..." << endl;
#endif

      // Call the service
      forwardingService_ = new Layer3ForwardingService(params);
      connect(forwardingService_, SIGNAL(queryFinished(bool)), this, SLOT(slotWanConnectionFound(bool)));
      forwardingService_->queryDefaultConnectionService();
    }
    else
    {
      // TODO: error
    }
  }
}



// A WAN connection service was found
void IgdControlPoint::slotWanConnectionFound(bool error)
{
#ifdef KMESSTEST
  ASSERT( rootService_          != 0 );
  ASSERT( forwardingService_    != 0 );
  ASSERT( wanConnectionService_ == 0 );
#endif

  if(! error)
  {
    // Get the retreived service description
    QString deviceUrn = forwardingService_->getConnectionDeviceUdn();
    QString serviceId = forwardingService_->getConnectionServiceId();
    ServiceParameters params = rootService_->getServiceById(serviceId, deviceUrn);

    if(! params.controlUrl.isNull())
    {
#ifdef KMESSDEBUG_UPNP_GENERAL
      kdDebug() << "UPnP::IgdControlPoint: wan/ipconnection service found, "
                << "querying service '" << params.serviceId << "' for external ip address..." << endl;
#endif

      // Call the service
      wanConnectionService_ = new WanConnectionService(params);
      connect(wanConnectionService_, SIGNAL(queryFinished(bool)), this, SLOT(slotWanQueryFinished(bool)));
      wanConnectionService_->queryExternalIpAddress();
    }
  }

  // No longer need the forwarding service
  forwardingService_->deleteLater();
  forwardingService_ = 0;
}



// A WAN connection query was finished
void IgdControlPoint::slotWanQueryFinished(bool error)
{
#ifdef KMESSTEST
  ASSERT( rootService_          != 0 );
  ASSERT( wanConnectionService_ != 0 );
#endif

  if(! error)
  {
#ifdef KMESSDEBUG_UPNP_GENERAL
    kdDebug() << "IgdControlPoint: UPnP Gateway Device found." << endl;
#endif
    gatewayAvailable_ = true;
  }
  else
  {
    // Just started, the request for the external IP failed. This should succeed, abort portation
    kdWarning() << "Requesting external IP address failed, leaving UPnP Gateway Device untouched." << endl;
  }
}



}  // End of namespace

#include "igdcontrolpoint.moc"
