/***************************************************************************
                          transferwindow.cpp -  description
                             -------------------
    begin                : Wed Nov 3 2004
    copyright            : (C) 2004 by Diederik van der Boor
    email                : vdboor --at-- codingdomain.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "transferwindow.h"
#include "transferentry.h"
#include "transferwindowinterface.h"
#include "../kmessdebug.h"

#include <klocale.h>

#include <qlayout.h>
#include <qptrlist.h>
#include <qpushbutton.h>
#include <qscrollview.h>


// Initialize the entries counter
int             TransferWindow::currentEntryID_(-1);
// Initialize the instance to zero
TransferWindow* TransferWindow::onlyInstance_(0);



// Constructor
TransferWindow::TransferWindow( QWidget *parent )
  : KMainWindow( parent, "TransferWindow", WType_Dialog ) // Explicitly set the window type to disable
                                                          // the destructive close behavior
{
  // Create the interface and connect it to our event handlers
  interface_ = new TransferWindowInterface( this );

  connect( interface_->closeButton_,   SIGNAL( clicked() ),
           this,                         SLOT(    hide() ) );
  connect( interface_->cleanUpButton_, SIGNAL( clicked() ),
           this,                         SLOT( cleanUp() ) );

  // Create a layout to expand the scrollview to the maximum space available
  QVBoxLayout *frameLayout = new QVBoxLayout( interface_->frame_, 2, 0 );

  // Make the central list for entries, capable of scrolling vertically
  QScrollView *scrollView = new QScrollView( interface_->frame_ );
  scrollView->viewport()->setPaletteBackgroundColor( paletteBackgroundColor() );
  scrollView->setHScrollBarMode( QScrollView::AlwaysOff );    // Disable horizontal scrolllbar
  scrollView->setResizePolicy( QScrollView::AutoOneFit );     // Stretch contents
  scrollView->setSizePolicy( QSizePolicy::Expanding, QSizePolicy::Expanding, false );  // Stretch widget

  // Add it to the layout
  frameLayout->addWidget( scrollView );

  // Create a container for all download entries, to lay them out vertically
  mainWidgets_ = new QWidget( scrollView->viewport() );
  mainWidgets_->setSizePolicy( QSizePolicy::Maximum, QSizePolicy::Preferred, false );  // Stretch widget
  scrollView->addChild( mainWidgets_ );

  // Create the layout and a spacer, to sort out the individual transfers
  mainLayout_ = new QVBoxLayout( mainWidgets_, 2, 0 );
  mainLayout_->addItem( new QSpacerItem( 20, 0, QSizePolicy::Ignored, QSizePolicy::Ignored ) );

  // Set up the window
  setCaption( i18n("File Transfers") );
  setCentralWidget( interface_ );
}



// Destructor
TransferWindow::~TransferWindow()
{
#ifdef KMESSDEBUG_TRANSFERWINDOW
  kdDebug() << "DESTROYED TransferWindow" << endl;
#endif

  // Clean up the list of transfers
  int id;
  TransferEntry *entry;
  QMapConstIterator<int,TransferEntry*> it;
  for( it = entryList_.begin(); it != entryList_.end(); )
  {
    // Copy the data before deleting, to avoid crashes
    id = it.key();
    entry = it.data();
    ++it;

    // Delete the entry from the GUI and our list
    mainLayout_->remove( entry );
    entryList_.remove( id );
    delete entry;
  }

  // Destroy the instance data
  delete mainLayout_;
  delete interface_;
  onlyInstance_ = 0;
}



// Add an entry to the transfer manager
int TransferWindow::addEntry( const QString filename, uint filesize, bool incoming, const QImage preview )
{
#ifdef KMESSDEBUG_TRANSFERWINDOW
  kdDebug() << "TransferWindow::addEntry() - adding entry for '" << filename << "'." << endl;
#endif

  // Create a new entry
  TransferEntry *entry = new TransferEntry( mainWidgets_, filename, filesize, incoming, preview );

  // Add it to the transfers list
  entryList_[ ++currentEntryID_ ] = entry;
  entry->setTransferID( currentEntryID_ );

  // Connect the dialog so that if the user closes it, it's deleted.
  connect( entry, SIGNAL( cancelTransfer(int) ) ,
           this,  SIGNAL( cancelTransfer(int) ) );

#ifdef KMESSDEBUG_TRANSFERWINDOW
  kdDebug() << "TransferWindow::addEntry() - Adding entry to layout, showing it." << endl;
#endif

  // Add it to the list of entries
  mainLayout_->insertWidget( 0, entry );
  entry->show();

  // Update and show the window
  update();
  show();

  return currentEntryID_;
}



// Mark a transfer as failed
void TransferWindow::failTransfer( int transferID, const QString &message )
{
  if( ! entryList_.contains( transferID ) )
  {
    return;
  }

  entryList_[ transferID ]->failTransfer( message );
}



// Mark a transfer as complete
void TransferWindow::finishTransfer( int transferID )
{
  if( ! entryList_.contains( transferID ) )
  {
    return;
  }

  entryList_[ transferID ]->finishTransfer();
}



// Set the status message of a transfer
void TransferWindow::setStatusMessage( int transferID, const QString &message )
{
  if( ! entryList_.contains( transferID ) )
  {
    return;
  }

  entryList_[ transferID ]->setStatusMessage( message );
}



// Update the progress bar of a transfer
void TransferWindow::updateProgress( int transferID, uint bytesTransferred )
{
  if( ! entryList_.contains( transferID ) )
  {
    return;
  }

  entryList_[ transferID ]->updateProgress( bytesTransferred );
}



// Return an instance of the singleton. It's not named instance() due to the base class having an instance() method itself.
TransferWindow *TransferWindow::getInstance()
{
  if(onlyInstance_ == 0)
  {
    onlyInstance_ = new TransferWindow();
  }
  return onlyInstance_;
}



// Destroy the singleton.
void TransferWindow::destroy()
{
  delete onlyInstance_;
  onlyInstance_ = 0;
}



// The "Clean up" button was pressed
void TransferWindow::cleanUp()
{
#ifdef KMESSDEBUG_TRANSFERWINDOW
  kdDebug() << "TransferWindow::cleanUp() - Cleaning up" << endl;
#endif

  int id;
  TransferEntry *entry;

  // Remove the entries from the dialog
  QMapConstIterator<int,TransferEntry*> it;
  for( it = entryList_.begin(); it != entryList_.end(); )
  {
    // Copy the data before deleting, to avoid crashes
    id = it.key();
    entry = it.data();
    ++it;

    // Remove the entry only if it's an ended transfer and we don't need it anymore
    if( entry->isDone() )
    {
      // Delete the entry from the GUI and our list
      mainLayout_->remove( entry );
      entryList_.remove( id );

#ifdef KMESSDEBUG_TRANSFERWINDOW
      kdDebug() << "TransferWindow::cleanUp() - Removed " << entry->getFileName() << endl;
#endif

      delete entry;
    }
  }

  // Issue an update to the UI, just to be sure
  interface_->update();
}



#include "transferwindow.moc"
