/***************************************************************************
                          chatmaster.h  -  description
                             -------------------
    begin                : Sat Jan 18 2003
    copyright            : (C) 2003 by Mike K. Bennett
    email                : mkb137b@hotmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef CHATMASTER_H
#define CHATMASTER_H

#include "chatmessage.h"
#include "../network/chatinformation.h"
#include "../network/mimemessage.h"

#include <qptrlist.h>
#include <qwidget.h>

// Forward declarations
class Application;
class ApplicationList;
class ChatInformation;
class ChatMessage;
class ChatWindow;
class Contact;
class ContactBase;
class MimeApplication;
class MimeMessage;
class P2PApplication;
class P2PMessage;
class MsnObject;
class MsnSwitchboardConnection;

/**
 * This class governs the chat windows, detecting when a chat
 * can be restarted in an open window and the like.
 * @author Mike K. Bennett
 */
class ChatMaster : public QObject
{
  Q_OBJECT

  friend class KMessTest;

  public: // Public methods
    // The constructor
                       ChatMaster(QObject *parent = 0);
    // The destructor
    virtual           ~ChatMaster();
    // Initialize the class
    bool               initialize();

  public slots: // Public slots
    // The user has disconnected, so close all open chats
    void               disconnected();
    // Raise an existing chat window
    void               raiseChat( ChatWindow *chatWindow, bool force = false );
    // Verify if a new chat can be opened and requests a switchboard
    void               requestChat( QString handle );
    // Deliver an application or offline message to the chat window (can be submitted by an Application or Offline-IM from MsnNotificationConnection).
    void               showSpecialMessage(const ChatMessage &message);
    // Start a switchboard connection with the information gathered from the Notification connection
    MsnSwitchboardConnection *startSwitchboard( const ChatInformation &chatInfo );
    // Start a file transfer with the information from the ChatWindow
    void               startFileTransfer( const QString &handle, const QString &filename );
    // Periodically update events. It's based on the pings sent by MsnNotificationConnection.
    void               timedUpdate();

  private slots: // Private slots
    // Create and initialize a new chat window.
    ChatWindow        *createChatWindow( MsnSwitchboardConnection *switchboard );
    // Forward the signal of a new incoming message from a chat window
    void               forwardNewChatMessage( const ChatMessage &message, ChatWindow *chatWindow );
    // Forward a new switchboard server request signal coming from an existing switchboard connection
    void               forwardRequestNewSwitchboard( QString handle );
    // Forward a request to add or remove a contact from the contactlist, which comes from the ContactFrame.
    void               forwardSetContactAdded( QString handle, bool isAdded );
    // Forward a request to allow a contact, which comes from the ContactFrame.
    void               forwardSetContactAllowed( QString handle );
    // Forward a request to block or unblock a contact, which comes from the ContactFrame.
    void               forwardSetContactBlocked( QString handle, bool isBlocked );
    // A chat window is closing
    void               slotChatWindowClosing(QObject *chatWindow);
    // A chat window was destroyed
    void               slotChatWindowDestroyed(QObject *chatWindow);
    // A new application was created for a contact.
    void               slotConnectApplication(Application *application);
    // Determine what to do when a contact changed it's picture.
    void               slotContactChangedMsnObject( Contact *contact );
    // A contact joined to one of our switchboard sessions.
    void               slotContactJoinedChat(QString handle);
    // Deliver an application command to the correct application
    void               slotDeliverAppCommand(QString cookie, QString handle, QString command);
    // Deliver a message from a Application to the switchboard connection.
    void               slotDeliverMimeMessage(const MimeMessage &message, const QString &handle, bool privateChatRequired);
    // The switchboard received a Mime message
    void               slotGotMessage(const MimeMessage &message, const QString &handle);
    // The switchboard received a P2P message
    void               slotGotMessage(const P2PMessage &message, const QString &handle);
    // The switchboard received an msn object
    void               slotGotMsnObject(const QString &msnObjectData, const QString &handle);
    // A msn object (picture, wink, emoticon) was received for the contact.
    void               slotMsnObjectReceived(const QString &handle, const MsnObject &msnObject);
    // Delete an existing switchboard
    void               slotSwitchboardDelete( MsnSwitchboardConnection *closing );
    // The switchboard is ready to send more messages.
    void               slotSwitchboardReady();
    // Start a netmeeting invitation
    void               startNetMeeting(const QString &handle);

  private:  // private methods
    // Check whether the contact is in any of the existing chat windows.
    bool               isContactInChat( const QString &handle );
    // Display an MSN Object that was received.
    void               showMsnObject(const QString &handle, const MsnObject &msnObject, ChatWindow *chatWindow);
    // Configure and start the mime application object.
    void               startApplication( MimeApplication *application );
    // Configure and start the P2P application object.
    void               startApplication( P2PApplication *application );
    // Start the transfer of the contact's picture
    void               startMsnObjectDownload(const QString &handle, const MsnObject *msnObject, ChatWindow *chatWindow);
    // Return the application list for a given contact
    ApplicationList   *getApplicationList(const QString &handle);
    // Return the chat window which uses the given switchboard connection
    ChatWindow        *getChatWindowBySwitchboard(const MsnSwitchboardConnection *connection);
    // Return the chat window where we're having an conversation with the given contact.
    ChatWindow        *getContactChatWindow(const QString &handle, bool privateChat);
    // Return the chat window where we're having a conversation with the given contacts.
    ChatWindow        *getContactsChatWindow(const QStringList &handles);
    // Return the chat connection where we're having an conversation with the given contact.
    MsnSwitchboardConnection * getContactSwitchboardConnection(const QString &handle, bool privateChat);
    // Append the message to the queue, waiting to be delivered
    void               queueMessage( const MimeMessage &message, const QString &handle, bool privateChatRequired );
    // Send all pending mime messages for the contact.
    void               sendPendingMimeMessages( const QString &handle, MsnSwitchboardConnection *connection );

  private: // Private attributes
    struct PendingMimeMessage
    {
      QString     handle;
      MimeMessage message;
      bool        privateChatRequired;
    };

    // The pointers to the chat windows
    QPtrList<ChatWindow> chatWindows_;
    // The pointers of chat windows that are closing
    QPtrList<ChatWindow> closingChatWindows_;
    // Whether or not the object was initialized
    bool               initialized_;
    // The list of contacts whose DPs must be downloaded
    QStringList        pendingDisplayPictures_;
    // A list of mime messages that should be sent to the contact when connection is reestablished
    QPtrList< PendingMimeMessage > pendingMimeMessages_;
    // The pointers to the switchboard server connections
    QPtrList<MsnSwitchboardConnection> switchboardConnections_;

  signals: // Public signals
    // Ask the Notification connection to add a contact
    void               addContact( QString handle );
    // Ask the Notification connection to allow a contact
    void               allowContact( QString handle );
    // Ask the Notification connection to block a contact
    void               blockContact( QString handle );
    // Signal the presence of a new incoming chat message
    void               newChatMessage( const ChatMessage &message, ChatWindow *chatWindow );
    // Ask the Notification connection to remove a contact
    void               removeContact( QString handle, bool block );
    // Ask the Notification connection to initiate a new switchboard connection with the contact
    void               requestSwitchboard( QString handle, ChatInformation::ConnectionType type );
    // Ask the Notification connection to unblock a contact
    void               unblockContact( QString handle );
};

#endif
