
#include <stdlib.h>
#include <sys/stat.h>
#include <pwd.h>
#include <grp.h>

#include <kdebug.h>
#include <kinstance.h>
#include <klocale.h>
#include <kuser.h>
#include <qfile.h>

#include "beagk.h"

using namespace KIO;

// from kio_locate
void addAtom(UDSEntry& entry, unsigned int uds, const QString& s)
{
    UDSAtom a;
    a.m_uds = uds;
    a.m_str = s;
    entry.append(a);
}


void addAtom(UDSEntry& entry, unsigned int uds, long l)
{
    UDSAtom a;
    a.m_uds = uds;
    a.m_long = l;
    entry.append(a);
}

const UDSEntry resultToUDSEntry (QString &, beagle_result_struct *);

const UDSEntry pathToUDSEntry(const QString& path, const QString& display = QString::null,
    const QString& url = QString::null, const QString& mime_type = QString::null,
	HitType type = Alien_type)
{
    UDSEntry entry;
    addAtom(entry, KIO::UDS_NAME, path);

    struct stat info;
    lstat(path, &info);

    addAtom(entry, KIO::UDS_SIZE, info.st_size);
    addAtom(entry, KIO::UDS_ACCESS, info.st_mode);
    addAtom(entry, KIO::UDS_MODIFICATION_TIME, info.st_mtime);
    addAtom(entry, KIO::UDS_ACCESS_TIME, info.st_atime);
    addAtom(entry, KIO::UDS_CREATION_TIME, info.st_ctime);

    if (mime_type != QString::null && !mime_type.isEmpty()) {
        addAtom(entry, KIO::UDS_MIME_TYPE, mime_type);
    }

    if (url.isEmpty()) {
        // List an existing file.
        addAtom(entry, KIO::UDS_URL, "file:" + path);

        mode_t type = info.st_mode;
        if (S_ISLNK(type)) {
            QString slink = "";
            char buff[1000];
            int n = readlink(path, buff, 1000);
            if (n != -1) {
                buff[n] = 0;
                slink = buff;
            }
            addAtom(entry, KIO::UDS_LINK_DEST, slink);
        } else {
            type &= S_IFMT;
        }
        addAtom(entry, KIO::UDS_FILE_TYPE, type);
    } else {
        // List a locate link.
        addAtom(entry, KIO::UDS_URL, url);
        addAtom(entry, KIO::UDS_FILE_TYPE, S_IFDIR);
    }

    return entry;
}

kio_beagkProtocol::kio_beagkProtocol(const QCString &pool_socket, const QCString &app_socket)
    : SlaveBase("kio_beagk", pool_socket, app_socket)
{
    kdDebug() << "kio_beagkProtocol::kio_beagkProtocol()" << endl;
    
    connect(& beagle_search, SIGNAL(found(BeagleResultList&)),
            this, SLOT(searchHasOutput(BeagleResultList&)));
    connect(& beagle_search, SIGNAL(finished()),
            this, SLOT(searchFinished()));
    connect(& beagle_search, SIGNAL(oops_error(const QString&)),
            this, SLOT(searchError(const QString&)));
}


kio_beagkProtocol::~kio_beagkProtocol()
{
    kdDebug() << "kio_beagkProtocol::~kio_beagkProtocol()" << endl;
}


void kio_beagkProtocol::get(const KURL& url )
{
    kdDebug() << "kio_beagk::get(const KURL& url)" << endl ;
    error(KIO::ERR_IS_DIRECTORY, QString::null);
}

// from kio_locate
void kio_beagkProtocol::stat(const KURL & url)
{
    UDSEntry entry;
    /// \todo Is UDS_NAME used for anything? If so we should
    /// at least strip of the protocol part.
    addAtom(entry, KIO::UDS_NAME, url.decode_string(url.url()));
    addAtom(entry, KIO::UDS_URL, url.url());
    addAtom(entry, KIO::UDS_FILE_TYPE, S_IFDIR);
    statEntry(entry);
    finished();
}

void kio_beagkProtocol::mimetype(const KURL & /*url*/)
{
    mimeType("inode/directory");
    finished();
}

void kio_beagkProtocol::listDir(const KURL& url)
{
	kdDebug() << "kio_beagkProtocol::listDir()" << url << endl;
	QString str = URL_ToSearchString(url);
	if (str == QString::null || str.isEmpty())
		finished();
	else {
		infoMessage(i18n("Searching via libbeagle for %1 ...").arg(str));
		beagle_search.search(str);
		//emit searchError("Failed!");
		//finished();
	}
}

extern "C"
{
    int kdemain(int argc, char **argv)
    {
        KInstance instance( "kio_beagk" );
        
        kdDebug(7101) << "*** Starting kio_beagk " << endl;
        
        if (argc != 4) {
            kdDebug(7101) << "Usage: kio_beagk  protocol domain-socket1 domain-socket2" << endl;
            exit(-1);
        }
        
        kio_beagkProtocol slave(argv[2], argv[3]);
        slave.dispatchLoop();
        
        kdDebug(7101) << "*** kio_beagk Done" << endl;
        return 0;
    }
} 


/*!
    \fn kio_beagkProtocol::URL_ToSearchString(const KURL &url)
 */
QString kio_beagkProtocol::URL_ToSearchString(const KURL &url)
{
    QString pattern = url.decode_string(url.url());
    if (pattern.startsWith("beagle:", false))
        pattern = pattern.mid(7);
    
	// When konqueror tries to do autocompletion, the URL ends in "/".
    // Likewise if an URL is entered by the user any trailing slashes are
    // removed by KDE. So we can detect autocompletion.
    int n = pattern.length();
	// no sense of autocompletion ... google style autocompletion
	// can be tried later
    if ((n > 0) && (pattern[n-1] == '/'))
		return QString::null;
		
    // If the pattern starts with "/ " this is (likely) because it was
    // automatically added.
    if (pattern.startsWith("/")) {
        pattern = pattern.mid(1);
    }
	// send the rest of the string to beagle-query
	// the query interface will figure out how to parse it
	return pattern;
}

void kio_beagkProtocol::searchHasOutput(BeagleResultList &items)
{
    UDSEntryList entries;
    for (BeagleResultList::ConstIterator it = items.begin(); it != items.end(); ++it) {
        beagle_result_struct *result = *it;
		kdDebug() << "hit:(" << result->type << ")" 
			  << *(result->uri) << " title=" << *(result->title)
			  << " mime-type=" << *(result->mime_type)
			  << " last index time=" << result->last_index_time << endl;
		KURL url(*(result->uri));
		// FIXME: what about URL link , mail-messages ?
		QString path =  url.path();
		// also send path ... since we anyway have it
		if (result->type == File && QFile::exists(path))
			entries += resultToUDSEntry (path, result);
		else if (result->type == Link)
			entries += resultToUDSEntry (*(result->title), result);
	}
	
	infoMessage(i18n("beagle found %1 search results").arg( items.count() ) );
	listEntries(entries);
}

const UDSEntry resultToUDSEntry(QString & path, beagle_result_struct *result)
{
	UDSEntry entry;
	
	// add name - use the path - no smartness please
	addAtom (entry, KIO::UDS_NAME, path);
	
	// add date
	// FIXME: semantically we should access date as the date from index
	//addAtom (entry, KIO::UDS_CREATION_TIME, result->last_index_time);
	addAtom(entry, KIO::UDS_ACCESS_TIME, result->last_index_time);
	
	// add mime type
	addAtom (entry, KIO::UDS_MIME_TYPE, *(result->mime_type));
	
	// add file or directory information
	// also add URL information
	if (result->type == Link) {
		addAtom (entry, KIO::UDS_FILE_TYPE, S_IFLNK);
	} else {
		if (*(result->mime_type) == "inode/directory")
			addAtom (entry, KIO::UDS_FILE_TYPE, S_IFDIR);
		else
			addAtom (entry, KIO::UDS_FILE_TYPE, S_IFREG);
	}
	addAtom(entry, KIO::UDS_URL, *(result->uri));
		
	return entry;
}

void kio_beagkProtocol::searchFinished()
{
	finished();
}

void kio_beagkProtocol::searchError(const QString& error)
{
	messageBox(SlaveBase::Information, error);
	finished();
}

#include "beagk.moc"
