/**
    Kaya run-time system
    Copyright (C) 2004, 2005 Edwin Brady

    This file is distributed under the terms of the GNU Lesser General
    Public Licence. See COPYING for licence.
*/

#include "VMState.h"
#include "VM.h"
#include "Heap.h"
#include "ValueFuns.h"
#include "stdfuns.h"
#include "Closure.h"

/// Functions on int

kint inttable_eq(Value* x, Value* y, map<Value*, Value*>& done)
{
    return (x->getInt()==y->getInt());
}

kint inttable_fasteq(Value* x, Value* y)
{
    return (x->getInt()==y->getInt());
}

kint inttable_cmp(Value* x, Value* y)
{
    int xi = x->getInt();
    int yi = y->getInt();
    if (xi<yi) return -1;
    else if (xi==yi) return 0;
    else return 1;
}

Value* inttable_copy(Value* x, map<Value*,Value*>& done)
{
    return MKVAL((void*)(x->getInt()),KVT_INT);
}
Value* inttable_fastcopy(Value* x)
{
    return MKVAL((void*)(x->getInt()),KVT_INT);
}

void inttable_marshal(VMState* vm,String* mdata,vector<Value*>& done,Value* x, kint id)
{
    wchar_t buf[50];
    SWPRINTF(buf,50,INTMARSHAL,x->getInt());
    mdata->append(buf);
}

Value* inttable_reflect(VMState* vm,map<KayaValue, KayaValue>& done,Value* x)
{
    Value* v = KayaUnion(0,1);
    Value* i = KayaInt(x->getInt());
    KayaUnionSetArg(v,0,i);
    return v;
}

int inttable_memusage(Value* x)
{
    return sizeof(x);
}

/// Functions on String

kint stringtable_eq(Value* x, Value* y, map<Value*, Value*>& done)
{
    return (x->getString()->eq(y->getString()));
}

kint stringtable_fasteq(Value* x, Value* y)
{
    return (x->getString()->eq(y->getString()));
}

kint stringtable_cmp(Value* x, Value* y)
{
    int ret = x->getString()->cmp(y->getString());
    if (ret<0)
	return -1;
    else if (ret>0)
	return 1;
    else return 0;
}

Value* stringtable_copy(Value* x, map<Value*,Value*>& done)
{
    return MKSTR(x->getString()->getVal());
}
Value* stringtable_fastcopy(Value* x)
{
    return MKSTR(x->getString()->getVal());
}

void stringtable_marshal(VMState* vm,String* mdata,vector<Value*>& done,Value* x, kint id)
{
    String* s = x->getString();
    int len = s->length();
    int sz = len+50;
    wchar_t buf[sz];
    //    cout <<  << endl;
    //#ifdef WIN32 // now handled by STRMARSHAL macro
    SWPRINTF(buf,sz,STRMARSHAL,len,s->space(),s->getVal());
    //#else
    //    SWPRINTF(buf,sz,L"S[[%d][%d]%S]",len,s->space(),s->getVal());
    //#endif
    mdata->append(buf);
}

Value* stringtable_reflect(VMState* vm,map<KayaValue, KayaValue>& done,Value* x)
{
    Value* v = KayaUnion(2,1);
    Value* s = KayaString(x->getString()->getVal());
    KayaUnionSetArg(v,0,s);
    return v;
}

int stringtable_memusage(Value* x)
{
    return sizeof(String)+(x->getString()->space()*sizeof(wchar_t));
}

/// Functions on closures

kint fntable_eq(Value* x, Value* y, map<Value*, Value*>& done)
{
    if (done.find(x)!=done.end()) {
      return (done[x]==y);
    }
    done[x]=y; //safe since we short-circuit if anything's unequal
    kint eq = (x->getFunc()->eq(y->getFunc(),done));
    return eq;
}

/// This is probably quite meaningless
kint fntable_cmp(Value* x, Value* y)
{
    return (kint)(x->getRaw())-(kint)(y->getRaw());
}

Value* fntable_copy(Value* x, map<Value*,Value*>& done)
{
    return MKVAL((void*)(x->getFunc()->copy(done)),KVT_FUNC);
}

void fntable_marshal(VMState* vm,String* mdata,vector<Value*>& done,Value* x, kint id)
{
    // Check for circles.
    int idx;
    vector<Value*>::iterator it = done.begin();
    for(idx=0;it!=done.end();++it,++idx) {
	if ((*it)->getRaw()==x->getRaw()) {
	    // Already done it, just spit out a reference to it.
	    wchar_t buf[25];
	    SWPRINTF(buf,25,CIRCMARSHAL,idx);
	    mdata->append(buf);
	    return;
	}
    }
    idx = done.size();
    done.push_back(x);

    Closure *c = x->getFunc();
    wchar_t buf[26];
    SWPRINTF(buf,26,FUNMARSHAL,idx);
    mdata->append(buf);
    wchar_t cfnid[50];
    SWPRINTF(cfnid,50,MARSHAL2FORMAT,c->getFnID(),c->getNumArgs());
    mdata->append(cfnid);
    Value** cargs = c->getArgs();

    for(int i=0;i<c->getNumArgs();++i) {
	funtable_marshal_aux(vm,mdata,done,cargs[i],id);
    }
    mdata->append(L"]");
}

KayaValue findIdentical(KayaValue v,map<KayaValue, KayaValue> done) {
    map<KayaValue, KayaValue>::iterator it = done.begin();
    for(;it!=done.end();++it) {
	if ((*it).first->getRaw() == v->getRaw()) {
	    return (*it).second;
	}
    }
    return NULL;
}

Value* fntable_reflect(VMState* vm,map<KayaValue, KayaValue>& done,Value* x) {
    // Check for circles.
    KayaValue c = findIdentical(x,done);
    if (c!=NULL) return c;

    Closure* inc = x->getFunc();

    Value* v = KayaUnion(5,2);
    done[x] = v;

    KArray args = newKayaArray(inc->getNumArgs());

    Value** cargs = inc->getArgs();
    for(int i=0;i<inc->getNumArgs();++i) {
	Value* el = funtable_reflect_aux(vm,done,cargs[i]);
	KayaArrayPush(args,el);
    }

    Value* uarr = KayaArrayVal(args);
    KayaUnionSetArg(v,0,KayaInt(inc->getFnID()));
    KayaUnionSetArg(v,1,uarr);

    return v;
}

int fntable_memusage(Value* x)
{
    int size = sizeof(Value)+sizeof(Closure);
    Value** args = x->getFunc()->getArgs();
    for(int i=0;i<x->getFunc()->getNumArgs();++i) {
	size+=4;
	size+=funtable_memusage(args[i]);
    }
    return size;
}

/// Functions on Reals

kint realtable_eq(Value* x, Value* y, map<Value*, Value*>& done)
{
    return (x->getReal()==y->getReal());
}

kint realtable_fasteq(Value* x, Value* y)
{
    return (x->getReal()==y->getReal());
}

kint realtable_cmp(Value* x, Value* y)
{
    double xv = x->getReal();
    double yv = y->getReal();
    if (yv>xv) return -1;
    else if (yv==xv) return 0;
    else return 1;
}

Value* realtable_copy(Value* x, map<Value*,Value*>& done)
{
    double xv = x->getReal();
    Value* v=MKVAL(NULL,KVT_REAL);
    v->setReal(xv);
    return v;
}
Value* realtable_fastcopy(Value* x)
{
    double xv = x->getReal();
    Value* v=MKVAL(NULL,KVT_REAL);
    v->setReal(xv);
    return v;
}

void realtable_marshal(VMState* vm,String* mdata,vector<Value*>& done,Value* x, kint id)
{
    wchar_t buf[255];
    SWPRINTF(buf,255,L"R[%e]",x->getReal());
    mdata->append(buf);
}

Value* realtable_reflect(VMState* vm,map<KayaValue, KayaValue>& done,Value* x)
{
    Value* v = KayaUnion(1,1);
    Value* f = KayaFloat(KFLOAT(x->getReal()));
    KayaUnionSetArg(v,0,f);
    return v;
}

int realtable_memusage(Value* x)
{
    return sizeof(Value)+sizeof(Real);
}

/// Functions on Arrays

kint arraytable_eq(Value* x, Value* y, map<Value*, Value*>& done)
{
    if (done.find(x)!=done.end()) {
	return (done[x]==y);
    }
    done[x]=y;
    kint eq = (x->getArray()->eq(y->getArray(),done));
    return eq;
}

kint arraytable_cmp(Value* x, Value* y)
{
    int ret = x->getArray()->cmp(y->getArray());
    if (ret<0)
	return -1;
    else if (ret>0)
	return 1;
    else return 0;
}

Value* arraytable_copy(Value* x, map<Value*,Value*>& done)
{
    // if x is already copied, return the copied value
    if (done.find(x)!=done.end()) {
	return done[x];
    }
    Value* copied = MKVAL(x->getArray()->copy(done),KVT_ARRAY);
    done[x]=copied;
    return copied;
}

void arraytable_marshal(VMState* vm,String* mdata,vector<KayaValue>& done,Value* x, kint id)
{
    // Check for circles.
    int idx;
    vector<Value*>::iterator it = done.begin();
    for(idx=0;it!=done.end();++it,++idx) {
	if ((*it)->getRaw()==x->getRaw()) {
	    // Already done it, just spit out a reference to it.
	    wchar_t buf[26];
	    SWPRINTF(buf,26,CIRCMARSHAL,idx);
	    mdata->append(buf);
	    return;
	}
    }
    idx = done.size();
    done.push_back(x);

    Array* v = x->getArray();
    wchar_t buf[34];
    SWPRINTF(buf,34,ARRMARSHAL,idx,v->reserved_size());
    mdata->append(buf);
    for(unsigned i=0;i<v->size();++i) {
	funtable_marshal_aux(vm,mdata,done,v->lookup(i),id);
    }
    mdata->append(L"]");
}

Value* arraytable_reflect(VMState* vm,map<KayaValue, KayaValue>& done,Value* x)
{
    // Check for circles.
    KayaValue c = findIdentical(x,done);
    if (c!=NULL) return c;

    Array* inarr = x->getArray();

    Value* v = KayaUnion(3,1);
    done[x] = v;

    KArray ar = newKayaArray(inarr->size());

    for(unsigned i=0;i<inarr->size();++i) {
	Value* el = funtable_reflect_aux(vm,done,inarr->lookup(i));
	KayaArrayPush(ar,el);
    }

    Value* varr = KayaArrayVal(ar);
    KayaUnionSetArg(v,0,varr);

    return v;
}

int arraytable_memusage(Value* x)
{
    int size = sizeof(Value);
    size+=x->getArray()->memusage();
    return size;
}

/// Functions on Unions

kint uniontable_eq(Value* x, Value* y, map<Value*, Value*>& done)
{
    if (done.find(x)!=done.end()) {
	return (done[x]==y);
    }
    done[x]=y;
    if (x->getType() != y->getType()) {
      return false;
      // would be even better with module-unique tags, but generally
      // compiler type-safety should sort that out
    }
    kint eq = (x->getUnion()->eq(y->getUnion(),done));
    return eq;
}

kint uniontable_cmp(Value* x, Value* y)
{
  kint xt = x->getType();
  kint yt = y->getType();
  if (xt == yt) {
    int ret = x->getUnion()->cmp(y->getUnion());
    if (ret<0)
	return -1;
    else if (ret>0)
	return 1;
    else return 0;
  } else if (xt < yt) {
    return -1;
  } else {
    return 1;
  }
}

Value* uniontable_copy(Value* x, map<Value*,Value*>& done)
{
    Value* thisval = MKVAL(NULL,KVT_NULL);
    done[x] = thisval;
    //    Value* copied = MKVAL(,x->getType());
    thisval->setPtr(x->getUnion()->copy(done),x->getType()); 
    return thisval;
}

void uniontable_marshal(VMState* vm,String* mdata,vector<KayaValue>& done,Value* x, kint id)
{
    // Check for circles.
    int idx;
    vector<Value*>::iterator it = done.begin();
    for(idx=0;it!=done.end();++it,++idx) {
	if ((*it)->getRaw()==x->getRaw()) {
	    // Already done it, just spit out a reference to it.
	    wchar_t buf[26];
	    SWPRINTF(buf,26,CIRCMARSHAL,idx);
	    mdata->append(buf);
	    return;
	}
    }
    idx = done.size();
    done.push_back(x);

    Union *u = x->getUnion();
    wchar_t buf[30];
    SWPRINTF(buf,30,UNIMARSHAL,idx);
    mdata->append(buf);
    wchar_t utag[50];
    SWPRINTF(utag,50,MARSHAL2FORMAT,x->getType()-KVT_UNION,U_ARITY(u));
    mdata->append(utag);

    for(unsigned i=0;i<U_ARITY(u);++i) {
	funtable_marshal_aux(vm,mdata,done,u->args[i],id);
    }
    mdata->append(L"]");
}



Value* uniontable_reflect(VMState* vm,map<KayaValue, KayaValue>& done,Value* x)
{
    // Check for circles.
    KayaValue c = findIdentical(x,done);
    if (c!=NULL) return c;

    Union* inu = x->getUnion();

    Value* v = KayaUnion(4,2);
    done[x] = v;

    KArray flds = newKayaArray(U_ARITY(inu));

    for(unsigned i=0;i<U_ARITY(inu);++i) {
	Value* el = funtable_reflect_aux(vm,done,inu->args[i]);
	KayaArrayPush(flds,el);
    }

    Value* uarr = KayaArrayVal(flds);
    KayaUnionSetArg(v,0,KayaInt(x->getType()-KVT_UNION));
    KayaUnionSetArg(v,1,uarr);

    return v;
}

int uniontable_memusage(Value* x)
{
    int size = sizeof(Value);
    size+=x->getUnion()->memusage();
    return size;
}

/// Functions on Unions with no parameters

kint cuniontable_eq(Value* x, Value* y, map<Value*, Value*>& done)
{
  return (x->getInt()==y->getInt());
}

kint cuniontable_cmp(Value* x, Value* y)
{
    int xi = x->getInt();
    int yi = y->getInt();
    if (xi<yi) return -1;
    else if (xi==yi) return 0;
    else return 1;
}

Value* cuniontable_copy(Value* x, map<Value*,Value*>& done)
{
    Value* thisval = MKVAL((void*)x->getInt(),KVT_CUNION);
    return thisval;
}

void cuniontable_marshal(VMState* vm,String* mdata,vector<KayaValue>& done,Value* x, kint id)
{
    // Check for circles.
    int idx;
    vector<Value*>::iterator it = done.begin();
    for(idx=0;it!=done.end();++it,++idx) {
	if ((*it)->getRaw()==x->getRaw()) {
	    // Already done it, just spit out a reference to it.
	    wchar_t buf[30];
	    SWPRINTF(buf,30,CIRCMARSHAL,idx);
	    mdata->append(buf);
	    return;
	}
    }
    idx = done.size();
    done.push_back(x);

    wchar_t buf[30];
    SWPRINTF(buf,30,UNIMARSHAL,idx);
    mdata->append(buf);
    wchar_t utag[50];
    SWPRINTF(utag,50,MARSHAL2FORMAT,x->getInt(),0);
    mdata->append(utag);
    mdata->append(L"]");
}



Value* cuniontable_reflect(VMState* vm,map<KayaValue, KayaValue>& done,Value* x)
{
    // Check for circles.
    KayaValue c = findIdentical(x,done);
    if (c!=NULL) return c;

    Value* v = KayaUnion(4,2);
    done[x] = v;

    KArray flds = newKayaArray(0);
    Value* uarr = KayaArrayVal(flds);
    KayaUnionSetArg(v,0,KayaInt(x->getInt()));
    KayaUnionSetArg(v,1,uarr);

    return v;
}

int cuniontable_memusage(Value* x)
{
    int size = sizeof(Value);
    return size;
}


/// Functions on Exceptions

kint exceptiontable_eq(Value* x, Value* y, map<Value*, Value*>& done)
{
    return (x->getExcept()->eq(y->getExcept()));
}

kint exceptiontable_fasteq(Value* x, Value* y)
{
    return (x->getExcept()->eq(y->getExcept()));
}

kint exceptiontable_cmp(Value* x, Value* y)
{
    kint ret = x->getExcept()->cmp(y->getExcept());
    if (ret<0)
	return -1;
    else if (ret>0)
	return 1;
    else return 0;
}

Value* exceptiontable_copy(Value* x, map<Value*,Value*>& done)
{
    // No point since they're immutable
    return x;
}
Value* exceptiontable_fastcopy(Value* x)
{
    // No point since they're immutable
    return x;
}


void exceptiontable_marshal(VMState* vm,String* mdata,vector<Value*>& done,Value* x, kint id)
{
    vm->kaya_rtsError(CANT_MARSHAL_EXCEPTIONS);
}

Value* exceptiontable_reflect(VMState* vm,map<KayaValue, KayaValue>& done,Value* x)
{
    vm->kaya_rtsError(CANT_REFLECT_EXCEPTIONS);
    return NULL;
}

int exceptiontable_memusage(Value* x)
{
    int size = sizeof(Value)+sizeof(Exception);
    return size; // FIXME: This is wrong, but not important.
}

