/*
 *   Jackbeat - JACK sequencer
 *    
 *   Copyright (c) 2004-2008 Olivier Guilyardi <olivier {at} samalyse {dot} com>
 *    
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *   SVN:$Id: song.c 121 2008-01-10 00:17:40Z olivier $
 */

#include <jack/jack.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include <config.h>

#include "song.h"
#include "util.h"

#ifdef MEMDEBUG
#include "memdebug.h"
#endif

#ifdef DMALLOC
#include "dmalloc.h"
#endif

#define DEBUG(M, ...) { printf("SNG  %s(): ", __func__); printf(M, ## __VA_ARGS__); printf("\n"); }
#define FREE_AND_RETURN(P,V) { free(P); return V; }
#define JACK_CLIENT_NAME "jackbeat"

struct song_t
{
  sequence_t **sequences;
  int sequences_num;
  sample_t **samples;
  int samples_num;
};

song_t *
song_new ()
{
  song_t *song = calloc (1, sizeof (song_t));

  song->sequences = NULL;
  song->sequences_num = 0;
  song->samples = NULL;
  song->samples_num = 0;
  return song;

}

void
song_destroy (song_t * song)
{
  if (song->sequences) free (song->sequences);
  if (song->samples) free (song->samples);
  free (song);
}



void
song_register_sequence (song_t *song, sequence_t *sequence)
{
  UTIL_ARRAY_ADD (sequence_t, song->sequences, song->sequences_num, sequence)
}

void
song_register_sequence_samples (song_t *song, sequence_t *sequence)
{
  int tn = sequence_get_tracks_num (sequence);
  int i;
  sample_t *sample;
  for (i=0; i < tn; i++) 
    if ((sample = sequence_get_sample (sequence, i)))
      song_register_sample (song, sample);
}

void
song_unregister_sequence (song_t *song, sequence_t *sequence)
{
  assert (song->sequences);
  UTIL_ARRAY_REMOVE (sequence_t, song->sequences, song->sequences_num, sequence)
}


int
song_count_sequences (song_t * song)
{
  return song->sequences_num;
}

sequence_t **
song_list_sequences (song_t * song)
{
  return song->sequences;
}


void
song_register_sample (song_t *song, sample_t *sample)
{
  UTIL_ARRAY_ADD (sample_t, song->samples, song->samples_num, sample)
}

void
song_unregister_sample (song_t *song, sample_t *sample)
{
  assert (song->samples);
  UTIL_ARRAY_REMOVE (sample_t, song->samples, song->samples_num, sample)
}


int
song_count_samples (song_t * song)
{
  return song->samples_num;
}

sample_t **
song_list_samples (song_t * song)
{
  return song->samples;
}

void
song_cleanup_unused_samples (song_t *song)
{ 
  int i,j,k;
  DEBUG ("Operating on %d sample(s) within %d sequence(s)", song->samples_num,
         song->sequences_num);
  for (i=0; i < song->samples_num; i++)
   {
    k = 0;
    for (j=0; j < song->sequences_num; j++)
     k += sequence_get_sample_usage (song->sequences[j], song->samples[i]);
    if (!k) { 
      DEBUG ("Destroying sample : %s", song->samples[i]->name);
      sample_destroy (song->samples[i]);
      song_unregister_sample (song, song->samples[i]);
    }
   }
}

int 
song_get_sequence_index (song_t *song)
{
  return song->sequences_num;
}

sample_t *
song_try_reuse_sample (song_t *song, char *filename)
{
  int i;
  for (i=0; i < song->samples_num; i++)
    if (sample_compare (song->samples[i], filename) == 0) 
      return song->samples[i];

  return NULL;
}

