
/****************************************************************************
 *
 * All portions copyright their respective authors.  All rights reserved.
 *
 * This file is part of IVMan (ivm).
 *
 * This file may be distributed under the terms of the Q Public License
 * as defined by Troll Tech AS of Norway and appearing in the file
 * LICENSE.QPL included in the packaging of this file.
 * 
 * See http://www.troll.no/qpl for QPL licensing information.
 *
 * $Id: IvmConfigConditions.c,v 1.5 2005/11/21 15:27:17 ro_han Exp $
 *****************************************************************************/

#include "common.h"
#include "IvmConfigConditions.h"

#include <string.h>
#include <glib.h>
#include <libxml/tree.h>
#include <libxml/parser.h>
#include <libxml/xpath.h>
#include <libxml/xpathInternals.h>
#include <dbus/dbus.h>
#include <libhal.h>

IvmConfigConditions *parseIvmConfigConditions(char const *const path,
                                              char const *udi,
                                              char const *const
                                              conditionName)
{
    gboolean mountable = ivm_device_is_mountable(udi);


    LIBXML_TEST_VERSION IvmConfigConditions *ret = NULL;

    // DEBUG("parseIvmConfigActions %s", path);
    ret = (IvmConfigConditions *) malloc(sizeof(IvmConfigConditions));

    if (ret == NULL)
    {
        DEBUG(_("Out of memory!\n"));
        return NULL;
    }

    memset(ret, 0, sizeof(IvmConfigConditions));

    ret->exec = NULL;
    int num_exec = 0;

    xmlDocPtr doc;
    xmlNodePtr cur;

    doc = xmlParseFile(path);
    if (doc == NULL)
    {
        DEBUG(_("Document not parsed successfully."));
        return NULL;
    }

    cur = xmlDocGetRootElement(doc);
    if (cur == NULL)
    {
        DEBUG(_("Document is empty!"));
        xmlFreeDoc(doc);
        return NULL;
    }
    if (xmlStrcmp(cur->name, (const xmlChar *) "ConditionsConfig"))
    {
        DEBUG(_
              ("Incorrect document type, root node should have been %s"),
              "ConditionsConfig");
        xmlFreeDoc(doc);
        return NULL;
    }
    cur = cur->children;
    while (cur != NULL)
    {
        // DEBUG("At XML node %s",cur->name);
        if (!xmlStrcmp(cur->name, (const xmlChar *) "Match"))
        {
            /* Check if our device matches - if so, continue iterating
               through siblings, otherwise jump back up to parent */

            gboolean matches = FALSE;


            xmlChar *name;
            name = xmlGetProp(cur, "name");

            xmlChar *value;
            value = xmlGetProp(cur, "value");

            if (name == NULL)
            {
                DEBUG(_("Warning: %s XML tag encountered with missing or bad attributes, ignored\n"), cur->name);
                goto nextMatch;
            }

            if (!xmlStrcmp(name, (const xmlChar *) "*"))
            {
                matches = TRUE;
                goto nextMatch;
            }
            
            if (value == NULL)
            {
                DEBUG(_("Warning: %s XML tag encountered with missing or bad attributes, ignored\n"), cur->name);
                goto nextMatch;
            }

            else if (!xmlStrcmp(name, (const xmlChar *) "ivm.mountable"))
            {
                if (((!xmlStrcmp
                      (value, (const xmlChar *) "true"))
                     && mountable)
                    ||
                    ((!xmlStrcmp
                      (value, (const xmlChar *) "false")) && !mountable))
                    matches = TRUE;
            }

            else if (!xmlStrncmp(name, (const xmlChar *) "hal.", 4))
                matches = hal_xml_property_matches(name, value, udi);
            else
                DEBUG(_("Invalid match name: %s"), (char *) name);

nextMatch:
            if (matches == TRUE && cur->children != NULL)
                cur = cur->children;
            else if (cur->next != NULL)
                cur = cur->next;
            else
                cur = cur->parent->next;

            if (value) xmlFree(value);
            if (name) xmlFree(name);
        }

        else if (!xmlStrcmp(cur->name, (const xmlChar *) "Condition"))
        {
            xmlChar *name = xmlGetProp(cur, "name");

            xmlChar *exec = xmlGetProp(cur, "exec");

            if ((name == NULL) || (exec == NULL))
            {
                DEBUG(_("Warning: %s XML tag encountered with missing or bad attributes, ignored\n"), cur->name);
                goto nextCondition;
            }
            
            if (!xmlStrcmp(name, (xmlChar *) conditionName))
            {
                num_exec++;
                ret->exec = realloc(ret->exec,
                                    (num_exec + 1) * sizeof(char *));
                ret->exec[num_exec - 1] = (char *) xmlStrdup(exec);
                ret->exec[num_exec] = NULL;
            }

nextCondition:
            if (name) xmlFree(name);
            if (exec) xmlFree(exec);

            if (cur->next != NULL)
                cur = cur->next;
            else
                cur = cur->parent->next;
        }
        else if (cur->next != NULL)
            cur = cur->next;
        else
            cur = cur->parent->next;

    }
    xmlFreeDoc(doc);

    xmlCleanupParser();

    return ret;
}
