/*
 * 
 *                This source code is part of
 * 
 *                 G   R   O   M   A   C   S
 * 
 *          GROningen MAchine for Chemical Simulations
 * 
 *                        VERSION 3.2.0
 * Written by David van der Spoel, Erik Lindahl, Berk Hess, and others.
 * Copyright (c) 1991-2000, University of Groningen, The Netherlands.
 * Copyright (c) 2001-2004, The GROMACS development team,
 * check out http://www.gromacs.org for more information.

 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * If you want to redistribute modifications, please consider that
 * scientific software is very special. Version control is crucial -
 * bugs must be traceable. We will be happy to consider code for
 * inclusion in the official distribution, but derived work must not
 * be called official GROMACS. Details are found in the README & COPYING
 * files - if they are missing, get the official version at www.gromacs.org.
 * 
 * To help us fund GROMACS development, we humbly ask that you cite
 * the papers on the package - you can find them in the top README file.
 * 
 * For more info, check our website at http://www.gromacs.org
 * 
 * And Hey:
 * Gromacs Runs On Most of All Computer Systems
 */

#ifndef _pppm_h
#define _pppm_h

#include <stdio.h>
#include "typedefs.h"
#include "gmxcomplex.h"

#ifdef __cplusplus
extern "C" {
#endif

int gmx_pppm_init(FILE *log,      t_commrec *cr,
			 const output_env_t oenv, gmx_bool bVerbose,
			 gmx_bool bOld,      matrix box,
			 char *ghatfn,   t_inputrec *ir,
                         gmx_bool bReproducible);
/* Setup stuff for PPPM. 
 * Either reads a ghat function from file (when the file exists)
 * or generate a ghat function from scratch.
 */

int gmx_pppm_do(FILE *log,       gmx_pme_t pme,
		       gmx_bool bVerbose,
		       rvec x[],        rvec f[],
		       real charge[],   rvec box,
		       real phi[],      t_commrec *cr,
		       int start,       int nt,
		       t_nrnb *nrnb,
		       int pme_order,   real *energy);
/* Do a PPPM calculation for the long range electrostatics. */

/******************************************************************
 *
 *   ROUTINES FOR GHAT MANIPULATION
 *
 ******************************************************************/
 
real gk(real k,real rc,real r1);
/* Compute the Ghat function for a single k-value */

real gknew(real k,real rc,real r1);
/* Compute the (new!) Ghat function for a single k-value */

void pr_scalar_gk(const char *fn,const output_env_t oenv,
                         int nx,int ny,int nz, rvec box,real ***ghat);

void mk_ghat(FILE *fp,int nx,int ny,int nz,
                    real ***ghat, rvec box,real r1,real rc,gmx_bool bSym,gmx_bool bOld);
/* Generate a Ghat function from scratch. The ghat grid should
 * be allocated using the mk_rgrid function. When bSym, only
 * the first octant of the function is generated by direct calculation
 * and the above mentioned function is called for computing the rest.
 * When !bOld a new experimental function form will be used.
 */

real ***rd_ghat(FILE *log,const output_env_t oenv,char *fn,ivec igrid,
                       rvec gridspacing, rvec beta,int *porder,
                       real *rshort,real *rlong);
/* Read a Ghat function from a file as generated by the program
 * mk_ghat. The grid size (number of grid points) is returned in
 * igrid, the space between grid points in gridspacing,
 * beta is a constant that determines the contribution of first
 * and second neighbours in the grid to the force
 * (See Luty et al. JCP 103 (1995) 3014)
 * porder determines whether 8 (when porder = 1) or 27 (when
 * porder = 2) neighbouring grid points are used for spreading
 * the charge.
 * rshort and rlong are the lengths used for generating the Ghat
 * function.
 */
		  
void wr_ghat(const char *fn,const output_env_t oenv, 
                    int n1max,int n2max, int n3max,real h1,
		    real h2,real h3,real ***ghat,int nalias,
		    int porder,int niter,gmx_bool bSym,rvec beta,
		    real r1,real rc,real pval,real zval,real eref,real qopt);
/* Write a ghat file. (see above) */

#ifdef __cplusplus
}
#endif

#endif
