// -*- c++ -*-
//------------------------------------------------------------------------------
// $Id: DeckInfo.cpp,v 1.19 2007/01/14 02:59:57 vlg Exp $
//------------------------------------------------------------------------------
//                            DeckInfo.cpp
//------------------------------------------------------------------------------
//  Copyright (c) 2004 by Vladislav Grinchenko 
//
//  This program is free software; you can redistribute it and/or 
//  modify it under the terms of the GNU General Public License   
//  as published by the Free Software Foundation; either version  
//  2 of the License, or (at your option) any later version.      
//------------------------------------------------------------------------------
//
// Date   : Mon Feb 16 22:08:26 EST 2004
//
//------------------------------------------------------------------------------

#ifdef HAVE_CONFIG_H
#    include "config.h"
#endif

#include <gtkmm/button.h>
#include <gtkmm/buttonbox.h>
#include <gtkmm/label.h>
#include <gtkmm/alignment.h>
#include <gtkmm/entry.h>
#include <gtkmm/table.h>
#include <gtkmm/box.h>
#include <gtkmm/stock.h>
#include <gtkmm/main.h>

#include "Granule-main.h"
#include "Granule.h"
#include "DeckInfo.h"
#include "Deck.h"
#include "GrappConf.h"
#include "MyFolderChooseDialog.h"
#include "TextAlignment.h"

#include "Intern.h"

//------------------------------------------------------------------------------
// Useful Macros
//------------------------------------------------------------------------------

#define DI_SET_LABEL(l) \
	l->set_alignment  (0.5,0.5); \
	l->set_padding    (0,0); \
	l->set_justify    (Gtk::JUSTIFY_LEFT); \
	l->set_line_wrap  (false); \
	l->set_use_markup (false); \
	l->set_selectable (false);

#define DI_SET_ENTRY(e, t, d)      \
	e->set_flags (Gtk::CAN_FOCUS); \
	e->set_visibility (true);      \
	e->set_editable   (d);         \
	e->set_max_length (0);         \
    e->set_text       (t);         \
	e->set_has_frame  (true);      \
	e->set_activates_default (false);

#define DI_SET_FRAME(f,w) \
	f->set_shadow_type  (Gtk::SHADOW_ETCHED_IN); \
	f->set_border_width (4); \
	f->set_label_align  (1, 1); \
	f->add (*w);


#define DI_SET_CHECKBUTTON(b,a)		\
	b->set_flags  (Gtk::CAN_FOCUS); \
	b->set_relief (Gtk::RELIEF_NORMAL); \
	b->set_mode   (true); \
    b->set_active (a); \
	b->set_border_width (4); \
	b->set_alignment (0.1, 0);

//------------------------------------------------------------------------------
// Class DIFontEntry
//------------------------------------------------------------------------------
DIFontEntry::
DIFontEntry ()
{
	Gtk::Frame* fonts_frame = this;

	m_font_table = Gtk::manage (new Gtk::Table (1, 3, false));

	m_label  = Gtk::manage(new Gtk::Label ("Question font:"));
	m_entry  = Gtk::manage(new Gtk::Entry);
	m_button = Gtk::manage(new Gtk::Button ("Select"));

	m_label->set_alignment(Gtk::ALIGN_RIGHT,Gtk::ALIGN_CENTER);
	m_label->set_padding(0,0);
	m_label->set_justify(Gtk::JUSTIFY_LEFT);
	m_label->set_line_wrap(false);
	m_label->set_use_markup(false);
	m_label->set_selectable(false);

	m_entry->set_flags(Gtk::CAN_FOCUS);
	m_entry->set_visibility(true);
	m_entry->set_editable(true);
	m_entry->set_max_length(0);
	m_entry->set_text("");
	m_entry->set_has_frame(true);
	m_entry->set_activates_default(false);
#ifdef IS_HILDON
	m_entry->set_width_chars (26);
#endif

	m_button->set_flags(Gtk::CAN_FOCUS);
	int i = 0;
						   
	m_font_table->attach (*m_label, 
						  0, 1, i, i+1,
						  Gtk::FILL, 
						  Gtk::AttachOptions (), 2, 2);

	m_font_table->attach (*m_entry, 
						  1, 2, i, i+1,
#ifdef IS_HILDON
						  Gtk::FILL,
#else
						  Gtk::EXPAND|Gtk::FILL,
#endif
						  Gtk::AttachOptions(), 2, 2);

	m_font_table->attach (*m_button, 
						  2, 3, i, i+1,
						  Gtk::FILL, 
						  Gtk::AttachOptions (), 2, 2);

	Gtk::Label* fonts_label = Gtk::manage(new Gtk::Label("<b>Fonts</b>"));
	fonts_label->set_padding    (0,0);   
	fonts_label->set_line_wrap  (false); 
	fonts_label->set_use_markup (true);  
	fonts_label->set_selectable (false);

	fonts_frame->set_border_width (4);
	fonts_frame->set_shadow_type  (Gtk::SHADOW_ETCHED_IN);
	fonts_frame->set_label_align  (Gtk::ALIGN_TOP, Gtk::ALIGN_CENTER);
	fonts_frame->set_label_widget (*fonts_label);
	fonts_frame->add              (*m_font_table);

	m_button->signal_clicked().connect (
		mem_fun (*this, &DIFontEntry::on_select_clicked));

	show_all_children ();
}

void
DIFontEntry::
on_select_clicked ()
{
	Gtk::FontSelectionDialog sfd ("Select Text Font");
	sfd.set_font_name (m_entry->get_text ());

	if (sfd.run () != Gtk::RESPONSE_OK) {
		return;
	}
	m_entry->set_text (sfd.get_font_name ());
}

//------------------------------------------------------------------------------
// Class DeckInfo
//------------------------------------------------------------------------------

DeckInfo::
DeckInfo (Gtk::Window& parent_, Deck& deck_) 
	:
	m_deck (deck_),
	m_author_entry   (Gtk::manage (new Gtk::Entry ())),
	m_desc_entry     (Gtk::manage (new Gtk::Entry ())),
	m_snd_path_entry (Gtk::manage (new Gtk::Entry ())),
	m_appearance_changed (false)
{  
	trace_with_mask("DeckInfo::DeckInfo", GUITRACE);

	Gtk::VBox*   all_vbox;

	Gtk::Button* cancel_button;
	Gtk::Button* ok_button;

	Gtk::Table*  info_table;

	Gtk::Label*  author_label;
	Gtk::Label*  desc_label;
	Gtk::Label*  snd_label;
	
	Gtk::Alignment* author_alignment;
	Gtk::Alignment* desc_alignment;
	Gtk::Alignment* snd_label_alignment;

	Gtk::HBox*  snd_path_box;
	Gtk::VBox*  snd_vbox;
	Gtk::Frame* snd_frame;

	Gtk::Frame* app_frame;
	Gtk::VBox*  app_vbox;

	/** Set up the dialog itself
	 */
	set_title (_("Deck Info"));
	set_has_separator (true);

	set_modal (true);

#ifdef IS_PDA
	set_resizable (false);
	Gdk::Geometry dp_geometry =	{ 240, 320,	240, 320, 240, 320,	-1, -1,	-1, -1};
	set_geometry_hints (*this, dp_geometry, 
				Gdk::HINT_MIN_SIZE | Gdk::HINT_MAX_SIZE | Gdk::HINT_BASE_SIZE);
#else

#ifdef IS_HILDON
	Gdk::Geometry dp_geometry =	{ 640, 480, 640, 480, 640, 480, -1, -1, -1, -1};
	set_geometry_hints (*this, dp_geometry, 
				Gdk::HINT_MIN_SIZE | Gdk::HINT_MAX_SIZE | Gdk::HINT_BASE_SIZE);
#else
	set_size_request (400, 370);	// width; height 
#endif	// @IS_HILDON

	set_resizable (true);
	set_transient_for (parent_);

#endif	// @IS_PDA

	/** Hold everything together
	 */
	all_vbox = Gtk::manage (new Gtk::VBox (false, 0));

	/***********************
	 *  Control buttons
	 ***********************/
	cancel_button = Gtk::manage (new Gtk::Button (Gtk::StockID ("gtk-cancel")));
	ok_button     = Gtk::manage (new Gtk::Button (Gtk::StockID ("gtk-ok")));

	/***********************
	 *  Author/Description
	 ***********************/
	info_table       = Gtk::manage (new Gtk::Table (2, 2, false));

	author_label     = Gtk::manage (new Gtk::Label (_("Author: ")));
	author_alignment = Gtk::manage (new Gtk::Alignment (Gtk::ALIGN_LEFT,
														Gtk::ALIGN_CENTER,
														0.2, 0));
	desc_label       = Gtk::manage (new Gtk::Label (_("Description: ")));
	desc_alignment   = Gtk::manage (new Gtk::Alignment (Gtk::ALIGN_LEFT,
														Gtk::ALIGN_CENTER,
														0.2, 0));
	/*************************
	 * Sound Dictionary Path
	 *************************/
	snd_label           = Gtk::manage (new Gtk::Label (_("      Path: ")));
	snd_label_alignment = Gtk::manage (new Gtk::Alignment (Gtk::ALIGN_LEFT,
														   Gtk::ALIGN_CENTER,
														   0.2, 0));

	m_snd_browse_button = Gtk::manage (new Gtk::Button (_("Browse")));
	snd_path_box        = Gtk::manage (new Gtk::HBox (false, 2));
	snd_vbox            = Gtk::manage (new Gtk::VBox (false, 2));
	snd_frame           = Gtk::manage (new Gtk::Frame ());
	m_enable_alt_snd    = Gtk::manage (new Gtk::CheckButton (
									 _("Enable alternative sound dictionary")));

	/************************
	 * Custom Appearance
	 ************************/
	m_question_font = Gtk::manage (new DIFontEntry ());

	m_enable_custom_app = Gtk::manage (new Gtk::CheckButton (
										   _("Enable custom appearance")));
	m_inner_app_vbox = Gtk::manage (new Gtk::VBox (false, 2));
	app_vbox         = Gtk::manage (new Gtk::VBox (false, 2));
	app_frame        = Gtk::manage (new Gtk::Frame ());

	MyTextAlignmentsWidget::value_list_t align_x_list;
	align_x_list.push_back ("center");
	align_x_list.push_back ("left");
	align_x_list.push_back ("right");

	MyTextAlignmentsWidget::value_list_t align_y_list;
	align_y_list.push_back ("center");
	align_y_list.push_back ("top");
	align_y_list.push_back ("bottom");

	MyTextAlignmentsWidget::value_list_t paragraph_list;
	paragraph_list.push_back ("center");
	paragraph_list.push_back ("left");
	paragraph_list.push_back ("right");
	paragraph_list.push_back ("fill");

	m_front_widget=new MyTextAlignmentsWidget("<b>Front Text Alignment</b>",
											  align_x_list,
											  align_y_list,
											  paragraph_list);

	m_back_widget=new MyTextAlignmentsWidget("<b>Back Text Alignment</b>",
											 align_x_list,
											 align_y_list,
											 paragraph_list);

	m_example_widget=new MyTextAlignmentsWidget("<b>Example Text Alignment</b>",
												align_x_list,
												align_y_list,
												paragraph_list);

	load_appearance ();

	m_inner_app_vbox->pack_start (*m_question_font,
								  Gtk::PACK_SHRINK, 4);
	m_inner_app_vbox->pack_start (*(m_front_widget->frame ()), 
								  Gtk::PACK_SHRINK, 4);
	m_inner_app_vbox->pack_start (*(m_back_widget->frame ()), 
								  Gtk::PACK_SHRINK, 4);
	m_inner_app_vbox->pack_start (*(m_example_widget->frame ()), 
								  Gtk::PACK_SHRINK, 4);

	/*******************
	 * Control buttons
	 *******************/
	cancel_button->set_flags  (Gtk::CAN_FOCUS);
	cancel_button->set_relief (Gtk::RELIEF_NORMAL);

	ok_button->set_flags  (Gtk::CAN_FOCUS);
	ok_button->set_relief (Gtk::RELIEF_NORMAL);

	get_action_area ()->property_layout_style ().set_value (Gtk::BUTTONBOX_END);

	DI_SET_LABEL(author_label);
	author_alignment->add (*author_label);

	DI_SET_LABEL(desc_label);
	desc_alignment->add (*desc_label);

	DI_SET_ENTRY(m_author_entry, m_deck.get_author (), true);
	DI_SET_ENTRY(m_desc_entry,   m_deck.get_desc   (), true);

	info_table->set_row_spacings (1);
	info_table->set_col_spacings (1);

	/** Default [xy]options is Gtk::FILL|Gtk::EXPAND
	 */
	info_table->attach (*author_alignment, 0, 1, 0, 1, Gtk::FILL);
	info_table->attach (*desc_alignment,   0, 1, 1, 2, Gtk::SHRINK);
	info_table->attach (*m_author_entry,   1, 2, 0, 1);
	info_table->attach (*m_desc_entry,     1, 2, 1, 2);

	/****************
	 *  Setup sound
	 ****************/
	DI_SET_LABEL(snd_label);
	snd_label_alignment->add (*snd_label);

	std::string snd_path = m_deck.get_snd_path ();

	if (snd_path.length () == 0) {
		DI_SET_CHECKBUTTON(m_enable_alt_snd,false);
		m_snd_browse_button->set_sensitive (false);
	}
	else {
		DI_SET_CHECKBUTTON(m_enable_alt_snd,true);
		m_snd_browse_button->set_sensitive (true);
	}

	DI_SET_ENTRY(m_snd_path_entry, m_deck.get_snd_path (), false);

	m_snd_browse_button->set_flags  (Gtk::CAN_FOCUS);
	m_snd_browse_button->set_relief (Gtk::RELIEF_NORMAL);

	snd_path_box->pack_start (*snd_label_alignment, Gtk::PACK_SHRINK,        2);
	snd_path_box->pack_start (*m_snd_path_entry,    Gtk::PACK_EXPAND_WIDGET, 6);
	snd_path_box->pack_start (*m_snd_browse_button, Gtk::PACK_SHRINK,        2);

	snd_vbox->pack_start (*m_enable_alt_snd, Gtk::PACK_EXPAND_WIDGET, 4);
	snd_vbox->pack_start (*snd_path_box,     Gtk::PACK_SHRINK,        4);

	DI_SET_FRAME(snd_frame, snd_vbox);

	/********************
	 * Setup Appearance
	 ********************/
	bool app_enabled = m_deck.with_custom_appearance ();
	DI_SET_CHECKBUTTON(m_enable_custom_app, app_enabled);
	m_inner_app_vbox->set_sensitive (app_enabled);

	app_vbox->pack_start (*m_enable_custom_app, Gtk::PACK_EXPAND_WIDGET, 4);
	app_vbox->pack_start (*m_inner_app_vbox,    Gtk::PACK_SHRINK,        4);

	DI_SET_FRAME(app_frame, app_vbox);

	/********************
	 * Pack all up
	 ********************/
	all_vbox->pack_start (*info_table, Gtk::PACK_SHRINK, 4);
	all_vbox->pack_start (*snd_frame,  Gtk::PACK_SHRINK, 4);
	all_vbox->pack_start (*app_frame,  Gtk::PACK_SHRINK, 4);

	get_vbox ()->set_homogeneous (false);
	get_vbox ()->set_spacing (0);

	Gtk::ScrolledWindow* scrollw = Gtk::manage (new Gtk::ScrolledWindow);
	scrollw->set_flags (Gtk::CAN_FOCUS);
	scrollw->set_shadow_type (Gtk::SHADOW_NONE);
	scrollw->set_policy (Gtk::POLICY_AUTOMATIC , Gtk::POLICY_AUTOMATIC);
	scrollw->property_window_placement ().set_value (Gtk::CORNER_TOP_LEFT);

	scrollw->add (*all_vbox);
	get_vbox ()->pack_start (*scrollw);

	/************************
	 * Initialize callbacks 
	 ************************/
	add_action_widget (*cancel_button, -6);
	add_action_widget (*ok_button, -5);

	ok_button->signal_clicked ().connect (
		sigc::mem_fun (*this, &DeckInfo::on_ok_clicked));

	m_snd_browse_button->signal_clicked ().connect (
		sigc::mem_fun (*this, &DeckInfo::on_browse_clicked));

	m_enable_alt_snd->signal_toggled ().connect (
		sigc::mem_fun (*this, &DeckInfo::on_enable_snd_clicked));

	m_enable_custom_app->signal_toggled ().connect (
		sigc::mem_fun (*this, &DeckInfo::on_enable_app_clicked));

	show_all  ();
}

void
DeckInfo::
on_enable_snd_clicked ()
{
	trace_with_mask("DeckInfo::on_enable_snd_clicked", GUITRACE);

	m_snd_browse_button->set_sensitive (m_enable_alt_snd->get_active ());
}

void
DeckInfo::
on_enable_app_clicked ()
{
	trace_with_mask("DeckInfo::on_enable_app_clicked", GUITRACE);

	m_inner_app_vbox->set_sensitive (m_enable_custom_app->get_active ());
}

void
DeckInfo::
on_browse_clicked ()
{
	trace_with_mask("DeckInfo::on_browse_clicked", GUITRACE);

	MyFolderChooseDialog f (_("Select Sound Dictionary"), this);

	if (!m_snd_path_entry->get_text ().empty ()) {
		f.set_current_folder (m_snd_path_entry->get_text() + G_DIR_SEPARATOR_S);
	}

    if (f.run () == Gtk::RESPONSE_OK) {
        m_snd_path_entry->set_text (Glib::filename_from_utf8(f.get_filename()));
    }
}

void
DeckInfo::
on_ok_clicked ()
{
	trace_with_mask("DeckInfo::on_ok_clicked", GUITRACE);

	if (m_deck.get_author () != m_author_entry->get_text ()) {
		m_deck.set_author (m_author_entry->get_text ());
		m_deck.mark_as_modified ();
	}

	if (m_deck.get_desc () != m_desc_entry->get_text ()) {
		m_deck.set_desc (m_desc_entry->get_text ());
		m_deck.mark_as_modified ();
	}

	DL ((APP,"[%c] enable alt snd path\n", 
		 (m_enable_alt_snd->get_active () ? 'Y' : 'N')));

	if (m_enable_alt_snd->get_active () == false) {
		m_deck.set_snd_path ("");
		m_deck.mark_as_modified ();
	}
	else {
		if (m_deck.get_snd_path () != m_snd_path_entry->get_text ()) {
			m_deck.set_snd_path (m_snd_path_entry->get_text ());
			m_deck.mark_as_modified ();
		}
	}

	if (m_enable_custom_app->get_active () == false) {
		m_deck.enable_custom_appearance (false);
	}
	else {
		m_deck.enable_custom_appearance ();
		save_appearance ();
		m_deck.mark_as_modified ();
	}
	m_appearance_changed = true;
}

/**-----------------------------------------------------------------------------
 ** Algnment widgets - this needs rethinking and consolidation with
 ** TextAlignment/GrappConf implementation (which is cut-n-paste).
 **-----------------------------------------------------------------------------
 */
void
DeckInfo::
load_appearance ()
{
	m_question_font->set_font (m_deck.question_font ());

	set_x_alignment   (FRONT, m_deck.x_alignment   (FRONT));
	set_y_alignment   (FRONT, m_deck.y_alignment   (FRONT));
	set_x_padding     (FRONT, m_deck.x_padding     (FRONT));
	set_y_padding     (FRONT, m_deck.y_padding     (FRONT));
	set_justification (FRONT, m_deck.justification (FRONT));

	set_x_alignment   (BACK,  m_deck.x_alignment   (BACK));
	set_y_alignment   (BACK,  m_deck.y_alignment   (BACK));
	set_x_padding     (BACK,  m_deck.x_padding     (BACK));
	set_y_padding     (BACK,  m_deck.y_padding     (BACK));
	set_justification (BACK,  m_deck.justification (BACK));

	set_x_alignment   (EXAMPLE, m_deck.x_alignment (EXAMPLE));
	set_y_alignment   (EXAMPLE, m_deck.y_alignment (EXAMPLE));
	set_x_padding     (EXAMPLE, m_deck.x_padding   (EXAMPLE));
	set_y_padding     (EXAMPLE, m_deck.y_padding   (EXAMPLE));
	set_justification (EXAMPLE,m_deck.justification(EXAMPLE));
}

void
DeckInfo::
save_appearance ()
{
	m_deck.question_font (m_question_font->get_font ());

	m_deck.x_alignment   (FRONT, get_x_alignment   (FRONT));
	m_deck.y_alignment   (FRONT, get_y_alignment   (FRONT));
	m_deck.x_padding     (FRONT, get_x_padding     (FRONT));
	m_deck.y_padding     (FRONT, get_y_padding     (FRONT));
	m_deck.justification (FRONT, get_justification (FRONT));

	m_deck.x_alignment   (BACK,  get_x_alignment   (BACK));
	m_deck.y_alignment   (BACK,  get_y_alignment   (BACK));
	m_deck.x_padding     (BACK,  get_x_padding     (BACK));
	m_deck.y_padding     (BACK,  get_y_padding     (BACK));
	m_deck.justification (BACK,  get_justification (BACK));

	m_deck.x_alignment  (EXAMPLE, get_x_alignment (EXAMPLE));
	m_deck.y_alignment  (EXAMPLE, get_y_alignment (EXAMPLE));
	m_deck.x_padding    (EXAMPLE, get_x_padding   (EXAMPLE));
	m_deck.y_padding    (EXAMPLE, get_y_padding   (EXAMPLE));
	m_deck.justification(EXAMPLE, get_justification(EXAMPLE));
}

/** Set property value
 */
void
DeckInfo::
set_x_alignment (SideSelection side_, const std::string& v_)
{
	int value;

	if      (v_ == "center") { value = TA_X_ALIGN_CENTER; }
	else if (v_ == "left")   { value = TA_X_ALIGN_LEFT;   }
	else if (v_ == "right")  { value = TA_X_ALIGN_RIGHT;  }
	
	if (side_ == FRONT) {
		m_front_widget->x_align_combo ()->set_active (value);
	}
	else if (side_ == BACK) {
		m_back_widget->x_align_combo ()->set_active (value);
	}
	else { /* side_ == EXAMPLE */
		m_example_widget->x_align_combo ()->set_active (value);
	}
}

void
DeckInfo::
set_y_alignment (SideSelection side_, const std::string& v_)
{
	int value;

	if      (v_ == "center") { value = TA_Y_ALIGN_CENTER; }
	else if (v_ == "top")    { value = TA_Y_ALIGN_TOP;    }
	else if (v_ == "bottom") { value = TA_Y_ALIGN_BOTTOM; }

	if (side_ == FRONT) {
		m_front_widget->y_align_combo ()->set_active (value);
	}
	else if (side_ == BACK) {
		m_back_widget->y_align_combo ()->set_active (value);
	}
	else { /* side_ == EXAMPLE */
		m_example_widget->y_align_combo ()->set_active (value);
	}
}

void
DeckInfo::
set_x_padding (SideSelection side_, const std::string& v_)
{
	if (side_ == FRONT)     { 
		m_front_widget->x_padding_entry ()->set_text (v_); 
	}
	else if (side_ == BACK) { 
		m_back_widget->x_padding_entry ()->set_text (v_); 
	}
	else /* == EXAMPLE */	{ 
		m_example_widget->x_padding_entry ()->set_text (v_); 
	}
}

void
DeckInfo::
set_y_padding (SideSelection side_, const std::string& v_)
{
	if (side_ == FRONT)     { 
		m_front_widget->y_padding_entry ()->set_text (v_); 
	}
	else if (side_ == BACK) { 
		m_back_widget->y_padding_entry ()->set_text (v_); 
	}
	else /* == EXAMPLE */	{ 
		m_example_widget->y_padding_entry ()->set_text (v_); 
	}
}

void
DeckInfo::
set_justification (SideSelection side_, const std::string& v_)
{
	int value;

	if      (v_ == "center") { value = TA_JUSTIFY_CENTER; }
	else if (v_ == "left")   { value = TA_JUSTIFY_LEFT;   }
	else if (v_ == "right")  { value = TA_JUSTIFY_RIGHT;  }
	else if (v_ == "fill")   { value = TA_JUSTIFY_FILL;   }

	if (side_ == FRONT) {
		m_front_widget->paragraph_combo ()->set_active (value);
	}
	else if (side_ == BACK) {
		m_back_widget->paragraph_combo ()->set_active (value);
	}
	else { /* EXAMPLE */
		m_example_widget->paragraph_combo ()->set_active (value);
	}
}

/** Get property value
 */
std::string
DeckInfo::
get_x_alignment (SideSelection side_) const
{
	std::string result;
	int value;

	if (side_ == FRONT) {
		value = m_front_widget->x_align_combo ()->get_active_row_number ();
	}
	else if (side_ == BACK) {
		value = m_back_widget->x_align_combo ()->get_active_row_number ();
	}
	else { /* side_ == EXAMPLE */
		value = m_example_widget->x_align_combo ()->get_active_row_number ();
	}

	switch (value) {
	case TA_X_ALIGN_CENTER:	result = "center"; break;
	case TA_X_ALIGN_LEFT:	result = "left";   break;
	case TA_X_ALIGN_RIGHT:	result = "right";  break;
	default:                result = "center";
	}

	return (result);
}


std::string
DeckInfo::
get_y_alignment (SideSelection side_) const
{
	std::string result;
	int value;

	if (side_ == FRONT) {
		value = m_front_widget->y_align_combo ()->get_active_row_number ();
	}
	else if (side_ == BACK) {
		value = m_back_widget->y_align_combo ()->get_active_row_number ();
	}
	else { /* EXAMPLE */
		value = m_example_widget->y_align_combo ()->get_active_row_number ();
	}

	switch (value) {
	case TA_Y_ALIGN_CENTER:	result = "center";	break;
	case TA_Y_ALIGN_TOP:    result = "top";    	break;
	case TA_Y_ALIGN_BOTTOM: result = "bottom"; 	break;
	default:                result = "center";
	}

	return (result);
}

std::string
DeckInfo::
get_x_padding (SideSelection side_) const
{
	if (side_ == FRONT)   { 
		return m_front_widget->x_padding_entry()->get_text();
	}
	else if(side_ == BACK){ 
		return m_back_widget->x_padding_entry()->get_text();
	}
	else /* == EXAMPLE */ {	
		return m_example_widget->x_padding_entry()->get_text();
	}
}

std::string
DeckInfo::
get_y_padding (SideSelection side_) const
{
	if (side_ == FRONT)   { 
		return m_front_widget->y_padding_entry()->get_text();
	}
	else if(side_ == BACK){ 
		return m_back_widget->y_padding_entry()->get_text();
	}
	else /* == EXAMPLE */ {	
		return m_example_widget->y_padding_entry()->get_text();
	}
}


/** Get property value
 */
std::string
DeckInfo::
get_justification (SideSelection side_) const
{
	std::string result;
	int value;

	if (side_ == FRONT) {
		value = m_front_widget->paragraph_combo ()->get_active_row_number ();
	}
	else if (side_ == BACK) {
		value = m_back_widget->paragraph_combo ()->get_active_row_number ();
	}
	else { /* side_ == EXAMPLE */
		value = m_example_widget->paragraph_combo ()->get_active_row_number ();
	}

	switch (value) {
	case TA_JUSTIFY_CENTER:	result = "center";	break;
	case TA_JUSTIFY_LEFT:	result = "left";	break;
	case TA_JUSTIFY_RIGHT:	result = "right";	break;
	case TA_JUSTIFY_FILL:	result = "fill";	break;
	default:                result = "center";
	}

	return (result);
}

