// -*- c++ -*-
//------------------------------------------------------------------------------
//                              Deck.h
//------------------------------------------------------------------------------
// $Id: Deck.h,v 1.34 2006/12/29 03:12:13 vlg Exp $
//
//  Copyright (c) 2004 by Vladislav Grinchenko
//
//  This program is free software; you can redistribute it and/or 
//  modify it under the terms of the GNU General Public License   
//  as published by the Free Software Foundation; either version  
//  2 of the License, or (at your option) any later version.      
//
// Date: Jan 8, 2004
//------------------------------------------------------------------------------
#ifndef DECK_H
#define DECK_H

#include "VDeck.h"
#include <libxml/tree.h>		// xmlDocPtr

typedef enum _ParserState {
	ST_BEGIN,
	ST_1ST_FIELD,
	ST_2ND_FIELD,
	ST_3RD_FIELD
} ParserState;

class Card;

class Deck : public VDeck
{
public:
	Deck (DeckEditStatus status_);
	Deck (const string& fname_, DeckEditStatus status_);
	~Deck ();

	virtual int  load (const string& fname_, DeckAction = INSTALL_DECK);
	void         insert (const string& from_file_);
	bool         import_from_csv (const string& file_);
	bool         export_to_csv   (const string& file_);
	virtual int  save ();
	virtual int  save_as (const string& fname_);

	virtual cardlist_const_iterator begin () const { return m_cards.begin (); }
	virtual cardlist_const_iterator end   () const { return m_cards.end   (); }

	virtual cardlist_iterator begin () { return m_cards.begin (); }
	virtual cardlist_iterator end   () { return m_cards.end   (); }

	virtual size_t size  () const { return m_cards.size ();  }
	virtual bool   empty () const { return m_cards.empty (); }

	virtual void push_back (VCard* card_) { m_cards.push_back (card_); }
	virtual void erase (cardlist_iterator iter_) { m_cards.erase (iter_); }
	virtual bool erase (VCard* card_);

	void swap_with_next (VCard* vcard_);
	void swap_with_prev (VCard* vcard_);

	virtual const string&  get_name (void) const { return m_fname; }
	virtual void get_progress (float& pct_done_, std::string& msg_, int idx_);
	virtual void play_card (SideSelection s_, Deck::cardlist_iterator& iter_);

	VCard*               find_card_by_id (long id_);
	DeckEditStatus       get_status   ();

	const string&        get_snd_path () const { return m_snd_path; }
	const Glib::ustring& get_author   () const { return m_author; }
	const Glib::ustring& get_desc     () const { return m_desc; }

	void set_snd_path (const string& spath_);
	void set_author   (const Glib::ustring& s_) { m_author = s_; }
	void set_desc     (const Glib::ustring& s_) { m_desc = s_; }

	void test_active_snd_path ();

	void enable_custom_appearance (bool v_ = true) { m_custom_app_enabled=v_; }
	bool with_custom_appearance () const { return m_custom_app_enabled; }

	/** Text alignment controls
	 */
	std::string question_font () const;

	std::string x_alignment   (SideSelection s_) const;
	std::string y_alignment   (SideSelection s_) const;
	std::string x_padding     (SideSelection s_) const;
	std::string y_padding     (SideSelection s_) const;
	std::string justification (SideSelection s_) const;

	void question_font (const std::string& v_);

	void x_alignment   (SideSelection s_, const std::string& v_);
	void y_alignment   (SideSelection s_, const std::string& v_);
	void x_padding     (SideSelection s_, const std::string& v_);
	void y_padding     (SideSelection s_, const std::string& v_);
	void justification (SideSelection s_, const std::string& v_);

	/** Utility functions
	 */
	void mark_as_modified     ()       { m_status = MODIFIED_DECK; }
	bool needs_renaming       () const { return m_needs_renaming; }
	DeckEditStatus get_status () const { return m_status; }

	void dump () const;
	void dump_deck_status () const;

private:
	/** Parse the Deck file card by card.
		@return 0 on success; -1 on error with error_ explaining the reason.
	*/
	int  parse_xml_tree (xmlDocPtr parser_, std::string& error_);

	void reset_snd_setup ();

	int split_tokens (const Glib::ustring& instream_, 
					  std::vector<Glib::ustring>& tokens_, 
					  char separator_,
					  bool bracket_flag_);

	void trim_escaped_quotes (Glib::ustring& s_);
	void escape_double_quotes (Glib::ustring& input_);
	void add_new_card (Glib::ustring& q_, 
					   Glib::ustring& a_, 
					   Glib::ustring& e_,
					   long& card_id_,
					   int& ok_count_);

	void open_double_quote (bool& flag);
	void close_double_quote (bool& flag);
	const char* get_state_name (ParserState state_);
	void change_state (ParserState& state_);
	void reset_state (ParserState& state_);

private:
	void init_appearance ();

private:
	string         m_fname;		// full XML file path name
	cardlist_type  m_cards;
	string         m_indent;	// for pretty printing

	DeckEditStatus m_status;
	bool           m_needs_renaming;

	Glib::ustring  m_author;
	Glib::ustring  m_desc;
	SideSelection  m_side_selection; // Tells which side to play next time

	/** Active sound path is used to construct the name of the 
	 *  sound bite file for playing it through the sound card.
	 */
	string         m_active_snd_path; 

	/** True if m_active_snd_path holds a valid path
	 */
	bool           m_snd_enabled;

	/** The sound path associated with the Deck. This is an alternative
	 *  sound path if the user wished to overwrite the default from
	 *  application preferences.
	 *  It is accessible via DeckInfo dialog and is recorded in
	 *  XML .dkf file. When empty, it is assumed disabled.
	 */
	string         m_snd_path; 

	/** Indicate if XML file carries a custom appearance definition block.
	 */
	bool           m_custom_app_enabled;

	std::string    m_question_font;

	std::string    m_front_x_alignment;
	std::string    m_front_y_alignment; 
	std::string    m_front_x_padding;
	std::string    m_front_y_padding; 
	std::string    m_front_justification; 

	std::string    m_back_x_alignment;
	std::string    m_back_y_alignment; 
	std::string    m_back_x_padding;
	std::string    m_back_y_padding; 
	std::string    m_back_justification; 

	std::string    m_example_x_alignment;
	std::string    m_example_y_alignment; 
	std::string    m_example_x_padding;
	std::string    m_example_y_padding; 
	std::string    m_example_justification; 
};

inline 
Deck::~Deck ()
{
	trace_with_mask("Deck::~Deck",GUITRACE|DECK);
	dump_deck_status ();
}

inline void
Deck::open_double_quote (bool& flag)
{
	DL ((GRAPP,"double_bracket_flag: <true>\n"));
	flag = true;
}

inline void
Deck::close_double_quote (bool& flag)
{
	DL ((GRAPP,"double_bracket_flag: <false>\n"));
	flag = false;
}

inline const char*
Deck::get_state_name (ParserState state_)
{
	return (state_ == ST_BEGIN     ? "ST_BEGIN" :
			state_ == ST_1ST_FIELD ? "ST_1ST_FIELD" :
			state_ == ST_2ND_FIELD ? "ST_2ND_FIELD" : "ST_3RD_FIELD");
}

/**-------------------------
 **** Retrieve a property **
 **-------------------------
 */
inline std::string 
Deck::question_font () const
{
	return m_question_font;
}

inline std::string 
Deck::x_alignment (SideSelection s_) const 
{ 
	if      (s_ == FRONT) { return m_front_x_alignment;   }
	else if (s_ == BACK)  { return m_back_x_alignment;    }
	else    /* EXAMPLE */ { return m_example_x_alignment; }
}

inline std::string
Deck::y_alignment (SideSelection s_) const
{ 
	if      (s_ == FRONT) { return m_front_y_alignment;   }
	else if (s_ == BACK)  { return m_back_y_alignment;    }
	else    /* EXAMPLE */ { return m_example_y_alignment; }
}

inline std::string 
Deck::x_padding (SideSelection s_) const 
{ 
	if      (s_ == FRONT) { return m_front_x_padding;   }
	else if (s_ == BACK)  { return m_back_x_padding;    }
	else    /* EXAMPLE */ { return m_example_x_padding; }
}

inline std::string
Deck::y_padding (SideSelection s_) const
{ 
	if      (s_ == FRONT) { return m_front_y_padding;   }
	else if (s_ == BACK)  { return m_back_y_padding;    }
	else    /* EXAMPLE */ { return m_example_y_padding; }
}

inline std::string
Deck::justification (SideSelection s_) const 
{ 
	if      (s_ == FRONT) { return m_front_justification;   }
	else if (s_ == BACK)  { return m_back_justification;    }
	else    /* EXAMPLE */ { return m_example_justification; }
}

/**--------------------
 **** Set a property **
 **--------------------
 */
inline void
Deck::question_font (const std::string& v_)
{
	m_question_font = v_;
}

inline void
Deck::x_alignment (SideSelection s_, const std::string& v_)
{ 
	if      (s_ == FRONT) { m_front_x_alignment   = v_; }
	else if (s_ == BACK)  { m_back_x_alignment    = v_; }
	else    /* EXAMPLE */ { m_example_x_alignment = v_; }
}

inline void
Deck::y_alignment (SideSelection s_, const std::string& v_)
{ 
	if      (s_ == FRONT) { m_front_y_alignment   = v_; }
	else if (s_ == BACK)  { m_back_y_alignment    = v_; }
	else    /* EXAMPLE */ { m_example_y_alignment = v_; }
}

inline void
Deck::x_padding (SideSelection s_, const std::string& v_)
{ 
	if      (s_ == FRONT) { m_front_x_padding   = v_; }
	else if (s_ == BACK)  { m_back_x_padding    = v_; }
	else    /* EXAMPLE */ { m_example_x_padding = v_; }
}

inline void
Deck::y_padding (SideSelection s_, const std::string& v_)
{ 
	if      (s_ == FRONT) { m_front_y_padding   = v_; }
	else if (s_ == BACK)  { m_back_y_padding    = v_; }
	else    /* EXAMPLE */ { m_example_y_padding = v_; }
}

inline void
Deck::justification (SideSelection s_, const std::string& v_)
{ 
	if      (s_ == FRONT) { m_front_justification   = v_; }
	else if (s_ == BACK)  { m_back_justification    = v_; }
	else    /* EXAMPLE */ { m_example_justification = v_; }
}

#endif /* DECK_H */
