/* Copyright (C) 2004-2006 Damir Zucic */

/*=============================================================================

			    hyphob_function2.c

Purpose:
	Draw the hydrophobicity function F2.  The sequence stored to the
	main sequence buffer  is used  to calculate  the function value.

Input:
	(1) Pointer to the storage where the minimal function value will
	    be stored.
	(2) Pointer to the storage where the maximal function value will
	    be stored.
	(3) Pointer to RuntimeS structure.

Output:
	(1) Function F2 calculated and stored.
	(2) Return value.

Return value:
	(1) Positive on success.
	(2) Negative on failure.

Notes:
	(1) The function  F2 may be modified and used for many purposes.
	    Originally, it was introduced while searching for the method
	    which will be suitable for prediction of the porin secondary
	    structure.

========includes:============================================================*/

#include <stdio.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xos.h>
#include <X11/Xatom.h>

#include "defines.h"
#include "typedefs.h"

/*======calculate the hydrophobicity function F2:============================*/

int HyphobFunction2_ (double *min_functionP, double *max_functionP,
		      RuntimeS *runtimeSP)
{
int             residuesN, residueI;   /* Do not use size_t instead of int ! */
int		windowI, combinedI;
int		used_residuesN;
double		sided30, sided31, sided32, sided33,
		sided34, sided35, sided36, sided37;
double		function_value;
double		hydrophobicity;

static double	weight30A[37] =
		{ 0.0, 0.0, 0.0, 0.0,
		 -1.0, 0.0,-1.0, 0.0,-1.0, 0.0,
		  1.0, 0.0, 1.0, 0.0, 1.0, 0.0, 1.0, 0.0, 1.0,
		  1.0, 0.0, 1.0, 0.0, 1.0, 0.0, 1.0, 0.0, 1.0,
		  0.0,-1.0, 0.0,-1.0, 0.0,-1.0,
		  0.0, 0.0, 0.0};

static double	weight31A[37] =
		{ 0.0, 0.0, 0.0,
		 -1.0, 0.0,-1.0, 0.0,-1.0, 0.0,
		  1.0, 0.0, 1.0, 0.0, 1.0, 0.0, 1.0, 0.0, 1.0,
		  0.0,
		  1.0, 0.0, 1.0, 0.0, 1.0, 0.0, 1.0, 0.0, 1.0,
		  0.0,-1.0, 0.0,-1.0, 0.0,-1.0,
		  0.0, 0.0, 0.0};

static double	weight32A[37] =
		{ 0.0, 0.0, 0.0,
		 -1.0, 0.0,-1.0, 0.0,-1.0, 0.0,
		  1.0, 0.0, 1.0, 0.0, 1.0, 0.0, 1.0, 0.0, 1.0,
		  0.0, 0.0,
		  1.0, 0.0, 1.0, 0.0, 1.0, 0.0, 1.0, 0.0, 1.0,
		  0.0,-1.0, 0.0,-1.0, 0.0,-1.0,
		  0.0, 0.0};

static double	weight33A[37] =
		{ 0.0, 0.0,
		 -1.0, 0.0,-1.0, 0.0,-1.0, 0.0,
		  1.0, 0.0, 1.0, 0.0, 1.0, 0.0, 1.0, 0.0, 1.0,
		  0.0, 0.0, 0.0,
		  1.0, 0.0, 1.0, 0.0, 1.0, 0.0, 1.0, 0.0, 1.0,
		  0.0,-1.0, 0.0,-1.0, 0.0,-1.0,
		  0.0, 0.0};

static double	weight34A[37] =
		{ 0.0, 0.0,
		 -1.0, 0.0,-1.0, 0.0,-1.0, 0.0,
		  1.0, 0.0, 1.0, 0.0, 1.0, 0.0, 1.0, 0.0, 1.0,
		  0.0, 0.0, 0.0, 0.0,
		  1.0, 0.0, 1.0, 0.0, 1.0, 0.0, 1.0, 0.0, 1.0,
		  0.0,-1.0, 0.0,-1.0, 0.0,-1.0,
		  0.0};

static double	weight35A[37] =
		{ 0.0,
		 -1.0, 0.0,-1.0, 0.0,-1.0, 0.0,
		  1.0, 0.0, 1.0, 0.0, 1.0, 0.0, 1.0, 0.0, 1.0,
		  0.0, 0.0, 0.0, 0.0, 0.0,
		  1.0, 0.0, 1.0, 0.0, 1.0, 0.0, 1.0, 0.0, 1.0,
		  0.0,-1.0, 0.0,-1.0, 0.0,-1.0,
		  0.0};

static double	weight36A[37] =
		{ 0.0,
		 -1.0, 0.0,-1.0, 0.0,-1.0, 0.0,
		  1.0, 0.0, 1.0, 0.0, 1.0, 0.0, 1.0, 0.0, 1.0,
		  0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
		  1.0, 0.0, 1.0, 0.0, 1.0, 0.0, 1.0, 0.0, 1.0,
		  0.0,-1.0, 0.0,-1.0, 0.0,-1.0};

static double	weight37A[37] =
		{-1.0, 0.0,-1.0, 0.0,-1.0, 0.0,
		  1.0, 0.0, 1.0, 0.0, 1.0, 0.0, 1.0, 0.0, 1.0,
		  0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
		  1.0, 0.0, 1.0, 0.0, 1.0, 0.0, 1.0, 0.0, 1.0,
		  0.0,-1.0, 0.0,-1.0, 0.0,-1.0};

/*------prepare some parameters:---------------------------------------------*/

/* The number of residues in sequence buffer: */
residuesN = (int) runtimeSP->residuesN;
if (residuesN == 0) return -1;

/*------calculate the function F2:-------------------------------------------*/

/* Initialize the extreme values: */
*min_functionP = +999999.0;
*max_functionP = -999999.0;

/* 20040909.1304: */
/* A window of 37 residues is scanned. The sided hydrophobicity is */
/* calculated for a number of subwindows,  counting from  30 to 37 */
/* residues.  The largest  sided hydrophobicity value  is taken as */
/* the final function value and stored to  the appropriate buffer. */

/* Initialize (reset) the function value: */
for (residueI = 0; residueI < residuesN; residueI++)
	{
	/* Reset the function value, it might be initialized before: */
	*(runtimeSP->function2P + residueI) = 0.0;
	}

/* Scan the sequence, ignoring 18 residues at each end: */
for (residueI = 18; residueI < residuesN - 18; residueI++)
	{
	/* Reset the number of residues used to prepare the raw values: */
	used_residuesN = 0;

	/* Reset the function value, it might be initialized before: */
	*(runtimeSP->function2P + residueI) = 0.0;

	/* Reset the sided hydrophobicities for all subwindows: */
	sided30 = 0.0;
	sided31 = 0.0;
	sided32 = 0.0;
	sided33 = 0.0;
	sided34 = 0.0;
	sided35 = 0.0;
	sided36 = 0.0;
	sided37 = 0.0;

	/* Scan the sliding window: */
	for (windowI = 0; windowI < 37; windowI++)
		{
		/* Prepare and check the combined index: */
		combinedI = residueI + windowI - 18;
		if (combinedI < 0) continue;
		if (combinedI >= residuesN) continue;

		/* The hydrophobicity of the residue defined by combinedI: */
		hydrophobicity =
			(double) *(runtimeSP->hydrophobicityP + combinedI);

		/* Update the sided hydrophobicity values: */
		sided30 += weight30A[windowI] * hydrophobicity;
		sided31 += weight31A[windowI] * hydrophobicity;
		sided32 += weight32A[windowI] * hydrophobicity;
		sided33 += weight33A[windowI] * hydrophobicity;
		sided34 += weight34A[windowI] * hydrophobicity;
		sided35 += weight35A[windowI] * hydrophobicity;
		sided36 += weight36A[windowI] * hydrophobicity;
		sided37 += weight37A[windowI] * hydrophobicity;

		/* Update the number of used residues: */
		used_residuesN++;
		}

	/* Check the number of used residues: */
	if (used_residuesN == 0) continue;

	/* The function value  should be  equal */
	/* to the highest sided hydrophobicity: */
	function_value = sided30;
	if (sided31 > function_value) function_value = sided31;
	if (sided32 > function_value) function_value = sided32;
	if (sided33 > function_value) function_value = sided33;
	if (sided34 > function_value) function_value = sided34;
	if (sided35 > function_value) function_value = sided35;
	if (sided36 > function_value) function_value = sided36;
	if (sided37 > function_value) function_value = sided37;

	/* Scale the function value (it should fit */
	/* to the same plot with other functions): */
	function_value /= 16.0;

	/* Store the function value: */
	*(runtimeSP->function2P + residueI) = function_value;

	/* Dummy values should not be used to determine extreme values: */
	if (used_residuesN < 37) continue;

	/* Find the extreme values: */
	if (function_value < *min_functionP) *min_functionP = function_value;
	if (function_value > *max_functionP) *max_functionP = function_value;

	/* End of residueI loop: */
	}

/*---------------------------------------------------------------------------*/

return 1;
}

/*===========================================================================*/


